/* dgraph -- general purpose graph drawing */

/* Mike Brookes - Imperial College London */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "SFSCONFG.h"
#include "dig.h"
#include "digdata.h"

/* Define variables in parameter block	*/

/* header file for dgraph external variables	*/

#define DGRAPH	/* flag for include file	*/
#include "dgraph.h"

#define sxmin 	dgraphp->xmin
#define sxmax 	dgraphp->xmax
#define sxmode  dgraphp->savexmode
#define symin 	dgraphp->ymin
#define symax 	dgraphp->ymax
#define symode  dgraphp->saveymode
#define xscal	dgraphp->xscale
#define yscal	dgraphp->yscale
#define xoff	dgraphp->xoffset
#define yoff	dgraphp->yoffset
#define xleft	dgraphp->left
#define xright	dgraphp->right
#define ybot	dgraphp->bot
#define ytop	dgraphp->top

#define DTMASK 3
#define DLMASK 28
#define APMASK 96	/* axis position */
#define ALMASK 1536	/* axis label */
#define SCMASK 2048	/* scale markings */
#define UNMASK 4096	/* unit specified */
#define PRMASK (DGlogax | DGzero | DGnoscale)
#define PRMASK2 (DGnoaxis | DGlowax | DGhighax | DGlowlab | DGhighlab)

#define GRTYPE 15

#define DEFCSIZE 0.018		/* target character size for graph labels */
#define AXTHICK 3.96		/* axis thickness */

static union { short *s;
	         int *i;
	       float *f;
 	      double *d; } px,py;


static double xfloat();
static double xdoub();
static double xint();
static double xshort();
static double yfloat();
static double ydoub();
static double yint();
static double yshort();

/* prototypes */
#ifdef __STDC__
static void minmax(double (*nextx)(),int n,int xmode,double *pmin,double *pmax,double *pinc);
#else
static void minmax();
#endif

/* type coercion */
static double xfloat()
{
    return ((double)(*(px.f++)));
    }
static double xdoub()
{
    return ((double)(*(px.d++)));
    }
static double xint()
{
    return ((double)(*(px.i++)));
    }
static double xshort()
{
    return ((double)(*(px.s++)));
    }
static double yfloat()
{
    return ((double)(*(py.f++)));
    }
static double ydoub()
{
    return ((double)(*(py.d++)));
    }
static double yint()
{
    return ((double)(*(py.i++)));
    }
static double yshort()
{
    return ((double)(*(py.s++)));
    }

/*******************************************************
	Subroutine starts here
********************************************************/

void	dgraph(x,y,n,xmode,ymode,type,bun,ptitle,xtitle,ytitle)
short *x,*y;
int n;
int xmode,ymode,type,bun;
char *xtitle,*ptitle,*ytitle;
{
    DIGSPACE space[2];

    int xdlmode,ydlmode;
    int i;
    int buns,buna,bung,bunt;
    int grtype;
    int repeating,repeatx=0;
    int xapmode,xalmode;
    int yapmode,yalmode;
    int xpixels,ypixels;
    int axmode;
    int savexmin,savexmax,saveymin,saveymax;
    int ix,iy,oldix=0,oldiy=0;
    int yzero,imin=0,imax=0;
    int xlog,ylog;

    char *label,*ulabel,label0[150];

    double xint(),xshort(),xfloat(),xdoub();
    double yint(),yshort(),yfloat(),ydoub();
    double (*nextx) () = xfloat;
    double (*nexty) () = yfloat;
    double xscalp,yscalp,xoffp,yoffp;
    double xmin,xmax,xinc,ymin,ymax,yinc;
    double oldx,oldy,newx,newy;
    double xheight,yheight;
    double uaspect;			/* height/width for unit square */
    double xax=0,yax=0;
    double xnotax=0,ynotax=0;
    double axthick;
    double sxsum,sysum,sxdif,sydif,aspect;

/* extract the graph type	*/

    grtype = type&GRTYPE;

/* Now sort out data types	*/

    switch (xmode & DTMASK) {
	case DGfloat:	nextx = xfloat; break;
	case DGdouble:	nextx = xdoub;	 break;
	case DGint:	nextx = xint;	 break;
	case DGshort:	nextx = xshort; break;
	}
    switch (ymode & DTMASK) {
	case DGfloat:	nexty = yfloat; break;
	case DGdouble:	nexty = ydoub;	 break;
	case DGint:	nexty = yint;	 break;
	case DGshort:	nexty = yshort; break;
	}

    if (xmode & (DGprev + DGsame)) {
	xmode = (xmode & ~PRMASK) | (sxmode & PRMASK);
	ymode = (ymode & ~PRMASK2) | (symode & PRMASK2);
	}
    if (ymode & (DGprev + DGsame)) {
	ymode = (ymode & ~PRMASK) | (symode & PRMASK);
	xmode = (xmode & ~PRMASK2) | (sxmode & PRMASK2);
	}
    sxmode = xmode;
    symode = ymode;
    xdlmode = xmode & DLMASK;
    xapmode = xmode & APMASK;
    xlog = xmode & DGlogax;
    xalmode = xmode & ALMASK;
    ydlmode = ymode & DLMASK;
    yapmode = ymode & APMASK;
    ylog = ymode & DGlogax;
    yalmode = ymode & ALMASK;

/* Now work out the buns	*/

    bung = bun % 100;
    buns = (bun /100) % 100;
    buna = (bun /10000) % 100;
    bunt = (bun /1000000) % 100;
    if (!buns) buns=bung;
    if (!buna) buna=buns;
    if (!bunt) bunt=buna;

/* Find min and max values of x and y */

    px.s=x;
    minmax(nextx,n,xmode,&xmin,&xmax,&xinc);
    if (!(xmode & (DGprev + DGsame))) {
	sxmax = xmax;
	sxmin = xmin;
	}

/* Ignore last y value for histograms	*/

    py.s=y;
    if ((grtype>=3) && (grtype<=5)) n-=1;
    minmax(nexty,n,ymode,&ymin,&ymax,&yinc);
    if (!(ymode & (DGprev + DGsame))) {
	symax = ymax;
	symin = ymin;
	}

/* now calculate a reasonable character size */

    
    xpixels = (int)(DEFCSIZE*digdata.xscale);
    ypixels = (int)(xpixels*digdata.aspect);
    if (ypixels > DEFCSIZE*digdata.yscale) ypixels = (int)(DEFCSIZE*digdata.yscale);
    if (ypixels<digdata.cheightmin) ypixels = digdata.cheightmin;
    xpixels = (int)(ypixels/digdata.aspect);
    if (xpixels<digdata.cheightmin) xpixels = digdata.cheightmin;
    xheight = xpixels/digdata.xscale;			/* for y-axis */
    uaspect = digdata.aspect * digdata.yscale / digdata.xscale;
    yheight = xheight / uaspect;			/* for x-axis */

/* Now calculate graph coordinates */

    axthick = yheight*AXTHICK;
    if (!xalmode) {			/* default label position */
	switch (xapmode) {
	    case DGhighax: xalmode = DGhighlab; break;
	    case DGlowax: xalmode = DGlowlab;  break;
	    case  0: xalmode = (symax < -symin) ? DGhighlab : DGlowlab;
	    }
	}
    ybot = yheight;
    ytop = 1.0 - yheight;
    switch (xapmode) {
	case DGhighax: yax = ytop -= axthick; ynotax = ybot; break;
	case DGlowax: yax = ybot += axthick; ynotax = ytop; break;
	case  0: 
	    yax = ybot - (ytop-ybot)*symin/(symax-symin);
	    if (xalmode == DGhighlab) {
		if (symax <= 0.0) 
		    yax = ytop -= axthick;
		else if (symin >= 0.0)
		    yax = ybot;
		else if ((yax+axthick) > ytop) {
		    yax = ytop - axthick;
		    ytop = yax -(yax-ybot)*symax/symin;
		    }
		}
	    else {
		if (symin >= 0.0) yax = ybot += axthick;
		else if (symax <= 0.0) yax = ytop;
		else if ((yax-axthick) < ybot) {
		    yax = ybot + axthick;
		    ybot = yax + (ytop-yax)*symin/symax;
		    }
		}
	    break;
	}

    axthick = xheight*AXTHICK;
    if (!yalmode) {			/* default label position */
	switch (yapmode) {
	    case DGhighax: yalmode = DGlowlab; break;
	    case DGlowax: yalmode = DGhighlab;  break;
	    case  0: yalmode = (sxmax < -sxmin) ? DGlowlab : DGhighlab;
	    }
	}
    xleft = xheight;
    xright = 1.0 - xheight;
    switch (yapmode) {
	case DGhighax: xax = xright -= axthick; xnotax = xleft; break;
	case DGlowax: xax = xleft += axthick; xnotax = xright; break;
	case  0: 
	    xax = xleft - (xright-xleft)*sxmin/(sxmax-sxmin);
	    if (yalmode == DGlowlab) {
		if (sxmax <= 0.0) 
		    xax = xright -= axthick;
		else if (sxmin >= 0.0)
		    xax = xleft;
		else if ((xax+axthick) > xright) {
		    xax = xright - axthick;
		    xright = xax -(xax-xleft)*sxmax/sxmin;
		    }
		}
	    else {
		if (sxmin >= 0.0) xax = xleft += axthick;
		else if (sxmax <= 0.0) xax = xright;
		else if ((xax-axthick) < xleft) {
		    xax = xleft + axthick;
		    xleft = xax + (xright-xax)*sxmin/sxmax;
		    }
		}
	    break;
	}
		    
/* we have now calculated xleft,xax,xright,ybot,yax,ytop
   which indicate the axis end coordinates */



/* Calculate graph scaling factors for both real & pixel units	*/

    sxdif = sxmax-sxmin;
    sydif = symax-symin;
    if (type & DGequal) {	/* force equal scales */
	aspect = digdata.aspect*(ytop-ybot)*digdata.yscale*sxdif/
		((xright-xleft)*digdata.xscale*sydif);
	if (aspect > 1.0) {	/* expand y axis */
	    sysum = symax+symin;
	    sydif *= aspect;
	    symax = 0.5*(sysum+sydif);
	    symin = 0.5*(sysum-sydif);
	    }
	else {
	    sxsum = sxmax + sxmin;
	    sxdif /= aspect;
	    sxmax = 0.5*(sxsum+sxdif);
	    sxmin = 0.5*(sxsum-sxdif);
	    }
	}
    xscal = (xright-xleft)/sxdif;
    xoff = xleft - sxmin*xscal ;
    yscal = (ytop-ybot)/sydif;
    yoff = ybot - symin*yscal ;

    xscalp = xscal*digdata.xscale;
    xoffp = xoff*digdata.xscale + digdata.xorig;
    yscalp = yscal*digdata.yscale;
    yoffp = yoff*digdata.yscale + digdata.yorig;

/* Draw axes	*/

buna = bunt + 100*buns + 10000*buna;
if(!(xapmode == DGnoaxis) && !(xmode & DGsame)) {
    if(xlog) {
	xmin = exp(sxmin);
	xmax = exp(sxmax);
	}
    else {
	xmin = sxmin;
	xmax = sxmax;
	}
    label = label0;
    strcpy(label,(xtitle)?xtitle:"");
    if (xmode & UNMASK) {
	ulabel = label + strlen(label);
	while ((ulabel>label) && (ulabel[-1] != ' ')) ulabel--;
	if (ulabel == label) label += strlen(label);
	else ulabel[-1] = '\0';
	}
    else ulabel = NULL;
    if (((xalmode == DGlowlab) && (yax > ybot + 0.5*yheight)) ||
	((xalmode == DGhighlab) && (yax < ytop - 0.5*yheight))) {
	    space[0].next = space+1;
	    space[1].right = 1.0 - xleft;
	    if (yalmode == DGlowlab) {
		space[0].right = xax -xleft - xheight*0.5;
		space[1].left = xax -xleft + xheight*(AXTHICK+0.5);
		}
	    else {
		space[0].right = xax -xleft - xheight*(AXTHICK+0.5);
		space[1].left = xax -xleft + xheight*0.5;
		}
	    }
    else {
	space[0].right = 1.0 - xleft;
	space[0].next = NULL;
	}
    space[0].left = -xleft;
    space[1].next = NULL;
    axmode = ((xlog) != 0) + 2*((xmode & DTMASK)/2) +
	     ((xmode & DGnoscale) ? 128 : 168)*(xalmode/DGlowlab);
    if (xmode & DGgrid) axmode += DAgridx;
    digaxis(buna,(float)xmin,(float)xmax,label,ulabel,axmode,
	    (float)xleft,(float)yax,(float)xright,(float)((xmode&DGgrid)?ynotax:yax),(float)yheight,(float)xheight,space);
    }

if(!(yapmode == DGnoaxis) && !(ymode & DGsame)) {
    if(ylog) {
	ymin = exp(symin);
	ymax = exp(symax);
	}
    else {
	ymin = symin;
	ymax = symax;
	}
    label = label0;
    strcpy(label,(ytitle)?ytitle:"");
    if (ymode & UNMASK) {
	ulabel = label + strlen(label);
	while ((ulabel>label) && (ulabel[-1] != ' ')) ulabel--;
	if (ulabel == label) label += strlen(label);
	else ulabel[-1] = '\0';
	}
    else ulabel = NULL;
    if (((yalmode == DGhighlab) && (xax > xleft + 0.5*xheight)) ||
	((yalmode == DGlowlab) && (xax < xright - 0.5*xheight))) {
	    space[0].next = space+1;
	    space[1].right = 1.0 - ybot;
	    if (xalmode == DGhighlab) {
		space[0].right = yax -ybot - yheight*0.5;
		space[1].left = yax -ybot + yheight*(AXTHICK+0.5);
		}
	    else {
		space[0].right = yax -ybot - yheight*(AXTHICK+0.5);
		space[1].left = yax -ybot + yheight*0.5;
		}
	    }
    else {
	space[0].right = 1.0 - ybot;
	space[0].next = NULL;
	}
    space[0].left = -ybot;
    space[1].next = NULL;
    axmode = (ylog != 0) + 2*((ymode & DTMASK)/2) +
	     ((ymode & DGnoscale) ? 128 : 168)*(yalmode/DGlowlab);
    if (ymode & DGgrid) axmode += DAgridy;
    digaxis(buna,(float)ymin,(float)ymax,label,ulabel,axmode,
	    (float)xax,(float)ybot,(float)((ymode&DGgrid)?xnotax:xax),(float)ytop,(float)xheight,(float)yheight,space);
    }


/* Now draw the graph	*/

if (grtype != DGaxonly) {

    savexmin = digdata.clipxmin;	/* save old clipping area */
    savexmax = digdata.clipxmax;
    saveymin = digdata.clipymin;
    saveymax = digdata.clipymax;
    digclip((float)xleft,(float)ybot,(float)xright,(float)ytop);

    yheight = digdata.cheightmin/digdata.yscale;	/* height for plotted labels */
    repeating = 0;		/* used to prevent lots of zero length plots */
    px.s=x; py.s=y;
    oldx = (*nextx) ();
    oldy = (*nexty) ();
    newx = (xlog) ? log(oldx) : oldx;
    newy = ylog ? log(oldy) : oldy;
    ix= (int)(newx*xscalp + xoffp);
    iy = (int)(newy*yscalp + yoffp);
    if (ptitle) digboxtext(bung,(float)(newx*xscal+xoff),
		    (float)(newy*yscal+yoff-0.5*yheight),(float)1.0,(float)yheight,(float)0.0,ptitle,1);
    yzero= (int)yoffp;
    switch (grtype) {
	case DGhistend:
	case DGhistdiv:
	    diglinep(bung,ix,yzero,ix,iy);
	    break;
	case DGbar: diglinep(bung,ix,yzero,ix,iy);
		    break;
	}
    

/* Main plotting loop */

    for (i=n-1; i>0; i--) {
	oldix=ix; oldiy=iy;
	switch (xdlmode) {
	    case DGlast:
	    case DGnext: newx += xinc; break;
	    case DGinc:
	    case DGfirst: newx = (xlog) ? log(oldx += xinc) : newx+xinc;
			  break;
	    case DGmult: newx = (xlog) ? newx+xinc : newx*xinc; break;
	    default: newx = (xlog) ? log((*nextx) ()) : (*nextx) ();
	    }
	switch (ydlmode) {
	    case DGlast:
	    case DGnext: newy += yinc; break;
	    case DGinc:
	    case DGfirst: newy = ylog ? log(oldy += yinc) : newy+yinc;
			  break;
	    case DGmult: newy = ylog ? newy+yinc : newy*yinc; break;
	    default: newy = ylog ? log((*nexty) ()) : (*nexty) ();
	    }

	ix= (int)(newx*xscalp + xoffp);
	iy = (int)(newy*yscalp + yoffp);
	if (ptitle) digboxtext(bung,(float)(newx*xscal+xoff),
			(float)(newy*yscal+yoff-0.5*yheight),(float)1.0,(float)yheight,(float)0.0,ptitle,1);
	switch (grtype) {
	    case DGbar:
		diglinep(bung,ix,yzero,ix,iy); break;
	    case DGhistdiv:
		diglinep(bung,oldix,oldiy,ix,oldiy);
		if (iy<oldiy) { imin = iy; imax = oldiy; }
		else          { imax = iy; imin = oldiy; }
		if (yzero<imin) imin = yzero;
		else if (yzero>imax) imax = yzero;
		diglinep(bung,ix,imin,ix,imax);
		break;
	    case DGline:
	    case DGhist:
	    case DGhistend:
		if (repeating) {
		    if (repeatx) {
			if (ix == oldix) {
			    if (iy<imin) imin = iy;
			    else if (iy>imax) imax = iy;
			    }
			else {
			    diglinep(bung,oldix,imin,oldix,imax);
			    repeating = 0;
			    }
			}
		    else {
			if (iy == oldiy) {
			    if (ix<imin) imin = ix;
			    else if (ix>imax) imax = ix;
			    }
			else {
			    diglinep(bung,imin,oldiy,imax,oldiy);
			    repeating = 0;
			    }
			}
		    }
		if (!repeating) {
		    if (ix == oldix) {
			repeating= repeatx = 1;
			if (iy<oldiy) {
			    imin = iy;
			    imax = oldiy;
			    }
			else {
			    imin = oldiy;
			    imax = iy;
			    }
			}
		    else if (iy == oldiy) {
			repeating = 1;
			repeatx = 0;
			if (ix<oldix) {
			    imin = ix;
			    imax = oldix;
			    }
			else {
			    imin = oldix;
			    imax = ix;
			    }
			}
		    else {
			if ((grtype == 3)||(grtype==4)) {
			    diglinep(bung,oldix,oldiy,ix,oldiy);
			    diglinep(bung,ix,oldiy,ix,iy);
			    }
			else diglinep(bung,oldix,oldiy,ix,iy);
			}
		    }
		break;
	    }
	}
	if (repeating) {
	    if(repeatx) diglinep(bung,oldix,imin,oldix,imax);
	    else diglinep(bung,imin,oldiy,imax,oldiy);
	    }
	if ((grtype>=3)&&(grtype<=5)) {
	    oldix=ix;
	    if (xdlmode) newx += xinc;
	    else newx = (*nextx) ();
	    ix= (int)(newx*xscalp + xoffp);
	    diglinep(bung,oldix,iy,ix,iy);
	    if (grtype>=4) diglinep(bung,ix,iy,ix,yzero);
	    }
	digdata.clipxmin = savexmin ; 
	digdata.clipxmax = savexmax ; 
	digdata.clipymin = saveymin ; 
	digdata.clipymax = saveymax ; 
	}
    
}


static void minmax(nextx,n,xmode,pmin,pmax,pinc)
int n,xmode;
double *pmin,*pmax,*pinc,(*nextx)();
/*
 find the min and max of the array pointed to by px	
 if log axis then take logs of them.

 *pinc is as follows (depending on mode and logax):

     Mode     log axis         lin axis

     Normal      x                x
     DGlast    log(*)             +
     DGnext    log(*)             +
     DGinc       +                +
     DGfirst     +                +
     DGmult    log(*)             *

*/
{
    int dlmode,loginc,logax,i;
    double lastx=0,val2;

    dlmode = xmode & DLMASK;
    loginc = logax = (xmode &  DGlogax); 

     *pmin =  *pmax = (*nextx)();
    if (dlmode) {		
	if (dlmode == DGfirst)  {	/* only 1 value supplied */
	     lastx = *pmin + (n-1);
	     *pinc = 1.0;
	    }
	else {				/* 2 values supplied */
	    val2 = (*nextx)();
	    if (loginc) {
		switch (dlmode) {
		    case DGnext:  *pinc = log(val2/ *pmin)/n;
				 lastx = val2/exp( *pinc);
				 break;
		    case DGlast:  *pinc = log(val2/ *pmin)/(n-1);
				 lastx = val2;
				 break;
		    case DGmult:  *pinc = log(val2);
				 lastx =  *pmin * exp( *pinc * (n-1));
				 break;
		    default: loginc = 0;		/* else treat as lin */
		    }
		}
	    if (!loginc) {
		switch(dlmode) {
		    case DGlast:  *pinc = (val2- *pmin)/(n-1);
				 lastx = val2;
				 break;
		    case DGnext:  *pinc = (val2- *pmin)/n;
				 lastx = val2 -  *pinc;
				 break;
		    case DGinc:   *pinc = val2;
				 lastx =  *pmin + (n-1)*val2;
				 break;
		    case DGmult:  *pinc = val2;
				 lastx =  *pmin * pow(val2,(double) (n-1));
				 break;
		    }
		}
	    }
        if (lastx> *pmin)  *pmax=lastx;
        else if (lastx< *pmax)  *pmin=lastx;
	}
    else {
	for (i=n-1; i>0; i--) {
	    lastx = (*nextx)();
	    if (lastx >  *pmax)  *pmax=lastx;
	    else if (lastx <  *pmin)  *pmin=lastx; 
	    }
	}
    if (logax) {
	 *pmin = log( *pmin);
	 *pmax = log( *pmax);
	}
    if ( *pmin ==  *pmax)
	if ( *pmin<0.0)  *pmax=0.0;
	else if ( *pmin>0.0)  *pmin=0.0;
	else  *pmax=1.0;
    if (xmode &  DGzero)
	if ( *pmin > 0.0)  *pmin = 0.0;
	else if ( *pmax < 0.0)  *pmax = 0.0;
    return;
    }


int mappoint(double xpos,double ypos,double *xval,double *yval)
{
	if ((xpos < dgraphp->left) || (xpos > dgraphp->right) ||
	    (ypos < dgraphp->bot)  || (ypos > dgraphp->top)) {
	    	*xval = *yval = 0;
		return(0);
	}
	else {
		*xval = dgraphp->xoffset + (xpos - dgraphp->left)/dgraphp->xscale;
		*yval = dgraphp->yoffset + (ypos - dgraphp->bot)/dgraphp->yscale;
		return(1);
	}
}

#ifdef EMO

#define NDG 4		/* number of general descriptors */
#define NDXY 8		/* number of axis descriptors */
#define NDESC (NDG+2*NDXY)	/* total number */
#define MAXN 10

static char *desc[NDESC] = {
    "Number of points",
    "Type: 0=line,1=axes,2=points,3=hist,4=histend,5=histdiv,6=bar",
    "0=normal,1=scales equal",
    "0=normal,1=points labelled",
    "0=all,1=end,2=end+,3=inc,4=first,5=mult",
    "0=normal,1=no axis,2=axis at bottom,3=axis at top",
    "0=normal,1=log axis",
    "0=normal,1=include 0.0",
    "0=normal,1=label below,2=label above",
    "0=normal,1=no scale",
    "0=no unit,1=unit",
    "0=normal,1=use previous,2=same graph",
    "0=all,1=end,2=end+,3=inc,4=first,5=mult",
    "0=normal,1=no axis,2=axis at left,3=axis at right",
    "0=normal,1=log axis",
    "0=normal,1=include 0.0",
    "0=normal,1=label below,2=label above",
    "0=normal,1=no scale",
    "0=no unit,1=unit",
    "0=normal,1=use previous,2=same graph"
    };

static int imult[NDXY] = {4,32,128,256,512,2048,4096,8192};

main ()
{
    float x[MAXN],y[MAXN];

    static int ival[NDESC];
    int i,j,prompted;
    int nx,ny;
    int n,type,xmode,ymode;
    char c;
    static char *ptitle,ptit[] = "A";

    ival[0] = nx = ny = 3;
    x[0] = 1.0;
    x[1] = 2.0;
    x[2] = 3.0;
    y[0] = 1.0;
    y[1] = 4.0;
    y[2] = 9.0;
    for (;;) {
	printf("Test for dgraph\n\n");
	for (i=0; i<NDESC; i++) {
	    if (i == NDG) printf("X-axis: ");
	    else if (i == NDG+NDXY) printf("Y-axis: ");
	    else printf("        ");
	    printf("%c%3d %s\n",'a'+i,ival[i],desc[i]);
	    }

	printf("x:");
	for (i=0; i<nx; i++) printf(" %g",x[i]);
	printf("\n");

	printf("y:");
	for (i=0; i<ny; i++) printf(" %g",y[i]);
	printf("\n");

	printf("Q : quit    R : run    V : verify\n");
	c = '\n';
	prompted = 0;
	for (;;) {
	    if (ival[0] > MAXN) ival[0] = MAXN;
	    if (ival[NDG] == 0) nx = ival[0];
	    else if (ival[NDG] == 4) nx = 1;
	    else nx = 2;
	    if (ival[NDG+NDXY] == 0) ny = ival[0];
	    else if (ival[NDG+NDXY] == 4) ny = 1;
	    else ny = 2;
	    if((c == '\n') && (!prompted)) {
		printf("Enter command: ");
		prompted = 1;
		}
	    scanf("%c",&c);
	    if (c == 'x') {
		for (i=0; i<nx; i++) scanf("%f",&x[i]);
		}
	    if (c == 'y') {
		for (i=0; i<ny; i++) scanf("%f",&y[i]);
		}
	    if ((c>='a') && (c< 'a'+NDESC)) {
		scanf("%d",&ival[c - 'a']);
		prompted = 0;
		}
	    if (c == 'Q') {
		exit(0);
		}
	    if (c == 'V') break;
	    if (c == 'R') {
		n = ival[0];
		type = ival[1] + 16*ival[2];
		if (ival[3]) ptitle = ptit;
		else ptitle = NULL;
		xmode = ymode = 0;
		for (i=0; i<NDXY; i++) {
		    xmode += imult[i]*ival[NDG+i];
		    ymode += imult[i]*ival[NDG+NDXY+i];
		    }
		printf("n=%d,type=%d,xmode=%d,ymode=%d\n",n,type,xmode,ymode);
    digstart('\0',NULL,1);
		digclearscreen();
		dgraph((short *)x,(short *)y,n,xmode,ymode,type,20,ptitle,"x-axis label Hz","Hz");
		digflush();
		digquit(15);
		if (digdata.device == 's') {
		    digquit(15);
		    exit(0);
		}
		getchar();

		break;
		}
	    }
	}
    }
#endif
