/* winp -- DIG routines for WIN32 Printer Driver compatible graphics */

/* Mark Huckvale - University College London */

/* version 1.0 - November 1997 */

/* written for Microsoft Visual C/C++ V4 */

#include "SFSCONFG.h"
#ifdef	EVD

#include <windows.h>
#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <time.h>
#include <string.h>
#include "dig.h"
#include "digdata.h"

/* local storage of parameters */
struct {
	HANDLE	hPrinter;
	HDC	hDC;		/* printer device context */
	HFONT	hFont;		/* suitable printing font */
	int	mode;		/* system modes */
	int	subtype;	/* subtype as supplied from digmap */
	int	xoff,yoff;	/* page offset */
	int	dpi;		/* dots per inch */
	int	chwidth,chheight;
	int	xdpmm;		/* x pixels per mm */
	int	ydpmm;		/* y pixels per mm */
	int	lwidth;		/* line width = 1/8mm */
	int	gwidth;		/* grey maxel width = 1/4mm */
} digDEVD;

#define W32P_HDC_OK	1
#define W32P_START	2
#define W32P_FONT	4

/* forward declarations */
static void w32p_pen_colour();
static void w32p_brush_colour();

static void w32p_get_hdc()
{
	PRINTER_DEFAULTS pd;
	DWORD 		dwNeeded = 0;
	PRINTER_INFO_2 	*pi2;
	char	lbuf[256];
	char	*n,*d,*p;

	if (!(digDEVD.mode & W32P_HDC_OK)) {

		GetProfileString("windows", "device", ",,,", lbuf, 256);
		n = strtok(lbuf,",");
		d = strtok(NULL,", ");
		p = strtok(NULL,", ");

		if (n && d && p && *n && *d && *p) {

			/* Open printer handle */
			ZeroMemory(&pd, sizeof(pd));
			pd.DesiredAccess = PRINTER_ACCESS_USE;
			if (!OpenPrinter(n, &digDEVD.hPrinter, &pd))
				error("Failed to open printer");

			/* get the printer name */
			GetPrinter(digDEVD.hPrinter, 2, 0, 0, &dwNeeded);
			pi2 = (PRINTER_INFO_2 *)calloc(1,dwNeeded);
			if (!GetPrinter(digDEVD.hPrinter, 2, (LPBYTE)pi2, dwNeeded, &dwNeeded)) {
				ClosePrinter(digDEVD.hPrinter);
				error("Failed to get printer information");
			}

			/* set landscape */
			pi2->pDevMode->dmFields = DM_ORIENTATION;
			pi2->pDevMode->dmOrientation = DMORIENT_LANDSCAPE;

			/* get a device context */
			if ((digDEVD.hDC = CreateDC(NULL,pi2->pPrinterName,NULL,pi2->pDevMode))==NULL) {
				ClosePrinter(digDEVD.hPrinter);
				error("Failed to create printer device context");
			}

			free(pi2);

			digDEVD.mode |= W32P_HDC_OK;
		}
		else
			error("failed to find printer driver");
	}
}

static void w32p_get_font()
{
	LOGFONT		lf;
	HFONT		oldfont;
	TEXTMETRIC	tm;

	if (!(digDEVD.mode & W32P_FONT)) {
		memset((char *)&lf,0,sizeof(LOGFONT));

		lf.lfHeight = (300*(int)digDEVD.dpi)/2540;	/* 3mm */
		if (lf.lfHeight < 14) lf.lfHeight=14;
		lf.lfWeight = FW_NORMAL;
		lf.lfEscapement = 0;
		lf.lfOrientation = 0;
		lf.lfPitchAndFamily=FIXED_PITCH | FF_MODERN;
		strcpy(lf.lfFaceName,"Courier New");

		digDEVD.hFont = CreateFontIndirect(&lf);

		/* set up font */
		oldfont = SelectObject(digDEVD.hDC,digDEVD.hFont);
		DeleteObject(oldfont);
		SetTextAlign(digDEVD.hDC,TA_BOTTOM|TA_LEFT);
		GetTextMetrics(digDEVD.hDC,&tm);

		digDEVD.chheight = tm.tmHeight;
		digDEVD.chwidth = tm.tmAveCharWidth;

		digDEVD.mode |= W32P_FONT;
	}

}

/* get graphics parameters */
int w32p_params(subtype,digdev)
int	subtype;
struct digdevice *digdev;
{
	w32p_get_hdc();

	digDEVD.dpi = GetDeviceCaps(digDEVD.hDC,LOGPIXELSX);
	digdev->nhoriz = GetDeviceCaps(digDEVD.hDC,HORZRES)-2*digDEVD.dpi/3;
	digdev->nvert  = GetDeviceCaps(digDEVD.hDC,VERTRES)-2*digDEVD.dpi/3;
	digDEVD.xoff = digDEVD.dpi/3;
	digDEVD.yoff = digdev->nvert + digDEVD.dpi/3;
	digDEVD.xdpmm = GetDeviceCaps(digDEVD.hDC,LOGPIXELSX)/25;
	digDEVD.ydpmm = GetDeviceCaps(digDEVD.hDC,LOGPIXELSY)/25;
	digDEVD.lwidth = digDEVD.xdpmm/6;	/* target: 150 per inch */
	if (digDEVD.lwidth < 1) digDEVD.lwidth=1;
	digDEVD.gwidth = digDEVD.xdpmm/3;	/* target: 75 per inch */
	if (digDEVD.gwidth < 4) digDEVD.gwidth=4;
	if (digDEVD.gwidth > 6) digDEVD.gwidth=6;

	w32p_get_font();

	/* return other image characteristics */
	digdev->aspect = (float)1.0;
 	digdev->chwidth = digDEVD.chwidth;
	digdev->chheight = digDEVD.chheight;
	digdev->greylevels = 16;
	digdev->greypixwidth = digDEVD.gwidth;
	digdev->greypixheight = digDEVD.gwidth;

	return(0);
}

/* open output */
int w32p_open()
{
	DOCINFO	di;

	if (!(digDEVD.mode & W32P_START)) {
		w32p_get_hdc();
		w32p_get_font();

		memset((char *)&di,0,sizeof(DOCINFO));
		di.cbSize = sizeof(DOCINFO);
		di.lpszDocName = "SFS/DIG Print Job";
		di.lpszOutput = NULL;
		di.lpszDatatype = "emf" /*NULL*/;
		di.fwType = 0;

		if (StartDoc(digDEVD.hDC,&di)==SP_ERROR)
			error("StartDoc fails");

		if (StartPage(digDEVD.hDC)<=0)
			error("StartPage fails");

		digDEVD.mode |= W32P_START;
	}
	return(0);
}

/* close file */
void w32p_close()
{
	HFONT	font;

	if (digDEVD.mode & W32P_START) {

		/* End of page */
		if (EndPage(digDEVD.hDC)<=0)
			error("EndPage fails");

		/* End of document */
		if (EndDoc(digDEVD.hDC)<=0)
			error("EndDoc fails");

	}

	if (digDEVD.mode & W32P_FONT) {
		font = SelectObject(digDEVD.hDC,GetStockObject(SYSTEM_FONT));
		DeleteObject(font);
	}

	if (digDEVD.mode & W32P_HDC_OK) {
		/* Delete the printer DC. */
		DeleteDC(digDEVD.hDC);
		ClosePrinter(digDEVD.hPrinter);
	}

	digDEVD.mode = 0;
}

void w32p_inter()
{
	w32p_close();
}

/* clear screen */
int w32p_clear()
{
	return(0);
}

/* set up pen colour from bundle */
static void w32p_pen_colour(bundle)
int	bundle;
{
	static int oldbundle = -1;
	HPEN	pen,oldpen;
	int	dash=0,width=0;

	if (bundle==oldbundle) return;
	oldbundle = bundle;

	if (bundle < 6)
		width = 4*digDEVD.lwidth;
	else if (bundle < 20)
        	width = 2*digDEVD.lwidth;
	else
		width = 1*digDEVD.lwidth;

        if (bundle < 35)
		dash = PS_SOLID;
        else if (bundle < 50)
		dash = PS_DASH;
        else if (bundle < 65)
		dash = PS_DOT;
        else if (bundle < 80)
		dash = PS_DASHDOT;
        else
		dash = PS_DASHDOTDOT;

	if (bundle==0)
		pen = CreatePen(dash,width,RGB(255,255,255));
	else
		pen = CreatePen(dash,width,RGB(0,0,0));

	oldpen = SelectObject(digDEVD.hDC,pen);

	DeleteObject(oldpen);
}

/* set up brush colour from bundle */
static void w32p_brush_colour(bundle)
int	bundle;
{
	static int oldbundle = -1;
	HBRUSH	brush,oldbrush;

	if (bundle==oldbundle) return;
	oldbundle = bundle;

	brush = (bundle==0) ? GetStockObject(WHITE_BRUSH) : GetStockObject(BLACK_BRUSH);

	oldbrush = SelectObject(digDEVD.hDC,brush);

	DeleteObject(oldbrush);
}

/* plot poly-line */
void w32p_polyline(bundle,buff,len)
int bundle;
short *buff;
int	len;
{
	int     x,y;

	/* set up plot colour */
	w32p_pen_colour(bundle);

	/* move to first point */
	x = digDEVD.xoff + *buff++;
	y = digDEVD.yoff - *buff++;
	len-=(int32)2;
	MoveToEx(digDEVD.hDC,x,y,NULL);

	/* plot lines */
	while (len > (int32)0) {
		x = digDEVD.xoff + *buff++;
		y = digDEVD.yoff - *buff++;
		len -= (int32)2;
		LineTo(digDEVD.hDC, x, y);
	}
}

/* draw text */
void w32p_text(bundle,x,y,str)
int bundle;
int x;
int y;
char *str;
{
	/* set up plot colour */
	w32p_pen_colour(bundle);

	/* draw string */
	SetBkMode(digDEVD.hDC,TRANSPARENT);
	TextOut(digDEVD.hDC,digDEVD.xoff+x,digDEVD.yoff-y,str,strlen(str));
	SetBkMode(digDEVD.hDC,OPAQUE);

}

/* draw a filled rectangle */
void w32p_fillrect(bundle,x1,y1,x2,y2)
int bundle;
int x1;
int y1;
int x2;
int y2;
{
	w32p_pen_colour(bundle);
	w32p_brush_colour(bundle);

	Rectangle(digDEVD.hDC,digDEVD.xoff+x1,digDEVD.yoff-y1,digDEVD.xoff+x2,digDEVD.yoff-y2);

}

/* draw a circle */
void w32p_circle(bundle,xc,yc,r)
int bundle;
int xc;
int yc;
int r;
{
	/* set up plot colour */
	w32p_pen_colour(bundle);

	/* draw circle */
	Arc(digDEVD.hDC,digDEVD.xoff+xc-r,digDEVD.yoff-yc-r,digDEVD.xoff+xc+r,digDEVD.yoff-yc+r,
		digDEVD.xoff+xc+r,digDEVD.yoff-yc,digDEVD.xoff+xc+r,digDEVD.yoff-yc);
}

/* draw a filled circle */
void w32p_fcircle(bundle,xc,yc,r)
int bundle;
int xc;
int yc;
int r;
{
	w32p_pen_colour(bundle);
	w32p_brush_colour(bundle);

	Ellipse(digDEVD.hDC,digDEVD.xoff+xc-r,digDEVD.yoff-yc-r,digDEVD.xoff+xc+r,digDEVD.yoff-yc+r);

}

/* grey-level display */
void w32p_gscale(x,y,height,buff)
int x;
int y;
int height;
unsigned char *buff;
{
	HANDLE		oldobj;
	static HPEN	hpen[16]={NULL};
	int		i,lasty;
	unsigned char 	*p,lastc;

        /* make 16 pens */
        if (hpen[0]==NULL) {
		hpen[0] = CreatePen(PS_SOLID,digDEVD.gwidth,RGB(255,255,255));
		hpen[1] = CreatePen(PS_SOLID,digDEVD.gwidth,RGB(238,238,238));
		hpen[2] = CreatePen(PS_SOLID,digDEVD.gwidth,RGB(221,221,221));
		hpen[3] = CreatePen(PS_SOLID,digDEVD.gwidth,RGB(204,204,204));
		hpen[4] = CreatePen(PS_SOLID,digDEVD.gwidth,RGB(187,187,187));
		hpen[5] = CreatePen(PS_SOLID,digDEVD.gwidth,RGB(170,170,170));
		hpen[6] = CreatePen(PS_SOLID,digDEVD.gwidth,RGB(153,153,153));
		hpen[7] = CreatePen(PS_SOLID,digDEVD.gwidth,RGB(136,136,136));
		hpen[8] = CreatePen(PS_SOLID,digDEVD.gwidth,RGB(119,119,119));
		hpen[9] = CreatePen(PS_SOLID,digDEVD.gwidth,RGB(102,102,102));
		hpen[10] = CreatePen(PS_SOLID,digDEVD.gwidth,RGB(85,85,85));
		hpen[11] = CreatePen(PS_SOLID,digDEVD.gwidth,RGB(68,68,68));
		hpen[12] = CreatePen(PS_SOLID,digDEVD.gwidth,RGB(51,51,51));
		hpen[13] = CreatePen(PS_SOLID,digDEVD.gwidth,RGB(34,34,34));
		hpen[14] = CreatePen(PS_SOLID,digDEVD.gwidth,RGB(17,17,17));
		hpen[15] = CreatePen(PS_SOLID,digDEVD.gwidth,RGB(0,0,0));
	}

	/* get ready */
	oldobj = SelectObject(digDEVD.hDC,hpen[0]);
	x += digDEVD.xoff+1;

	/* draw coloured lines matching greys */
	p = buff;
	lastc = *p;
	lasty = digDEVD.yoff - y - digDEVD.gwidth*height +1;
	MoveToEx(digDEVD.hDC,x,lasty,NULL);
	for (i = 1; i < height; i++, p++) {
		if (*p != lastc) {
			SelectObject(digDEVD.hDC,hpen[lastc]);
			LineTo(digDEVD.hDC,x,lasty+digDEVD.gwidth*i);
			lastc = *p;
		}
	}
	SelectObject(digDEVD.hDC,hpen[lastc]);
	LineTo(digDEVD.hDC,x,lasty+digDEVD.gwidth*(height-1));
	SelectObject(digDEVD.hDC,oldobj);
}

/* wait for key routine - NULL don't wait */
void w32p_pause()
{
}
#endif
