/* complex.c -- routines for complex arithmetic */

/*
 * (c) 1994 Mark Huckvale
 * Search Systems Ltd
 * 52-54 High Holborn
 * London WC1E 6BT
 * Tel. 071-430 1685
 */

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "complex.h"

/* make a complex number */
COMPLEX CMake(rval,ival)
double rval,ival;
{
	COMPLEX	c;
	c.re = rval;
	c.im = ival;
	return(c);
}

/* add complex numbers */
COMPLEX CAdd(c1,c2)
COMPLEX c1,c2;
{
	COMPLEX	c;
	c.re = c1.re + c2.re;
	c.im = c1.im + c2.im;
	return(c);
}

/* subtract complex numbers */
COMPLEX CSub(c1,c2)
COMPLEX c1,c2;
{
	COMPLEX	c;
	c.re = c1.re - c2.re;
	c.im = c1.im - c2.im;
	return(c);
}

/* multiply complex numbers */
COMPLEX CMult(c1,c2)
COMPLEX c1,c2;
{
	COMPLEX c;
	c.re = c1.re * c2.re - c1.im * c2.im;
	c.im = c1.re * c2.im + c1.im * c2.re;
	return(c);
}

/* divide complex numbers */
COMPLEX CDiv(c1,c2)
COMPLEX c1,c2;
{
	COMPLEX c;
	double	denom;

	if ((c2.re==0) && (c2.im==0)) {
		fprintf(stderr,"CDiv: division by zero\n");
		return(c2);
	}
	else {
		denom = c2.re * c2.re + c2.im * c2.im;
		c.re = (c1.re * c2.re + c1.im * c2.im) / denom;
		c.im = (c1.im * c2.re - c1.re * c2.im) / denom;
		return(c);
	}
}

/* magnitude of complex number */
double CMag(c)
COMPLEX c;
{
	return(sqrt(c.re*c.re + c.im*c.im));
}

/* argument of complex number */
double CArg(c)
COMPLEX c;
{
	if ((c.im==0) && (c.re==0))
		return(0.0);
	else
		return(atan2(c.im,c.re));
}

/* complex exponential */
COMPLEX CExp(c1)
COMPLEX c1;
{
	COMPLEX c;
	double	r;
	r = exp(c1.re);
	c.re = r * cos(c1.im);
	c.im = r * sin(c1.im);
	return(c);
}

/* complex square root */
COMPLEX CSqrt(cin)
COMPLEX cin;
{
	COMPLEX c;
	double x,y,w,r;

	if ((cin.re==0) && (cin.im==0)) {
		c.re = 0.0;
		c.im = 0.0;
	}
	else {
		x = fabs(cin.re);
		y = fabs(cin.im);
		if (x >= y) {
			r = y/x;
			w = sqrt(x)*sqrt(0.5*(1.0+sqrt(1.0+r*r)));
		}
		else {
			r = x/y;
			w = sqrt(y)*sqrt(0.5*(r+sqrt(1.0+r*r)));
		}
		if (cin.re >= 0.0) {
			c.re = w;
			c.im = cin.im/(2.0*w);
		}
		else if (cin.im >= 0.0) {
			c.im = w;
			c.re = cin.im/(2.0*c.im);
		}
		else {
			c.im = -w;
			c.re = cin.im/(2.0*c.im);
		}
	}
	return(c);
}

/* scale a complex number */
COMPLEX CScale(cin,fac)
COMPLEX cin;
double fac;
{
	COMPLEX c;
	c.re = cin.re * fac;
	c.im = cin.im * fac;
	return(c);
}

#ifdef EMO

main()
{
	COMPLEX	a,b,c;

	a = CMake(1.0,2.0);
	b = CMake(3.0,4.0);
	c = CAdd(a,b);
	CPrint(a); putchar('+'); CPrint(b); putchar('='); CPrint(c); putchar('\n');

	a = c;
	b = CMake(3.0,1.0);
	c = CSub(a,b);
	CPrint(a); putchar('-'); CPrint(b); putchar('='); CPrint(c); putchar('\n');

	a = c;
	b = CMake(2.0,5.0);
	c = CMult(a,b);
	CPrint(a); putchar('*'); CPrint(b); putchar('='); CPrint(c); putchar('\n');

	a = c;
	b = CMake(3.0,2.0);
	c = CDiv(a,b);
	CPrint(a); putchar('/'); CPrint(b); putchar('='); CPrint(c); putchar('\n');

	printf("Magnitude"); CPrint(c); putchar('='); printf("%g\n",CMag(c));
	printf("Argument"); CPrint(c); putchar('='); printf("%g\n",CArg(c));
	printf("SquareRoot"); CPrint(c); putchar('='); CPrint(CSqrt(c)); putchar('\n');

	a = CSqrt(c);
	c = CMult(a,a);
	CPrint(a); putchar('*'); CPrint(a); putchar('='); CPrint(c); putchar('\n');
	
}
#endif
	
