/* respeed -- change speaking rate without pitch change */

/* M.A.Huckvale - University College London */

/* version 0.1 - January 1994 */

/* version 2.0 - October 2002
	- add option for control file
*/

#define PROGNAME "respeed"
#define PROGVERS "2.0"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH RESPEED 1 SFS UCL
.SH NAME
respeed - change speaking rate without change in pitch
.SH SYNOPSIS
.B respeed
(-I) (-i item) (-r rate-change|-m MBROLA-control-file) file
.SH DESCRIPTION
.I respeed
is a program to speed-up or slow-down speech signals without changing
the pitch.  It is based on the SOLA algorithm of Roucos and Wilgur
(ICASSP-86).  Options control whether the whole signal is changed, or
whether different annotated regions are changed by different amounts.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -i item
Select input item number.
.TP 11
.BI -r rate-change
Relative speed up factor. 2.0 = twice as fast, 0.5=half as fast.  Default 1.
.TP 11
.BI -m MBROLA-control-file
Specify a new duration for each annotated region using the format of an
MBROLA control file.  Respeed only looks at the first two entries on each line;
the first should be the name of an annotated region, the second should be
the required duration in milliseconds.  Use the anmbrola program to build
a default control file from a set of annotations.
.SH INPUT ITEMS
.IP SP 11
Speech item
.IP AN 11
Annotation item (control file input).
.SH OUTPUT ITEMS
.IP SP 11
Rate changed speech
.SH HISTORY
.IP rate=
proportional rate-change.
.IP control=
Control file name
.SH VERSION/AUTHOR
.IP 2.0
Mark Huckvale
.SH SEE ALSO
anmbrola, repros, repitch.
.SH BUGS
*/
/*--------------------------------------------------------------------------*/

/* include files */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
#include <malloc.h>
#include "sfs.h"
#define MAX(x,y) (((x)>(y))?(x):(y))
#define ABS(x) (((x)>0)?(x):-(x))

/* global defines */
#define WINTIME		0.032		/* size of input windows */
#define SYNCTIME	0.010		/* search for synchronisation
						+/- SYNCTIME */

/* global data */
struct item_header	spitem;
struct item_header	opitem;
struct item_header	anitem;
short			*isp,*osp;	/* input/output signal */
float			*risp,*rosp;	/* real input/output signal */
float			*wisp,*wosp;	/* real input/output window */
struct an_rec	*an;
int			bufsize;	/* size of internal buffers */
int			winsize;	/* size of an analysis window */
int			stepsize;	/* step size for input frames */
double			newrate=1.0;	/* speed-up rate */
int			verbose=0;	/* dump debugging info */
char		mfilename[256];		/* MBROLA control file name */

/* segment record */
struct segment_rec {
	char	*label;		/* segment name */
	double	ostime;		/* original start time */
	double	odur;		/* original duration */
	double	nstime;		/* new start time */
	double	ndur;		/* new duration */
};
struct segment_rec *stab;
int	scount;
double	totdur;

/* save string in memory */
char *strsave(char *str)
{
	char *ptr=malloc(strlen(str)+1);
	if (ptr==NULL)
		error("out of memory");
	strcpy(ptr,str);
	return(ptr);
}

/* load MBROLA control file */
void loadmbrola(char *fname)
{
	FILE	*ip;
	char	line[1024];
	char	label[128];
	char	*p;
	int	dur;
	int	i,j,idx;
	double	t,pc;
	int	fx;
	int	f1,f2;

	/* open file */
	if ((ip=fopen(fname,"r"))==NULL)
		error("could not find '%s'",fname);

	/* count lines in file */
	scount=0;
	while (fgets(line,1024,ip)) if (line[0]!=';') {
		if (sscanf(line,"%s %d",label,&dur)==2) scount++;
	}
	rewind(ip);

	/* get memory table for segments */
	stab = (struct segment_rec *)calloc(scount,sizeof(struct segment_rec));

	/* run through control lines, relating to segments on file */
	stab[0].nstime=an[0].posn * anitem.frameduration;
	for (i=0;fgets(line,1024,ip);) if (line[0]!=';') {
		if (i>0) {
			stab[i].nstime = stab[i-1].nstime+stab[i-1].ndur;
		}
		p = strtok(line," \t\n");
		if ((i<scount) && (i < anitem.numframes) && (strcmp(p,an[i].label)==0)) {
			p = strtok(NULL," \t\n");
			stab[i].ndur = atoi(p)/1000.0;
			stab[i].ostime = an[i].posn * anitem.frameduration;
			if (i < anitem.numframes-1)
				stab[i].odur = (an[i+1].posn - an[i].posn) * anitem.frameduration;
			else
				stab[i].odur = an[i].size * anitem.frameduration;
		}
		else {
			fprintf(stderr,"Line %d. Target:'%s' Control-File:'%s'\n",i,an[i].label,p);
			error("segment mismatch between control file and target file");
		}

		i++;
	}
	totdur = stab[scount-1].nstime + stab[scount-1].ndur;

	fclose(ip);
}

/* given a new time, find matching old time */
double findoldtime(double newtime)
{
	int	i;
	double t;

	if (newtime < stab[0].nstime)
		return(newtime);

	for (i=0;i<scount;i++) {
		if ((stab[i].nstime <= newtime) && (newtime < stab[i].nstime+stab[i].ndur)) {
			/* linear inside segment */
			t = stab[i].ostime + stab[i].odur * (newtime-stab[i].nstime) / stab[i].ndur;
			return(t);
		}
	}

	return(0.0);
}

/* half window signal */
void halfwindow(short *buf,int len,float *obuf,float *wbuf)
{
	double 	w;
	int	i;

	w = M_PI/(len+1);
	for (i=1;i<=len;i++,buf++,wbuf++,obuf++) {
		*wbuf = (float)(0.5 + 0.5*cos(i*w));
		*obuf = (float)*buf * *wbuf;
	}
}

/* window signal */
void window(short *buf,int len,float *obuf,float *wbuf)
{
	double 	w;
	int	i;

	w = 2.0*M_PI/(len+1);
	for (i=1;i<=len;i++,buf++,wbuf++,obuf++) {
		*wbuf = (float)(0.5 - 0.5*cos(i*w));
		*obuf = (float)*buf * *wbuf;
	}
}

/* align window with output */
int	align(short *wbuf,int wlen,short *sbuf,int slen,int omin,int omax,int otarg)
{
	int	i,j;
	float	sum;
	int	offset;
	float	peakval;
	float	ac[2048];
	int	currdist;

	if (omax > slen) omax=slen;
	if (omax > 2045) omax=2045;	/* to fit ac[], OK for 48kHz/32ms */
	if (omax > wlen) omax=wlen-1;
	if (omin >= wlen)
		return(slen-wlen/2);
	offset = (slen<wlen/2) ? 0 : wlen/2;
	ac[omin-2]=(float)0.0;
	ac[omin-1]=(float)0.0;
	for (i=omin;i<=omax;i++) {
		sum = (float)0.0;
		for (j=0;j<i;j++)
			sum += (float)wbuf[i-j-1] * (float)sbuf[slen-j-1];
		ac[i] = sum/i;
	}
	ac[omax+1]=(float)0.0;
	ac[omax+2]=(float)0.0;

	/* smooth */
	peakval=(float)0.0;
	for (i=omin;i<=omax;i++) {
		ac[i] = (ac[i-2]+ac[i-1]+ac[i]+ac[i+1]+ac[i+2])/5;
		if (ac[i]>peakval) peakval=ac[i];
	}

	/* search for peak close to target */
	peakval *= (float)0.75;
	currdist=omax;
	if (verbose==2) printf("peaks at ");
	for (i=omin;i<=omax;i++) if ((ac[i-1] < ac[i]) && (ac[i] > ac[i+1])) {
		if ((verbose==2) && (ac[i]>peakval))
			printf("%d(%g),",i,ac[i]);
		if ((ac[i]>peakval) && (ABS(otarg-i)<currdist)) {
			currdist = ABS(otarg-i);
			offset = i;
		}
	}
	if (verbose==2) printf(" chosen=%d\n",offset);

	return(slen-offset);
}

/* overlap-add two signals */
int	overlapadd(float *s1buf,float *w1buf,int s1len,float *s2buf,float *w2buf,int s2off)
{
	int	i;

	s2buf += s2off;
	w2buf += s2off;
	for (i=0;i<s1len;i++,s1buf++,w1buf++,s2buf++,w2buf++) {
		*s2buf += *s1buf;
		*w2buf += *w1buf;
	}
	return(s2off+s1len);
}

/* re-synthesize */
void outsynth(short *obuf,float *sbuf,float *wbuf,int slen)
{
	int	i;

	for (i=0;i<slen;i++,obuf++,sbuf++,wbuf++) {
		if (*wbuf==0.0) {
/*			fprintf(stderr,"zero window\n"); */
			*obuf = 32767;
		}
		else
			*obuf = (short)(*sbuf / *wbuf);
	}
}

/* main program */
void main(int argc,char *argv[])
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item selection */
	char		*ty;		/* item sub type */
	char		*sptype="0";
	char		*antype="0";
	/* file variables */
	char		filename[SFSMAXFILENAME]; /* SFS data file name */
	int		fid;		/* input file descriptor */
	int		ofid;
	double		curtime,endtime;
	double		halfwintime,steptime;
	int		cursamp,newsamp;
	int		frameno,framelen,offset;
	int		overtarg,overmin,overmax;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:r:vm:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Speed changing of signal V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					sptype=ty;
				else if (it == AN_TYPE)
					antype=ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'r' :	/* output rate */
			newrate = atof(optarg);
			break;
		case 'm':
			strcpy(mfilename,optarg);
			break;
		case 'v' :	/* verbose */
			verbose++;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-r rate-change|-m MBROLA-control-file) (-v) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open file */
	if ((fid=sfsopen(filename,"w",NULL))<0)
		error("access error on '%s'",filename);

	/* load annotations if required */
	if (mfilename[0]) {
		if (!sfsitem(fid,AN_TYPE,antype,&anitem))
			error("unable to find input annotation item in '%s'",filename);
		an = (struct an_rec *)sfsbuffer(&anitem,anitem.numframes);
		sfsread(fid,0,anitem.numframes,an);
	}

	/* load and map control file, if required */
	if (mfilename[0]) loadmbrola(mfilename);

	/* locate item */
	if (!sfsitem(fid,SP_TYPE,sptype,&spitem))
		error("unable to find input item in '%s'",filename);

	/* get sizes */
	halfwintime = WINTIME/2;
	winsize = (int)(WINTIME/spitem.frameduration);
	bufsize = 10*winsize;
	steptime = 0.5*newrate*WINTIME;
	stepsize = (int)(steptime/spitem.frameduration);

	/* get buffers */
	if ((isp=(short *)sfsbuffer(&spitem,bufsize))==NULL)
		error("could not get memory buffer");
	if ((osp=(short *)sfsbuffer(&spitem,bufsize))==NULL)
		error("could not get memory buffer");
	if ((risp=(float *)calloc(bufsize,sizeof(float)))==NULL)
		error("could not get memory buffer");
	if ((rosp=(float *)calloc(bufsize,sizeof(float)))==NULL)
		error("could not get memory buffer");
	if ((wisp=(float *)calloc(bufsize,sizeof(float)))==NULL)
		error("could not get memory buffer");
	if ((wosp=(float *)calloc(bufsize,sizeof(float)))==NULL)
		error("could not get memory buffer");

	/* make output header */
	sfsheader(&opitem,SP_TYPE,0,2,1,spitem.frameduration,spitem.offset,1,0,0);
	if (mfilename[0])
		sprintf(opitem.history,"%s(%d.%02d;control=%s)",
				PROGNAME,
				spitem.datatype,spitem.subtype,
				mfilename);
	else
		sprintf(opitem.history,"%s(%d.%02d;rate=%g)",
				PROGNAME,
				spitem.datatype,spitem.subtype,
				newrate);

	/* get output channel */
	if ((ofid=sfschannel(filename,&opitem))<0)
		error("could not open output channel to '%s'",filename);

	/* pre-charge buffer with half a window */
	memset((char *)osp,0,2*bufsize);
	memset((char *)rosp,0,4*bufsize);
	memset((char *)wosp,0,4*bufsize);
	framelen = sfsread(fid,0,winsize/2,isp);
	halfwindow(isp,winsize/2,rosp,wosp);
	cursamp=winsize/2;
	curtime=halfwintime;

	/* create 'dummy' output in osp */
	outsynth(osp,rosp,wosp,cursamp);

	/* calculate min and max overlap alignments */
	overtarg = winsize/2;
	overmin = (int)(overtarg-SYNCTIME/spitem.frameduration);
	overmax = (int)(overtarg+SYNCTIME/spitem.frameduration);
	if (overmin < winsize/16) overmin = winsize/16;
	if (overmax > 15*winsize/16) overmax = 15*winsize/16;

	/* processing loop */
	if (mfilename[0])
		endtime = totdur;
	else
		endtime=spitem.numframes*spitem.frameduration/newrate;
	while (curtime < endtime) {

		/* calculate next frame to use */
		if (mfilename[0])
			frameno = (int)(findoldtime(curtime)/spitem.frameduration);
		else
			frameno = (int)((curtime*newrate/steptime - 1)*stepsize);
		if (frameno < 0) frameno=0;
		if (frameno > (spitem.numframes - winsize)) frameno = spitem.numframes-winsize;

		/* load this frame */
		framelen = sfsread(fid,frameno,winsize,isp);
		if (framelen <= 0)
			error("read error on '%s'\n",filename);

		/* align with what we have so far */
		offset = align(isp,framelen,osp,cursamp,overmin,overmax,overtarg);

		/* window frame */
		window(isp,framelen,risp,wisp);

		/* add in */
		newsamp = overlapadd(risp,wisp,framelen,rosp,wosp,offset);

		if (verbose)
			printf("Out %6.3f-%.3f\tIn %6.3f-%.3f\tSync at %6.3fs delay.\n",
				curtime+(offset-cursamp)*spitem.frameduration,
				curtime+(newsamp-cursamp)*spitem.frameduration,
				frameno*spitem.frameduration,
				(frameno+framelen)*spitem.frameduration,
				(cursamp-offset)*spitem.frameduration);

		/* create 'dummy' output */
		outsynth(osp,rosp,wosp,newsamp);
		curtime += (newsamp-cursamp)*spitem.frameduration;
		cursamp = newsamp;

		/* purge output from time to time */
		if (cursamp >= (bufsize-winsize)) {
			outsynth(osp,rosp,wosp,bufsize/2);
			/* write out first half */
			sfswrite(ofid,bufsize/2,osp);
			memcpy((char *)osp,(char *)(osp+bufsize/2),bufsize);
			memcpy((char *)rosp,(char *)(rosp+bufsize/2),bufsize*2);
			memcpy((char *)wosp,(char *)(wosp+bufsize/2),bufsize*2);
			memset((char *)(osp+bufsize/2),0,bufsize);
			memset((char *)(rosp+bufsize/2),0,bufsize*2);
			memset((char *)(wosp+bufsize/2),0,bufsize*2);
			cursamp -= bufsize/2;
		}
	}

	/* write last buffer */
	outsynth(osp,rosp,wosp,cursamp);
	sfswrite(ofid,cursamp,osp);

	/* that's all folks */
	if (!sfsupdate(filename))
		error("update error on '%s'",filename);
	exit(0);
}

