/* sprint -- spectrogram print */

/* M.A.Huckvale - University College London */

/* version 1.0 - January 1994 */

#define PROGNAME "sprint"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH SPRINT 1 SFS UCL
.SH NAME
sprint - print paged spectrograms of long utterances
.SH SYNOPSIS
.B sprint
(-i item) (-s starttime) (-e endtime) (-t title) file
.SH DESCRIPTION
.I sprint
is a program to print conventional wide and narrow band spectrograms
of a waveform at standard sizes, and that extend over more than page
if necessary.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -i item
Select input item number.
.TP 11
.BI -s starttime
Specify start time in seconds.  Default 0.
.TP 11
.BI -e endtime
Specify end time in seconds.  Default: end of signal.
.TP 11
.BI -t title
Specify title.
.TP 11
.B -a
Leave room for annotation between spectrograms.
.SH INPUT ITEMS
.IP SP 11
Speech signal.
.SH VERSION/AUTHOR
.IP 1.0
Mark Huckvale
.SH SEE ALSO
Es
.SH BUGS
*/
/*--------------------------------------------------------------------------*/

/* include files */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <string.h>
#include "sfs.h"
#include "dig.h"
#include "digdata.h"
#include "digitem.h"

/* definitions */
#define MAX(x,y) (((x)>(y))?(x):(y))
#define MIN(x,y) (((x)<(y))?(x):(y))
#define PAGETIME	1.25		/* 1.25 seconds/page */
#define OVERTIME	0.15		/* 0.15 seconds overlap */
#define WINTIME		0.05		/* additional buffer time, to avoid blank strips */

/* global variables */
char			filename[SFSMAXFILENAME]; /* SFS data file name */
char			*title;
struct main_header	head;
struct item_header	spitem;
short			*sp;
double			starttime = -1;
double			endtime = -1;
int			doannot=0;

/* plot a page */
void plotspect(item,buff,start,len,pageno)
struct item_header	*item;
short			*buff;
int			start;
int			len;
int			pageno;
{
	char	ptitle[256];
	double	tsize;
	double	stime,etime;
	double	y1,y2;
	struct item_header anitem;
	struct an_rec anrec;

	/* get basic size of a character */	
	tsize = 2 * digdata.chheight / digdata.yscale;
	y1 = (doannot) ? 0.425+1.5*tsize : 0.425+0.5*tsize;
	y2 = (doannot) ? 0.425-0.5*tsize : 0.425+0.5*tsize;
	stime = start * item->frameduration;
	etime = stime + PAGETIME;

	/* display filename, speaker and token */
	digclearscreen();
	sprintf(ptitle,"Page %d: file=%s speaker=%s token=%s",
		pageno,filename,head.speaker,head.token);
	digtext(24,0.0,1.0-2*tsize/3,ptitle);

	/* display print title */
	if (title)
		digtext(24,1.0-strlen(title)*digdata.chwidth/digdata.xscale,
			1.0-2*tsize/3,title);

	/* display top timescale */
	digitemtime(24,0.0,1.0-2*tsize,1.0,1.0-tsize,stime,etime,1);

	/* display waveform */
	digitemSP(232320,0.0,0.85,1.0,1.0-2*tsize,item,(short *)buff,
		stime,etime,
		DIGITEMBOX | DIGITEMTITLE | DIGITEMLABEL);

	/* display narrow-band */
	digitemFFT(242420,0.0,y1,1.0,0.85,item,(short *)buff,
		stime,etime,
		DIGITEMBOX | DIGITEMFFTNARROW | DIGITEMGRID );

	if (doannot) {
		sfsheader(&anitem,AN_TYPE,-1,1,-1,1.0,0.0,0,0,1);
		sprintf(anitem.history,"Annotations");
		anrec.posn=0;
		anrec.size=0;
		anrec.label="";
		anitem.numframes=0;
		digitemAN(242420,0.0,y2,1.0,y1,&anitem,&anrec,stime,etime,
			DIGITEMBOX);
	}
	
	/* display wide-band */
	digitemFFT(242420,0.0,tsize,1.0,y2,item,(short *)buff,
		stime,etime,
		DIGITEMBOX | DIGITEMGRID );

	/* display bottom timescale */
	digitemtime(24,0.0,0.0,1.0,tsize,stime,etime,1);

}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item selection */
	char		*ty;		/* item sub type */
	char		*sptype="0";
	/* file variables */
	int		fid;		/* input file descriptor */
	int		startsamp,endsamp,pagesamp,oversamp,winsamp,nsamp;
	int		pno=0;		/* page number */
	struct item_header item;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:s:e:t:a")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Print spectrograms V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					sptype = ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 's' :	/* start time */
			starttime = atof(optarg);
			break;
		case 'e' :	/* end time */
			endtime = atof(optarg);
			break;
		case 't' :	/* title */
			title = optarg;
			break;
		case 'a' :	/* do annotation box */
			doannot++;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-s starttime) (-e endtime) (-t title) (-a) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open input file */
	if ((fid=sfsopen(filename,"r",&head)) < 0)
		error("could not open '%s'",filename);

	/* locate speech item */
	if (!sfsitem(fid,SP_TYPE,sptype,&spitem))
		error("could not locate input item in '%s'",filename);

	/* calculate page size */
	pagesamp = (int)(PAGETIME/spitem.frameduration);
	oversamp = (int)(OVERTIME/spitem.frameduration);
	winsamp = (int)(WINTIME/spitem.frameduration);

	/* get speech buffer */
	if ((sp=(short *)sfsbuffer(&spitem,pagesamp+2*winsamp))==NULL)
		error("could not get memory buffer");

	/* get start and end samples */
	if (starttime > 0)
		startsamp = (int)(starttime/spitem.frameduration);
	else
		startsamp = 0;
	if (endtime > 0)
		endsamp = (int)(endtime/spitem.frameduration);
	else
		endsamp = spitem.numframes;
	if (endsamp <= startsamp)
		error("no signal to analyse in '%s'",filename);
		
	/* loop through spectrogram pages */
	while (startsamp < endsamp) {

		/* load sufficient speech data (more than abs. min.) */
		nsamp = sfsread(fid,MAX(0,startsamp-winsamp),
				MIN(endsamp-startsamp+winsamp,pagesamp+2*winsamp),sp);

		/* open graphics */
		digstart(DIG_DEFAULT_PRINTER,NULL,1);

		/* plot spectrogram for this page */
		item = spitem;
		item.offset += MAX(0,startsamp-winsamp)*item.frameduration;
					/* move header to where buffer starts */
		item.numframes = nsamp;	/* limit on contents of buffer */
		plotspect(&item,sp,startsamp,MIN(nsamp,pagesamp),++pno);

		/* close printer */
		digflush();
		digend();

		/* next page */
		startsamp += pagesamp-oversamp;
	}

	/* close */
	sfsclose(fid);
	exit(0);
}

