/* anload -- load an annotation item from a text description of times and labels */

/* M.A.Huckvale - December 1987 */

/* version 1.0 */
/* version 1.1 - February 1989
	- fix lxsync in output AN
*/
/* version 1.2 - June 1989
	- TIMIT database option, annotations with sample ranges
*/
/* version 1.3 - March 1992
	- add -h HTK format load option
*/
/* version 1.4 - October 1998
	- allow spaces in labels
	- add -e ESPS format load option
*/
/* version 1.5 - February 2000
	- add -m MBROLA format load option
	- add -M millisecond format load option
*/
/* version 1.6 - March 2000
	- add loading of Fx into mbrola option
*/
/* version 1.7 - February 2002
	- attempt to support sample format with single times automatically
*/

/*-------------------------------------------------------------------*/
/**MAN
.TH ANLOAD SFS1 UCL
.SH NAME
anload -- load an annotation item from text description
.SH SYNOPSIS
.B anload
(-f freq) (-t type) (-s|-h|-e|-m|-M) textfile file
.SH DESCRIPTION
.I anload
constructs a new annotation item in the given file from text
descriptions of annotations in a text file.
The default format of the text file is one annotation per line.  In the
normal mode this should have two
entries: firstly the time in seconds, secondly the text label (not
containing spaces).  In the "sample" mode (-s switch), each line
should contain three entries: the starting sample number, the final sample number,
and the annotation.  Ensure that the sampling frequency is correct in the
sample mode.
.PP
.B Options
.TP 11
.B -I
Identify the program version.
.TP 11
.BI -f freq
Use a sampling interval based on 1/freq seconds for sampling the time of each
annotation. Default 10000.
.TP 11
.BI -t type
Place a "type" indicator in the created item header, with the argument
.I type.
.TP 11
.B -s
Sample mode.  Text lines contain sample numbers or sample ranges, not times.
For example: "1000 label" or "1000 2000 label".
Ensure sampling frequency is supplied with the -f switch.
.TP 11
.B -h
HTK format load.  Equivalent to switches: -s -f 10000000.
.TP 11
.B -e
ESPS compatible label file format load.
.TP 11
.B -m
MBROLA .PHO file compatible label file format load.
.TP 11
.B -M
Millisecond format.  Lines contain individual durations in milliseconds (not times)
and then label.
.SH OUTPUT ITEMS
.IP AN 11
Loaded annotation item
.SH VERSION/AUTHOR
1.7 - Mark Huckvale
.SH BUGS
Except in sample mode, annotation sizes are calculated automatically, the last annotation
has length zero.
*/
/*---------------------------------------------------------------*/

/* program name and version */
#define	PROGNAME "anload"
#define PROGVERS "1.7"
char	*progname = PROGNAME;

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>		/* standard i-o routines */
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include "sfs.h"		/* database filing system structures */

/* global data */
struct item_header 	anitem;			/* annotation item header  */
struct an_rec		*an;			/* annotation record */
struct item_header	fxitem;			/* Fx item header */
short			*fx;			/* Fx data */
int			fxcnt;
char			textname[SFSMAXFILENAME];	/* text file name */
char			filename[SFSMAXFILENAME];	/* dbase file name */
double			fdur=1.0E-4;		/* default sample duration */
char			*antype;		/* annotation type */
int			sampmode=0;		/* sample mode */
int			esps=0;			/* ESPS mode */
int			mbrola=0;		/* MBROLA mode */
int			millisec=0;		/* millisecond mode */
int			samp1;			/* one sample per line not two */

/* process fx entry in MBROLA file */
void processfx(char *line,double stime,double etime)
{
	char	*p;
	double	percent,fxval;
	int	idx;

	/* skip first two fields */
	p = strtok(line," \t\r\n");
	p = strtok(NULL," \t\r\n");
	p = strtok(NULL," \t\r\n");
	while (p && *p && isdigit(*p)) {
		/* first value is percentage through segment */
		percent = atof(p);
		/* second value is fx value */
		p = strtok(NULL," \t\r\n");
		if (p && *p && isdigit(*p)) {
			fxval = atof(p);
			idx = (int)((stime + (etime-stime)*percent/100)/0.01);
			if ((0<=idx)&&(idx<fxcnt))
				fx[idx] = (short)fxval;
		}
		p = strtok(NULL," \t\r\n");
	}
}

/* interpolate fx contour */
void interpfx()
{
	int	i,j,ifx=0;
	double	fx1=0,fx2=0,fxslope;

	/* find first fx value */
	for (i=0;i<fxcnt;i++)
		if (fx[i]!=0) {
			fx1=fx[i];
			break;
		}

	/* do interpolations */
	for (i=0;i<fxcnt;i++) {
		if (fx[i]!=0) {
			fx2 = fx[i];
			fxslope = (fx2-fx1)/(i-ifx);
			for (j=ifx;j<=i;j++)
				fx[j] = (short)(fx1 + fxslope*(j-ifx));
			fx1 = fx[i];
			ifx = i;
		}
	}

	/* finish off */
	for (j=ifx;j<fxcnt;j++)
		fx[j] = (short)fx1;
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;			/* option index */
	extern char	*optarg;		/* option argument ptr */
	int		errflg = 0;		/* option error flag */
	int		c;			/* option switch */

	/* processing variables */
	int		i,first;
	FILE		*ip;			/* input text file */
	int		ofid,ofid2;			/* output file descriptors */
	double		tim,l1;
	char		line[512],lab[256];
	int		s1,s2;
	double		totdur;

	/* decode switches */
	while ( (c = getopt(argc,argv,"If:t:shemM")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Load AN item from text V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'f' :	/* select sampling frequency */
			i = atoi(optarg);
			if (i<=0)
				error("bad sampling frequency: %s",optarg);
			else
				fdur = 1.0/i;
			break;
		case 't' :	/* annotation type */
			antype=optarg;
			break;
		case 's' :	/* sample mode */
			esps=0;
			mbrola=0;
			millisec=0;
			sampmode=1;
			break;
		case 'h' :	/* HTK format load */
			esps=0;
			mbrola=0;
			millisec=0;
			sampmode=1;
			fdur = 1.0E-7;
			break;
		case 'e' :	/* ESPS mode */
			esps=1;
			mbrola=0;
			millisec=0;
			sampmode=0;
			break;
		case 'm' :	/* MBROLA mode */
			esps=0;
			mbrola=1;
			millisec=0;
			sampmode=0;
			break;
		case 'M' :	/* millisecond timing */
			esps=0;
			mbrola=0;
			millisec=1;
			sampmode=0;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	/* check command line */
	if (errflg || (argc<3))
		error("usage: %s (-I) (-f freq) (-t type) (-s|-h|-e|-m|-M) textfile file",PROGNAME);

	/* get text file name */
	if (optind < argc)
		strcpy(textname,argv[optind]);
	else
		error("no text file specified",NULL);
	optind++;

	/* get data filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open text file */
	if ((ip=fopen(textname,"r"))==NULL)
		error("cannot open %s",textname);

	/* open data file */
	if ((ofid=sfsopen(filename,"w",NULL)) < 0) {
		if (ofid==-1)
			error("cannot find file %s",filename);
		else
			error("access error on %s",filename);
	}
	sfsclose(ofid);

	/* create output item header */
	sfsheader(&anitem,AN_TYPE,-1,1,-1,fdur,0.0,0,0,1);
	if (antype)
		sprintf(anitem.history,"%s(file=%s,type=%s)",PROGNAME,textname,antype);
	else
		sprintf(anitem.history,"%s(file=%s)",PROGNAME,textname);

	/* open output channel */
	if ((ofid=sfschannel(filename,&anitem)) < 0)
		error("unable to open temporary file",NULL);

	/* get output record buffer */
	if ((an=(struct an_rec *)sfsbuffer(&anitem,1))==NULL)
		error("could not get memory buffer",NULL);

	/* skip any ESPS header */
	if (esps) {
		while (fgets(line,512,ip) != NULL)
			if (line[0]=='#') break;
	}

	/* find overall duration of MBROLA file */
	if (mbrola) {
		totdur=0;
		while (fgets(line,512,ip) != NULL) {
			if (line[0]==';') continue;
			if (sscanf(line,"%s %lf",lab,&l1)<2) break;
			totdur += l1*0.001;
		}
		rewind(ip);
		sfsheader(&fxitem,FX_TYPE,0,2,1,0.01,0.0,1,0,0);
		sprintf(fxitem.history,"%s/FX(file=%s)",PROGNAME,textname);
		fxcnt = (int)(0.5+totdur/0.01);
		fx = (short *)calloc(fxcnt,sizeof(short));
		ofid2 = sfschannel(filename,&fxitem);
	}

	/* process input file line by line */
	first=1;
	tim=0;
	while (fgets(line,512,ip) != NULL) {
		if (esps) {
			if (first)
				an->posn = 0;
			else
				an->posn = (int)(tim/fdur);
			if (sscanf(line,"%lf %d %[^\n]\n",&tim,&s1,lab) < 3) break;
			an->size = (int)(tim/fdur - an->posn);
			strncpy(an->label,lab,MAXANLABEL-1);
			if (sfswrite(ofid,1,an) != 1)
				error("write error on output file",NULL);
		}
		else if (mbrola) {
			if (line[0]==';') continue;
			if (first)
				an->posn = 0;
			else
				an->posn = (int)(tim/fdur);
			if (sscanf(line,"%s %lf",lab,&l1)<2) break;
			an->size = (int)((0.001*l1)/fdur);
			strncpy(an->label,lab,MAXANLABEL-1);
			if (sfswrite(ofid,1,an) != 1)
				error("write error on output file",NULL);
			processfx(line,tim,tim+0.001*l1);
			tim += 0.001*l1;
		}
		else if (sampmode) {
			if (samp1) {
				if (sscanf(line,"%d %[^\n]\n",&s1,lab) < 2)
					break;
			}
			else {
				if (sscanf(line,"%d %d %[^\n]\n",&s1,&s2,lab) < 3) {
					if (first && (sscanf(line,"%d %[^\n]\n",&s1,lab)==2))
						samp1=1;
					else
						break;
				}
			}
			if (!first && (s1 < an->posn))
				error("annotations not in order in '%s'",textname);
			if (samp1) {
				if (!first) {
					an->size = s1 - an->posn;
					if (sfswrite(ofid,1,an) != 1)
						error("write error on output file",NULL);
				}
				an->posn = s1;
				strncpy(an->label,lab,MAXANLABEL-1);
			}
			else {
				an->posn = s1;
				an->size = s2-s1;
				strncpy(an->label,lab,MAXANLABEL-1);
				if (sfswrite(ofid,1,an) != 1)
					error("write error on output file",NULL);
			}
		}
		else if (millisec) {
			if (first)
				an->posn = 0;
			else
				an->posn = (int)(tim/fdur);
			if (sscanf(line,"%lf %[^\n]\n",&l1,lab) < 2) break;
			an->size = (int)(0.001*l1/fdur);
			strncpy(an->label,lab,MAXANLABEL-1);
			if (sfswrite(ofid,1,an) != 1)
				error("write error on output file",NULL);
			tim += 0.001*l1;
		}
		else {
			if (sscanf(line,"%lf %[^\n]\n",&tim,lab) < 2) break;
			if (!first) {
				an->size = (int)(tim/fdur - an->posn);
				if (sfswrite(ofid,1,an) != 1)
					error("write error on output file",NULL);
			}
			an->posn = (int)(tim/fdur);
			strncpy(an->label,lab,MAXANLABEL-1);
		}
		first=0;
	}
	if (!first && ((!sampmode && !esps && !mbrola && !millisec)||(samp1==1))) {
		an->size = 0;
		if (sfswrite(ofid,1,an) != 1)
			error("write error on output file",NULL);
	}
	if (mbrola) {
		interpfx();
		sfswrite(ofid2,fxcnt,fx);
	}

	/* update data file */
	if (!sfsupdate(filename))
		error("backup error on %s",filename);

	/* that's all folks */
	exit(0);

}
