/* Ds - device independent item display for static pictures */

/* M.A.Huckvale - October 1987 */

/* version 1.1
	- full SFS version
*/
/* version 1.2
	- numerous small changes
	- better "unsupported" items
	- get key on exit
*/
/* version 1.3a - May 1988
	- display control file
*/
/* version 1.4 - September 1989
	- portrait mode
	- tracks with TOP HATS
	- bug in item selection with fixed scales
*/
/* version 1.5 - February 1993
	- draw cycling for windowed displays
*/
/* version 1.6 - September 1993
	- allow blank lines and '#' comment lines in control file
*/
/* version 1.7 - November 1995
	- a size of 0 in control file means overlay on previous item
*/
/* version 1.8 - November 1996
	- allow display of spectrograms
*/
/* version 1.9 - October 1998
	- bug fixes on overlay displays
*/

/*--------------------------------------------------------------------------*/
/**MAN
.TH DS SFS1 UCL SFS
.SH NAME
Ds - device independent static data display
.SH SYNOPSIS
.B Ds
(-i item) (-a item) (-d dispfile) (-s start) (-e end) (-p|-P) (-f item lo hi) file
.SH DESCRIPTION
.I Ds
is a program to display portions of data sets in a file using device-independent graphics.
Options allow for the selection of items, the display of part of the timescale and for
direct output to the laser printer.  Default operation is to display all 
items in file.  Use the display control file option to get non-default sizes, 
order or labels.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select first item matching item specification.
.TP 11
.BI -g spitem
Display selected speech item as spectrogram.
.TP 11
.BI -a type
Select all items matching item specification.
.TP 11
.BI -d dispfile
Select items using a display control file, see below.
.TP 11
.BI -s start
Specify a start time (in seconds) other than 0.0.
.TP 11
.BI -e end
Specify an end time (in seconds) other than the length of the first speech 
item in the file.  If no speech item found, default timescale is 1 second.
.TP 11
.BI -f "item lo hi"
Fix all displays of a given type to the y-range lo..hi.
.TP 11
.B -p
Print on laser printer (landscape).
.TP 11
.B -P
Print on laser printer (portrait).
.SH CONTROL FILE
For displays that require special configuration, 
.I Ds
supports the use of an optional display control file as an alternative
to the "-i" and "-a" options.  Each line in the control file refers to one
displayed item on the screen, and the items are displayed vertically in the 
order of specifications in the file.  Each line in the file consists
of an item specification (in the format accepted by the "-i" flag) followed
by a list of display parameter settings.  Each parameter setting consists
of a parameter name, an equals "=", and a parameter value.  For numerical
arguments, the number should follow the "=" with no white space.  For 
string arguments the "=" should be followed immediately by a string in 
double quotes.  The parameter name can be truncated to any degree that
preserves its identity, thus: size=2, siz=2, si=2 or s=2 are equivalent.
.PP
Currently supported parameters are:
.IP size 11
Numerical value indicating relative vertical size of displayed item.
If zero, means overlay on top of immediately previous graph (some
item types only).
Default 5.
.IP low 11
Numerical value indicating the setting for the bottom of the y-axis.  
Only supported by some item types.  Default is data driven. Must be
combined with setting for "high".
.IP high 11
Numerical value indicating the setting for the top of the y-axis.
Only supported by some item types.  Default is data driven. Must be
combined with setting for "low".
.IP title 11
String value indicating item title displayed at upper left of item box.
Default is generated from item label or history.  Use "drawtitle" parameter
to disable.
.IP label 11
String value specifying item label displayed at upper right of item box.
Default is item number.  Use "drawlabel" parameter to disable.
.IP drawbox 11
Numerical value controlling display of a surrounding box. 0=off, 1=on (default).
.IP drawtitle 11
Numerical value controlling display of title string.  0=off, 1=on (default).
.IP drawlabel 11
Numerical value controlling display of label string. 0=off, 1=on (default).
.IP drawgap 11
Numerical value controlling display of gap between items (used for title and/or label if present). 0=off(default), 1=on.
.PP
Thus a display control file might look like:
.nf

     sp.01 size=10
     di^dicode?spectran size=20 low=500 hi=1000
     an. title="Manual Annotations" drawgap=1

.fi
Note that blank lines and lines beginning with '#' are ignored.
.SH INPUT ITEMS
.IP 1.xx 11
Any speech item.  Displayed as waveform.
.IP 2.xx 11
Any excitation item.  Displayed as waveform.
.IP 3.xx 11
Any period data item.  Displayed at period markers.
.IP 4.xx 11
Any fundamental frequency item.  Displayed as fx trace on linear scale.
.IP 5.xx 11
Any annotation item.  Displayed as markers.
.IP 7.xx 11
Any synthesizer control data item.  Displayed in graphical form.
.IP 9.xx 11
Fixed frame Display data in SFS format (use 
.I dicode(UCL1)
to encode CO items).  Displayed as grey scale.
.IP 11.xx 11
Any coefficient data item.  Displayed as spectra.
.IP 12.xx 11
Any raw formant estimates item.  Displayed as numbered points.
.IP 16.xx 11
Any parameter track item.  Displayed as trace.
.PP
Other items are merely noted in the display.
.SH VERSION/AUTHOR
Ds 1.7 - Mark Huckvale
.SH SEE ALSO
dig(SFS3) dig(SFS1)
*/
/*--------------------------------------------------------------------------*/

/* program name and version */
#define	PROGNAME "Ds"
#define PROGVERS "1.9"
char	*progname=PROGNAME;

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>		/* standard i-o routines */
#include <stdlib.h>
#include <unistd.h>
#include <ctype.h>
#include <string.h>
#include <malloc.h>
#include "sfs.h"		/* database filing system structures */
#include "dig.h"
#include "digdata.h"
#include "digitem.h"

/* item selection data */
#define MAXSELECT 40		/* maximum number of selections */
struct selrec {
	int	it;		/* item data type */
	int	ty;		/* item sub type */
	char	*match;		/* history match */
	int	all;		/* all matches reqd */
	int	size;		/* height proportion */
	float	yt,yb;		/* y-axis values */
	int	fix;		/* fix y-axis */
	double	lo,hi;		/* y-axis values */
	int	flags;		/* display flags */
	char	*title;		/* item title */
	char	*label;		/* item label */
	int	dospect;	/* do spectrogram */
} isel[MAXSELECT];
int	iscnt=0;
int	someoverlay=0;

/* default item control table */
int	sizes[MAXDATATYPE+1]={0,3,3,1,5,1,0,7,7,5,0,5,7,0,0,0,5};
int	inum[MAXDATATYPE+1];
int	cumsizes[MAXDATATYPE+1];
struct {
	int	fix;
	double	lo,hi;
} fixtab[MAXDATATYPE+1];

#ifdef __STDC__
void display(int fid,struct item_header *,double yb,double yt,double t1,double t2,int idx);
int itrequired(struct item_header *item,char *hist,int flag,int first);
void getdata(int fid,struct item_header *item,double t1,double t2,char **buff);
int deconline(char *dline,struct selrec	*sel);
#else
void display();
int itrequired();
void getdata();
int deconline();
#endif

/* main program */
int main(argc,argv)
int	argc;
char	*argv[];
{
        /* option decoding variables */
        extern int      optind;
        extern char     *optarg;
        int             errflg=0;
        int             c;
	int		laser = 0;	/* laser print */
	double		starttime= -1.0,endtime= -1.0;
	double		atof();
	int		it;
	char		*ty;
	/* file variables */
        int             dbfil;
        char            filename[SFSMAXFILENAME];
	struct main_header head;
	/* matching variables */
        int             i;
	int		itemsel=0;
	int		idx;
	char		*exphist,*lab_store();
	struct item_header item;
	/* data variables */
	double		tsize,fixfrac=0,varfrac=0,totime=0,yt;
	char		title[256];
	/* display control file */
	char		disfile[SFSMAXFILENAME];
	int		discontrol=0;
	FILE		*df;
	char		disline[256];
	char		devcode;

        /* decode switches */
        while ( (c = getopt(argc,argv,"Ii:g:a:s:e:pPf:d:l:")) != EOF) switch (c) {
                case 'I' :      /* Identify */
                        fprintf(stderr,"%s: Display SFS file V%s\n",PROGNAME,PROGVERS);
                        exit(0);
                        break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&isel[iscnt].it,&isel[iscnt].match) != 0)
				error("bad 'i' option : %s",optarg);
			isel[iscnt].size=sizes[isel[iscnt].it];
			isel[iscnt].fix=0;
			isel[iscnt].flags= DIGITEMBOX | DIGITEMTITLE | DIGITEMLABEL;
			iscnt++;
			if (iscnt >= MAXSELECT)
				error("too many item selections",NULL);
			itemsel++;
			break;
		case 'g' :	/* speech item for spectrogram */
			if (itspec(optarg,&isel[iscnt].it,&isel[iscnt].match) != 0)
				error("bad 'i' option : %s",optarg);
			if (isel[iscnt].it!=SP_TYPE)
				error("spectrograms only of Speech items");
			isel[iscnt].size=2*sizes[isel[iscnt].it];
			isel[iscnt].fix=0;
			isel[iscnt].flags= DIGITEMBOX | DIGITEMTITLE | DIGITEMLABEL;
			isel[iscnt].dospect=1;
			iscnt++;
			if (iscnt >= MAXSELECT)
				error("too many item selections",NULL);
			itemsel++;
			break;
		case 'a' :	/* generic item */
			if (itspec(optarg,&isel[iscnt].it,&isel[iscnt].match) != 0)
				error("bad 'a' option : %s",optarg);
			if (strcmp(isel[iscnt].match,"0")==0)
				isel[iscnt].match = "*";
			isel[iscnt].size=sizes[isel[iscnt].it];
			isel[iscnt].fix = 0;
			isel[iscnt].flags= DIGITEMBOX | DIGITEMTITLE | DIGITEMLABEL;
			isel[iscnt].all=1;
			iscnt++;
			if (iscnt >= MAXSELECT)
				error("too many item selections",NULL);
			itemsel++;
			break;
		case 's' :	/* start time */
			starttime=atof(optarg);
			break;
		case 'e' :	/* end time */
			endtime=atof(optarg);
			break;
		case 'f' :	/* fix scales */
			if (itspec(optarg,&it,&ty) != 0)
				error("bad 'f' option : %s",optarg);
			fixtab[it].fix++;
			fixtab[it].lo=atof(argv[optind++]);
			fixtab[it].hi=atof(argv[optind++]);
			break;
		case 'p' :	/* laser print (landscape) */
			laser=1;
			break;
		case 'P' :	/* laser print (portrait) */
			laser=2;
			break;
		case 'd' :	/* display file */
			strcpy(disfile,optarg);
			discontrol++;
			break;
		case 'l' :	/* dummy flag for compatibility with Es */
			break;
	        case '?' :      /* unknown */
                        errflg++;
        }
        if (errflg || (argc<2))
                error("usage: %s (-I) (-i item) (-a item) (-g spitem) (-d dispfile) (-s start) (-e end) (-f item lo hi) (-p|-P) file",argv[0]);

        /* get filename */
        if (optind < argc)
                strcpy(filename,sfsfile(argv[optind]));
        else
                error("no database file specified",NULL);

        /* open datafile */
        if ((dbfil = sfsopen(filename,"r",&head)) < 0) {
                if (dbfil == -1)
                        error("cannot open %s",filename);
                else
                        error("access error on %s",filename);
        }

	/* process display control file */
	if (discontrol) {
		if ((df=fopen(disfile,"r"))==NULL)
			error("unable to open control file '%s'",disfile);
		while (fgets(disline,256,df) != NULL) if ((disline[0]!='\n') && (disline[0]!='#')) {
			if (deconline(disline,&isel[iscnt]))
				error("format error in display file '%s'",disfile);
			iscnt++;
			if (iscnt >= MAXSELECT)
				error("too many item selections",NULL);
		}
	}

	/* do first pass to resolve item matches */
	while (sfsnextitem(dbfil,&item)) {
		if ((item.datatype==SP_TYPE) && (totime==0.0))
			totime=item.numframes*item.frameduration;
		exphist = lab_store(&item);
		for (idx=0;(idx=itrequired(&item,exphist,0,idx))>=0;idx++) /* loop */;

		if (!discontrol && !itemsel && (item.datatype > 0)) {
			/* display everything mode */
			inum[item.datatype]++;
			cumsizes[item.datatype] += sizes[item.datatype];
			isel[iscnt].it=item.datatype;
			isel[iscnt].ty=item.subtype;
			isel[iscnt].size=sizes[item.datatype];
			isel[iscnt].flags = DIGITEMBOX | DIGITEMTITLE | DIGITEMLABEL;
			iscnt++;
			if (iscnt >= MAXSELECT)
				error("too many items to display in '%s'",filename);
		}
	}

	/* get start and end times */
	if (totime==0.0) totime=1.0;
	if (starttime < 0.0) starttime=0.0;
	if (endtime < 0.0) endtime=totime;

	/* open graphics */
	if (laser==1)
		devcode=DIG_DEFAULT_PRINTER;
	else if (laser==2)
		devcode=DIG_DEFAULT_PRINTER;
	else
		devcode=DIG_DEFAULT_TERM;
	digstart(devcode,NULL,1);

	/* display/redraw loop */
do {
	digscale((float)1.0,(float)1.0,0);

	/* reposition */
	sfsnextitem(dbfil,NULL);

	/* count size data sets to display */
	tsize = 2 * digdata.chheight / digdata.yscale;
	fixfrac = 3*tsize;
	varfrac = 0;
	for (i=itemsel;i<iscnt;i++)
		if (isel[i].size==0)
			inum[isel[i].it]--;
	for (i=1;i<=MAXDATATYPE;i++) {
		fixfrac += inum[i] * tsize;
		varfrac += cumsizes[i];
	}

	if (varfrac==0.0) {
		digend();
		error("nothing to display",NULL);
	}

	/* check amount of data against screen size */
	if (fixfrac > 1.0) {
		digend();
		error("screen/window too small",NULL);
	}

	/* allocate co-ordinates to controlled items */
	yt = 1.0 - 2 * tsize;
	varfrac = (1.0 - fixfrac)/varfrac;
	if (discontrol) {
		for (i=itemsel;i<iscnt;i++) if (isel[i].ty) {
			if ((i>0) && (isel[i].size==0)) {
				isel[i].yt = isel[i-1].yt;
				isel[i].yb = isel[i-1].yb;
			}
			else {
				isel[i].yt=(float)yt;
				yt -= tsize + isel[i].size*varfrac;
				isel[i].yb=(float)yt;
			}
		}
	}
#ifdef IAG
printf("tsize=%.3f fixfrac=%.3f varfrac=%.3f\n",tsize,fixfrac,varfrac);
for (i=0;i<iscnt;i++) printf("%d.%02d yt=%.2f yb=%.2f size=%d flags=%02X\n",isel[i].it,isel[i].ty,isel[i].yt,isel[i].yb,isel[i].size,isel[i].flags);
#endif
	/* display filename, speaker and token */
	digclearscreen();
	sprintf(title,"file=%s speaker=%s token=%s",filename,head.speaker,head.token);
	digtext(20,(float)0.0,(float)(1.0-2*tsize/3),title);

	/* display top timescale */
	digitemtime(20,(float)0.0,(float)(1.0-2*tsize),(float)1.0,(float)(1.0-tsize),starttime,endtime,1);

        /* process data items */
        while (sfsnextitem(dbfil,&item)) {
		exphist = lab_store(&item);
		idx = 0;
		do {
			if ((idx=itrequired(&item,exphist,1,idx)) >= 0) {
				if (isel[idx].yt > 0.0) {
					if (!(isel[idx].flags & DIGITEMOVERLAY))
						display(dbfil,&item,isel[idx].yb,isel[idx].yt,starttime,endtime,idx);
				}
				else {
					display(dbfil,&item,yt-tsize-isel[idx].size*varfrac,yt,starttime,endtime,idx);
					yt -= tsize + isel[idx].size*varfrac;
				}
			}
#ifdef IAG
printf("item %d.%02d idx=%d\n",item.datatype,item.subtype,idx);
#endif
			if (idx >= 0) idx++;
		} while (idx >= 0);
	}

	/* second pass to do overlays */
	if (someoverlay) {
		sfsnextitem(dbfil,NULL);
	        while (sfsnextitem(dbfil,&item)) {
			idx = 0;
			do {
				if ((idx=itrequired(&item,exphist,1,idx)) >= 0) {
					if (isel[idx].yt > 0.0) {
						if (isel[idx].flags & DIGITEMOVERLAY)
							display(dbfil,&item,isel[idx].yb,isel[idx].yt,starttime,endtime,idx);
					}
					else {
						yt -= tsize + isel[idx].size*varfrac;
					}
				}
				if (idx >= 0) idx++;
			} while (idx >= 0);
		}
	}

	/* put up second timescale */
	digitemtime(20,(float)0.0,(float)0.0,(float)1.0,(float)tsize,starttime,endtime,1);

	digflush();

} while (digwait()==DIG_REDRAW);

	/* close file */
	sfsclose(dbfil);

	/* that's all folks */
	digend();
	exit(0);
}

/* display an item */
void display(fid,item,yb,yt,t1,t2,idx)
int		fid;
struct item_header *item;
double		yb,yt;
double		t1,t2;
int		idx;
{
	char	*buff;
	int	flags;
	int32	bundles;
#ifdef IAG
printf("display(%d.%02d,%.3f,%.3f,%.3f,%.3f,%d)\n",
	item->datatype,item->subtype,
	yb,yt,t1,t2,idx);
#endif
	
	/* set up display flags */
	if (isel[idx].flags >= 0)
		flags=isel[idx].flags;
	else
		flags=DIGITEMBOX | DIGITEMTITLE | DIGITEMLABEL;

	/* check if axes need fixing */
	if ((isel[idx].fix==0) && (isel[idx].size==0)) {
		flags |= DIGITEMFIX;
	}
	else if (isel[idx].fix==3) {
		digitab.hi=isel[idx].hi;
		digitab.lo=isel[idx].lo;
		flags |= DIGITEMFIX;
	}
	else if (fixtab[item->datatype].fix) {
		digitab.hi=fixtab[item->datatype].hi;
		digitab.lo=fixtab[item->datatype].lo;
		flags |= DIGITEMFIX;
	}

	/* check if alternate labels */
	digitab.title=isel[idx].title;
	digitab.label=isel[idx].label;

	/* load data from file */
	getdata(fid,item,t1,t2,&buff);
	if (buff==NULL)
		error("could not get buffer for item",NULL);

	/* choose bundles */
	if (flags & DIGITEMOVERLAY)
		bundles = 20202081;
	else
		bundles = 20201626;

	/* display it */
	switch (item->datatype) {
	case SP_TYPE:
			if (isel[idx].dospect)
				digitemFFT(20201626,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(short *)buff,t1,t2,flags);
			else
				digitemSP(bundles,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(short *)buff,t1,t2,flags);
			break;
	case LX_TYPE:
			digitemLX(bundles,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(short *)buff,t1,t2,flags);
			break;
	case TX_TYPE:
			digitemTX(bundles,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(int32 *)buff,t1,t2,flags);
			break;
	case FX_TYPE:
			digitemFX(bundles,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(short *)buff,t1,t2,flags);
			break;
	case AN_TYPE:
			digitemAN(bundles,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(struct an_rec *)buff,t1,t2,flags);
			break;
	case SY_TYPE:
			digitemSY(bundles,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(short *)buff,t1,t2,flags);
			break;
	case WD_TYPE:
			digitemWD(bundles,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(struct wd_rec *)buff,t1,t2,flags);
			break;
	case DI_TYPE:
			digitemDI(232920,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(struct di_rec *)buff,t1,t2,flags);
			break;
	case CO_TYPE:
			digitemCO(bundles,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(struct co_rec *)buff,t1,t2,flags);
			break;
	case FM_TYPE:
			digitemFM(bundles,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(struct fm_rec *)buff,t1,t2,flags);
			break;
	case TR_TYPE:
			digitemTR(bundles,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(float *)buff,t1,t2,flags | DIGITEMTOPHAT);
			break;
	default:
			digitemNULL(bundles,(float)0.0,(float)yb,(float)1.0,(float)yt,item,(void *)buff,t1,t2,flags);
			break;
	}

	/* free data */
	free(buff);
}

/* check if item required */
int itrequired(item,hist,flag,first)
struct item_header 	*item;
char			*hist;
int			flag;
int			first;
{
	int	i,ty;
	char	junk[16];
	
	for (i=first;i<iscnt;i++) if ((isel[i].it==0) || (isel[i].it==item->datatype)) {
		if (isel[i].ty==item->subtype)
			return(i);
		else if (!isel[i].match)
			continue;
		else if (isdigit(isel[i].match[0]) || (isel[i].match[0]=='-')) {
			ty=atoi(isel[i].match);
			if ((ty==-1) || (ty==item->subtype)) {
				if (isel[i].all==0) isel[i].ty=item->subtype;
				inum[abs(item->datatype)]++;
				cumsizes[abs(item->datatype)] += isel[i].size;
				return(i);
			}
			else if ((ty==0) && (flag==0)) {
				if (isel[i].ty==0) {
					inum[abs(item->datatype)]++;
					cumsizes[abs(item->datatype)] += isel[i].size;
				}
				isel[i].it=abs(item->datatype);
				isel[i].ty=item->subtype;
			}
		}
		else if (isel[i].ty==0) {
			if (histmatch(isel[i].match,hist,junk,junk,junk,junk,junk,junk,junk,junk,junk,junk)==1) {
				if (isel[i].all==0) isel[i].ty=item->subtype;
				inum[abs(item->datatype)]++;
				cumsizes[abs(item->datatype)] += isel[i].size;
				return(i);
			}
		}
	}
	return(-1);
}

/* load data from file */
void getdata(fid,item,t1,t2,buff)
int		fid;
struct item_header *item;
double		t1,t2;
char		**buff;
{
	int	start,stop,len;
	char	itemno[12];

	sprintf(itemno,"%d.%02d",item->datatype,item->subtype);

	if ((sfsstruct[item->datatype]==0) && (item->datatype != TX_TYPE)) {
		/* unstructured item */
		/* load part we need */
		start = (t1-item->offset)/item->frameduration - 1;
		stop = 2 + (t2-item->offset)/item->frameduration;
		if (start < 0) start=0;
		if (stop > item->numframes) stop=item->numframes;
		len = stop - start;
		if ((*buff=sfsbuffer(item,len))==NULL)
			error("could not get buffer for item '%s'",itemno);
		if (sfsread(fid,start,len,*buff) != len)
			error("read error on input file",NULL);
		item->offset += start*item->frameduration;
		item->numframes = len;
	}
	else {
		/* structured item */
		/* -- load all of it */
		if ((*buff=sfsbuffer(item,item->numframes))==NULL)
			error("could not get buffer for item '%s'",itemno);
		if (sfsread(fid,0,item->numframes,*buff) != item->numframes)
			error("read error on input file",NULL);
	}
}

/* decode line in display control file */
int deconline(dline,sel)
char	*dline;
struct selrec	*sel;
{
	char	*spec,*comm,*ty,*strtok();
	int	errflg=0,len;
	int	flon=0,floff=0;
	double	atof();

	/* get item spec */
	spec=strtok(dline," \t\n");
	if (itspec(spec,&(sel->it),&ty) != 0) {
		if (spec[0]=='g') {
			if ((itspec(spec+1,&(sel->it),&ty)==0) &&
				(sel->it==SP_TYPE)) {
				/* was a spectrogram */
				sel->dospect=1;
				goto okafterall;
			}
		}
		fprintf(stderr,"%s: bad item specification: '%s'\n",progname,spec);
		errflg++;
	}
okafterall:
	sel->match=malloc(strlen(ty)+1);
	strcpy(sel->match,ty);

	/* reset values */
	sel->size = 5;
	sel->fix  = 0;
	sel->flags= DIGITEMBOX | DIGITEMTITLE | DIGITEMLABEL;

	/* get other commands */
	comm=strtok(0,"= \t\n");
	while (comm && *comm) {
		len=strlen(comm);
		if (strncmp(comm,"size",len)==0) {
			comm=strtok(0," \t\n");
			sel->size=atoi(comm);
			if (sel->size <= 0) {
				flon |= DIGITEMOVERLAY;
				someoverlay=1;
			}
			else
				floff |= DIGITEMOVERLAY;

		}
		else if (strncmp(comm,"low",len)==0) {
			comm=strtok(0," \t\n");
			sel->lo=atof(comm);
			sel->fix |= 1;
		}
		else if (strncmp(comm,"high",len)==0) {
			comm=strtok(0," \t\n");
			sel->hi=atof(comm);
			sel->fix |= 2;
		}
		else if (strncmp(comm,"drawbox",len)==0) {
			comm=strtok(0," \t\n");
			if (atoi(comm))
				flon |= DIGITEMBOX;
			else
				floff |= DIGITEMBOX;
		}
		else if (strncmp(comm,"drawtitle",len)==0) {
			comm=strtok(0," \t\n");
			if (atoi(comm))
				flon |= DIGITEMTITLE;
			else
				floff |= DIGITEMTITLE;
		}
		else if (strncmp(comm,"drawlabel",len)==0) {
			comm=strtok(0," \t\n");
			if (atoi(comm))
				flon |= DIGITEMLABEL;
			else
				floff |= DIGITEMLABEL;
		}
		else if (strncmp(comm,"drawgap",len)==0) {
			comm=strtok(0," \t\n");
			if (atoi(comm))
				flon |= DIGITEMGAP;
			else
				floff |= DIGITEMGAP;
		}
		else if (strncmp(comm,"title",len)==0) {
			if (comm[len+1]=='"') {
				comm[len+1]='#';
				comm=strtok(0,"\"");
				if (comm && *comm) {
					sel->title=malloc(strlen(comm));
					strcpy(sel->title,&(comm[1]));
				}
				else {
					fprintf(stderr,"%s: bad label\n",progname);
					errflg++;
				}
			}
			else {
				fprintf(stderr,"%s: title not in quotes: '%s'\n",progname,&comm[3]);
				errflg++;
			}
		}
		else if (strncmp(comm,"label",len)==0) {
			if (comm[len+1]=='"') {
				comm[len+1]='#';
				comm=strtok(0,"\"");
				if (comm && *comm) {
					sel->label=malloc(strlen(comm));
					strcpy(sel->label,&(comm[1]));
				}
				else {
					fprintf(stderr,"%s: bad label\n",progname);
					errflg++;
				}
			}
			else {
				fprintf(stderr,"%s: label not in quotes: '%s'\n",progname,&comm[3]);
				errflg++;
			}
		}
		else {
			fprintf(stderr,"%s: unknown command '%s'\n",progname,comm);
			errflg++;
		}
		comm=strtok(0,"= \t\n");
	}

	/* set up flags */
	if (!flon && !floff)
		sel->flags = DIGITEMBOX | DIGITEMTITLE | DIGITEMLABEL;
	else
		sel->flags = (sel->flags | flon) & ~floff;

	return(errflg);
}
