/* replay -- replay waveforms from a database file */

/* m.a. huckvale - december 1985 */

/* Version 3.0 - August 1987 
	- SFS version
*/
/* version 4.0 - August 1990
	- start/end options
	- 16bit -> 12 bit conversion option
*/
/* version 5.0 - April 1993
	- support for network replay
	- auto 12/16 bit detection
	- many item replay
	- selection menu
*/
/* version 5.1 - November 1996
	- add stereo replay (for compatible output devices)
*/
/* version 5.2 - May 1998
	- allow stereo replay with unequal channel lengths
*/

/*--------------------------------------------------------------------------*/
/**MAN
.TH REPLAY SFS1 UCL SFS
.SH NAME
replay - replay speech or lx item through D-A
.SH SYNOPSIS
.B replay
(-i item|-a item|-l item) (-r item) (-2) (-s startsec) (-S startsamp) (-e endsec) (-E endsamp) (-12) (-m|-M) file
.SH DESCRIPTION
.I replay
is a program to replay waveforms through SFS supported Digital to Analogue
conversion equipment.  Control over which equipment is supported is defined
in SFSCONFG.h at SFS compile time.  Selection between a number of supported DAC
devices is made by (i) the DAC environment variable and (ii) the SFS terminal to DAC
map file $(SFSBASE)/data/dactable.
.SS
Supported DAC environment variable settings are:
.IP sun	11
SPARC-2 with 8-bit ulaw
.IP sparc2 11
SPARC-2 with 8-bit ulaw
.IP sun8 11
SPARC-2 with 8-bit ulaw
.IP sun8-spkr 11
SPARC-2 with 8-bit ulaw (speaker output)
.IP sun8-phone 11
SPARC-2 with 8-bit ulaw (headphone output)
.IP sun8-line 11
SPARC-2 with 8-bit ulaw (line output)
.IP sparc10 11
SPARC-10 with ISDN 16bit sound interface
.IP sun16	11
SPARC-10 with ISDN 16bit sound interface
.IP dbri	11
SPARC-10 with ISDN 16bit sound interface
.IP sun16-spkr	11
SPARC-10 with ISDN 16bit sound interface (speaker output)
.IP sun16-phone	11
SPARC-10 with ISDN 16bit sound interface (headphone output)
.IP sun16-line	11
SPARC-10 with ISDN 16bit sound interface (line output)
.IP 0..7	11
Masscomp DA08 - channel 0..7
.IP ansi	11
UCL/ANSI stdout transport - 16 bit
.IP ansi16	11
UCL/ANSI stdout transport - 16 bit
.IP ansi12	11
UCL/ANSI stdout transport - 12 bit
.IP pp	11
PC: UCL Parallel printer DAC
.IP dt2811	11
PC: Data Translation DT2811 board
.IP pclx	11
PC: Laryngograph PC/LX board
.IP eb12	11
PC: UCL expansion-bus 12-bit DAC
.IP sb8	11
PC: SoundBlaster 8 bit
.IP sb16	11
PC: SoundBlaster 16 bit
.IP extend	11
Vista Extend, (X-Server/PC) UCL Replay daemon
.IP pipe	11
sends data to $SFSBASE/bin/dacpipe
.IP linux	11
standard LINUX soundcard /dev/dsp
.SS
The default replay size is now 16-bits.  Old 12-bit SP items can be
flagged with the program
.I sp12.
Alternatively, use the '-12' switch to
.I replay
to force all items to 12-bits.
.SS
The Masscomp external set up should be:
.br
	Output signal from DAC channel 0.
.br
	Clock 0 wired to DAC gate.
.br
.SS
Waveforms are loaded into memory before replay.  Default action is to
replay the last speech item in the file.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select input item.  A number of items may be selected,
in which case a menu appears.
.TP 11
.BI -a item
Select all items of a given type.  A menu appears, even if there
is only a single item to replay.
.TP 11
.BI -l item
Select waveform item for left channel of stereo replay.
.TP 11
.BI -r item
Select waveform item for rigth channel of stereo replay.
.TP 11
.B -2
Specify stereo replay - looks for two speech items.
.TP 11
.B -m
Force replay menu.
.TP 11
.B -M
Prevent display of menu.
.TP 11
.BI -s startsec
Specify the start time for replay.  Default 0 seconds.
.TP 11
.BI -S startsamp
Specify the starting sample number for replay.  Default 0.
.TP 11
.BI -e endsec
Specify the end time for replay.  Default: length of item.
.TP 11
.BI -E endsamp
Specify the ending sample for replay.  Default: # samples in item.
.TP 11
.B -12
Input signal is 12 bits, scale if necessary prior to replay.
.SH INPUT ITEMS
.IP SP.xx 11
Any speech item.
.IP LX.xx 11
Any excitation item.
.SH VERSION/AUTHOR
5.0 - Mark Huckvale.
.SH SEE ALSO
inwd(1), syout(1), Es(1), sfsdac(3)
*/
/*--------------------------------------------------------------------------*/

#define PROGNAME "replay"
#define PROGVERS "5.2s"
char *progname=PROGNAME;

/* global structures */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <ctype.h>
#include <malloc.h>
#include "sfs.h"

#define MAX(x,y) (((x)>(y))?(x):(y))

/* item replay management */
#define MAXREPLAY 32
struct sel_rec {
	int32	it;
	char	*ty;
	int	sub;
	short	*sp;
	int	numframes;
	double	srate;
	int	nbits;
	char	*history;
} seltab[MAXREPLAY];
int	selcnt;
int	doall=0;
int	domenu=0;
int	doscale12=0;
int	dostereo=0;
int	startsamp= -1;
int	endsamp= -1;
double	starttime= -1;
double	endtime= -1;

char *strsave(s)
char	*s;
{
	char *p;
	p = malloc(strlen(s)+1);
	strcpy(p,s);
	return(p);
}

/* print menu */
int menu()
{
	int	i;
	char	ans[80];

	do {
		for (i=0;i<selcnt;i++)
			printf("%2d. %d.%02d %s\n",
				i+1,
				seltab[i].it,seltab[i].sub,
				seltab[i].history);
		printf("Enter: ");
		fflush(stdout);
		gets(ans);
		if (!isdigit(ans[0]))
			return(-1);
		else
			i = atoi(ans);
	} while ((i <= 0) || (i > selcnt));
	return(i-1);
}

/* main program */
void main(argc,argv)
int argc;
char *argv[];
{
	/* option decoding */
	extern int	optind;
	extern char	*optarg;
	int		errflg=0;
	char		c;
	int32		it=SP_TYPE;	/* selected item type */
	char		*ty="0";	/* item specification */
	/* file variables */
	char		filename[SFSMAXFILENAME];	/* data file */
	struct item_header item;	/* item header */
	int		i,fid;
	int		offset,numf=0;
	char		itemtxt[32];
	short		*rbuf,*ptr;
	
	/* set up default entries in table */
	seltab[0].it = SP_TYPE;
	seltab[0].ty = "0";
	seltab[1].it = SP_TYPE;
	seltab[1].ty = "0";

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:a:s:S:e:E:1:mMl:r:2")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Waveform replay V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* item specifier */
			if (itspec(optarg,&it,&ty) == 0) {
				if ((it != SP_TYPE) && (it != LX_TYPE)) 
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			if (selcnt == MAXREPLAY)
				error("too many items to replay");
			seltab[selcnt].it = it;
			seltab[selcnt].ty = ty;
			selcnt++;
			if (selcnt > 1) domenu=1;
			break;
		case 'a':	/* all items */
			if (itspec(optarg,&it,&ty) == 0) {
				if ((it != SP_TYPE) && (it != LX_TYPE)) 
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			selcnt=0;
			doall=1;
			domenu=1;
			break;
		case 'l' :	/* item specifier */
			if (itspec(optarg,&it,&ty) == 0) {
				if ((it != SP_TYPE) && (it != LX_TYPE)) 
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			if (selcnt == MAXREPLAY)
				error("too many items to replay");
			seltab[0].it = it;
			seltab[0].ty = ty;
			selcnt=2;
			dostereo=1;
			break;
		case 'r' :	/* item specifier */
			if (itspec(optarg,&it,&ty) == 0) {
				if ((it != SP_TYPE) && (it != LX_TYPE)) 
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			if (selcnt == MAXREPLAY)
				error("too many items to replay");
			seltab[1].it = it;
			seltab[1].ty = ty;
			selcnt=2;
			dostereo=1;
			break;
		case '2' :	/* generic stereo */
			seltab[0].it = SP_TYPE;
			seltab[0].ty = "*";
			seltab[1].it = SP_TYPE;
			seltab[1].ty = "0";
			selcnt=2;
			dostereo=1;
			break;
		case 's' :	/* start time */
			starttime = atof(optarg);
			break;
		case 'S' :	/* start sample */
			startsamp = atoi(optarg);
			break;
		case 'e' :	/* end time */
			endtime = atof(optarg);
			break;
		case 'E' :	/* end sample */
			endsamp = atoi(optarg);
			break;
		case '1' :	/* 16 bits */
			if (*optarg=='2')
				doscale12++;
			else
				error("bad '-1' option, -12 intended?");
			break;
		case 'm':	/* force menu */
			domenu=1;
			break;
		case 'M':	/* no menu */
			domenu=0;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: replay (-I) (-i item|-a item|-l item) (-r item) (-2) (-s startsec|-S startsamp) (-e endsec|-E endsamp) (-12) (-m|-M) file",NULL);

	/* get filename */
	if (optind < argc) 
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open file */
	if ((fid=sfsopen(filename,"r",NULL)) < 0)
		error("access error on '%s'",filename);

	/* find items */
	if (doall) {
		while (sfsnextitem(fid,&item) && (selcnt < MAXREPLAY)) {
			if (item.datatype==it) {
				seltab[selcnt].it = it;
				sprintf(itemtxt,"%d",item.subtype);
				seltab[selcnt].ty = strsave(itemtxt);
				selcnt++;
			}
		}
	}

	/* check stereo OK */
	if (dostereo && (selcnt!=2))
		error("need 2 items for stereo replay");

	/* load data */
	if (selcnt==0) selcnt=1;	/* default mode */
	for (i=0;i<selcnt;i++) {
		if (sfsitem(fid,seltab[i].it,seltab[i].ty,&item)==0)
			error("could not find input item in '%s'",filename);
		seltab[i].sub = item.subtype;
		seltab[i].srate = 1.0/item.frameduration;
		seltab[i].history = strsave(item.history);

		/* get start and end samples */
		offset = 0;
		if (starttime >= 0)
			offset = (int)((starttime-item.offset)/item.frameduration);
		if (endtime >= 0)
			numf = (int)((endtime-item.offset)/item.frameduration - offset);
		else if (endsamp < 0)
			numf = item.numframes - offset;
		offset = (startsamp < 0) ? offset : startsamp;
		offset = (offset > item.numframes) ? item.numframes : offset;
		numf = (endsamp < 0) ? numf : endsamp-offset;
		numf = ((offset+numf) > item.numframes) ? item.numframes-offset : numf;
		seltab[i].numframes = numf;

		/* get buffer */
		if ((seltab[i].sp=(short *)sfsbuffer(&item,numf))==NULL)
			error("could not get memory buffer for item");
			/* read in item into memory */
		if (sfsread(fid,offset,numf,seltab[i].sp)!=numf)
			error("read error on input item");
	
		/* check for 12-bit data */
		seltab[i].nbits = (int)param(item.params,"bits",(doscale12)?12.0:16.0);
	}

	/* open DAC */
	if (dac_open(NULL) < 0)
		error("unable to open DAC device");

	if (domenu) {
		/* wants a menu */
		while ((i=menu())>= 0)
			dac_playback(seltab[i].sp,
					seltab[i].numframes,
					seltab[i].srate,
					seltab[i].nbits,
					1,1);
	}
	else if (dostereo) {
		numf = MAX(seltab[0].numframes,seltab[1].numframes);
		if ((rbuf=(short *)calloc(2*numf,sizeof(short)))==NULL)
			error("could not get replay buffer");
		ptr = rbuf;
		for (i=0;i<seltab[0].numframes;i++) {
			*ptr++ = seltab[0].sp[i];
			ptr++;
		}
		ptr = rbuf;
		for (i=0;i<seltab[1].numframes;i++) {
			ptr++;
			*ptr++ = seltab[1].sp[i];
		}
		dac_playback(rbuf,2*numf,seltab[0].srate,seltab[0].nbits,2,1);
	}
	else if (selcnt > 1) {
		/* wants them all replayed */
		for (i=0;i<selcnt;i++) {
			printf("%2d. %d.%02d %s\n",
				i+1,
				seltab[i].it,seltab[i].sub,
				seltab[i].history);
			dac_playback(seltab[i].sp,
					seltab[i].numframes,
					seltab[i].srate,
					seltab[i].nbits,
					1,1);
		}
	}
	else {
		/* just the one */
		dac_playback(seltab[0].sp,
				seltab[0].numframes,
				seltab[0].srate,
				seltab[0].nbits,
				1,1);
	}
		
	/* that's all folks */
	dac_close(0);
	sfsclose(fid);
	exit(0);
}


