/* scopy -- copy an individual data item within or across files */

/* M.A.Huckvale - April 1986 */
/* Vs 1.1 - Amended by a hot and sweaty Mike Johnson - June 86,
          to copy between files */
/* vs 1.2 - allow change of subtype when copying across files */
/* vs 3.0 - copy history across into target history */
/* Version 3.1 - August 1987
	- SFS version
*/
/* version 3.3 - January 2001
	- allow copying of multiple items at once
*/

/*--------------------------------------------------------------------------*/
/**MAN
.TH SCOPY SFS1 UCL SFS
.SH NAME
scopy - duplicate or replicate a data item
.SH SYNOPSIS
.B scopy
{-i item}  file (file2)
.SH DESCRIPTION
.I scopy
is a program to copy one or more selected data sets within a file or across files.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version number.
.TP 11
.BI -i item
Select input item to copy.
.SH VERSION/AUTHOR
3.3 - Mark Huckvale and Mike Johnson
.SH SEE ALSO
remove(SFS1), slink(SFS1)
*/
/*--------------------------------------------------------------------------*/

#define PROGNAME "scopy"
#define PROGVERS "3.3"
char *progname=PROGNAME;

/* main transfer buffer size */
#define BUFSIZE 8192

/* global structures */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "sfs.h"

struct item_sel_rec {
	int	it;
	char	*ty;
} itlist[MAXITEM];
int itcnt;

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it= -1;		/* item data type */
	char		*ty="0";	/* item sub-type specifier */
	int		samefile = 0;	/* 1/2 file flag */
	/* file variables */
	char		filename[SFSMAXFILENAME];	/* dbase file name */
	char		opfilename[SFSMAXFILENAME];	/* output file name */
	int		fid,ofid=-1;	/* file descriptors */
	/* data variables */
	struct item_header item,newitem;
	char		*buff;
	int		start,nframe,cnt;
	int		i;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Copy SFS item V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific input item */
			if (itspec(optarg,&it,&ty) != 0)
				error("illegal item specifier %s",optarg);
			itlist[itcnt].it=it;
			itlist[itcnt].ty=ty;
			itcnt++;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	/* check for command line errors */
	if (errflg || (argc<2))
		error("usage: %s (-I) {-i item} file (file2)",PROGNAME);

	/* get filenames */
	if (optind < argc) {
		strcpy(filename,sfsfile(argv[optind]));
		if (argc == optind + 1) {
			/* one file specified */
			samefile++;
			strcpy(opfilename,filename);
		}
		else
			/* two files specified */
			strcpy(opfilename,sfsfile(argv[optind+1]));
	}
	else
		error("no database files specified",NULL);

	/* check input item and type specified */
	if (it < 0)
		error("input item not specified",NULL);

	/* open input file */
        if ((fid = sfsopen(filename,"r",NULL)) < 0) {
                if (fid == -1)
                        error("cannot open %s",filename);
                else
                        error("access error on %s",filename);
        }

	/* check output file */
        if (!samefile && ((ofid = sfsopen(opfilename,"w",NULL)) < 0)) {
                if (ofid == -1)
                        error("cannot open %s",opfilename);
                else
                        error("access error on %s",opfilename);
        }
	sfsclose(ofid);

	for (i=0;i<itcnt;i++) {
		it = itlist[i].it;
		ty = itlist[i].ty;

		/* locate input item */
		if (!sfsitem(fid,it,ty,&item))
			error("could not find input item in %s",filename);

		/* create output item header */
		sfsheader(&newitem,
			item.datatype,
			item.floating,
			item.datasize,
			item.framesize,
			item.frameduration,
			item.offset,
			item.windowsize,
			item.overlap,
			item.lxsync);
		if (samefile)
			sprintf(newitem.history,"%s(%d.%02d)",
				PROGNAME,item.datatype,item.subtype);
		else {
			if (strlen(item.history)>200) item.history[200]='\0';
			sprintf(newitem.history,"%s(file=%s,item=%d.%02d,history=%s)",
				PROGNAME,filename,
				item.datatype,item.subtype,item.history);
		}
		strcpy(newitem.params,"");

		/* open channel to add item to new file */
		if ((ofid=sfschannel(opfilename,&newitem)) < 0)
			error("could not open temporary file",NULL);
	
		/* create buffer for data transfer */
		if (item.framesize > 0)
			nframe = BUFSIZE / (item.datasize*item.framesize) + 1;
		else
			nframe = 10;
		if ((buff = sfsbuffer(&item,nframe)) == NULL)
			error("could not get memory buffer",NULL);
	
		/* copy data */
		start=0;
		while ((cnt=sfsread(fid,start,nframe,buff)) > 0) {
			if (sfswrite(ofid,cnt,buff)!=cnt)
				error("write error on output file");
			start += cnt;
		}

		/* free buffer */
		free(buff);

		if ((i%10)==9) {
			/* update output file periodically */
			if (!sfsupdate(opfilename))
				error("backup error on %s",opfilename);
		}
	}

	/* close input file */
	sfsclose(fid);

	/* update output file */
	if (!sfsupdate(opfilename))
		error("backup error on %s",opfilename);

	/* that's all folks */
	exit(0);
}
