/* slink -- link primary data to secondary sfs file */

/* m.a. huckvale - august 1987 */

/* version 1 */
/* version 1.1 - November 1987
	- accept single file
	- maintain offset field in single files
*/
/* version 1.2 - January 1988
	- accept networked file with UCL PC file server
*/
/* version 2.0 - July 1988
	- option to link whole SFS files to new file
*/
/* version 2.1 - October 1988
	- don't link item stubs
	- ensure absolute pathnames
*/
/* version 3.0 - June 1994
	- changed name from 'link'
	- added support for 8-bit waveforms
	- added 'named' file formats
	- added choice of relative pathnames
*/
/* version 3.1 - September 1994
	- add -h headerlen
*/
/* version 3.2 - March 1995
	- fix fault in linking to linked + multiplexed SFS files
*/
/* version 3.3 - June 1995
	- add -r relative pathname switch
*/
/* version 3.4 - April 1996
	- fix bug in reading 8-bit VOC files
*/
/* version 3.5 - March 1997
	- fix absolute links to relatively linked items
*/
/* version 3.6 - February 1998
	- allow for new format WAV/RIFF format files
	- add support for Laryngograph TX files
*/
/* version 3.7 - January 2001
	- allow linking of multiple SFS items at once
*/
/* version 3.8 - April 2001
	- make two links for stereo files
*/

/*--------------------------------------------------------------------------*/
/**MAN
.TH SLINK SFS1 UCL SFS
.SH NAME
slink - link primary speech data into secondary SFS file
.SH SYNOPSIS
.B slink
sfs_source newfile
.br
.B slink
-i item (-s|-S start) (-e|-E end) (-r) sfs_source (destfile)
.SS
.B slink
-i item -f freq (-c chan/#chan) (-s|-S start) (-e|-E end)
(-b) (-d DC) (-m mult) (-h headerlen) (-r) binary_source destfile
.SS
.B slink
-i item -t filetype (-c chan/#chan|-C chantype) (-s|-S start) (-e|-E end)
(-r) formatted_source destfile
.SH DESCRIPTION
.I slink
is a program to create one or more "virtual items" in a SFS file
that link back
to some primary data in another file.  The data can be accessed through
the SFS file as normal, but the primary data cannot be affected by any
operation.  The program operates in three modes:
.IP 1. 11
Link whole contents of existing SFS file to new SFS file.
.IP 2. 11
Link single item in existing SFS file to existing SFS file.
.IP 3. 11
Link part of a foreign format (including binary) file to an item in an existing SFS file.
.PP
The program takes two files, the second of which ("destfile")
must be a SFS file.  In the first mode, "destfile" must not already exist.
The primary file ("xxx_source") can one of the following:
.PP
.nf
	SFS file
	binary file, single channel/multiple channels
	RIFF format file (.WAV), single/multiple channels
	VOC format file, single/multiple channels
	AU format file, single/multiple channels
	AIFF format file, single/multiple channels
	ILS format file, single channel
	HTK format file, single channel
	PCLX Tx data file, for access as SFS Tx item
.fi
Files on the host machine are specified with a filename.  Files
accessed through a file server running the UCL protocol are specified
with a host name ("networkname") and a filename.
.PP
For SP and LX items in SFS or binary files,
a part of the primary data set may be linked, using
specified start and end times (in seconds or samples).
.PP
Binary data may be transformed by (i) selecting channels, (ii) swapping bytes,
(iii) removing a DC offset and
(iv) shifting bit pattern up/down word.
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
(sfs source) Select a source item number.  Each item is linked.
.br
(binary source) Specify datatype.  Single item is linked.
.TP 11
.BI -f freq
(binary source)Specify sampling frequency in Hz.
.TP 11
.BI -t filetype
Specify file format. Options are: RIFF, WAV, VOC, AU, AIFF, ILS, HTK, PCLX.
Default: RAW (straight binary).
.TP 11
.BI -c chan/#chan
Specify channel number and number of channels for
multiplexed data.  For example: two-channel data acquired and
first channel required: "-c 1/2". Default is "-c 1/1".
.TP 11
.BI -C chantype
Specify speech and/or Lx for stereo linkage.  Use -C11 for stereo
speech, -C12 for speech and Lx, -C21 for Lx and speech, -C22 for stereo Lx.
.TP 11
.BI -s start
Specify start time for linked item (in seconds).
.TP 11
.BI -S start
Specify start time for linked item (in samples).
.TP 11
.BI -e end
Specify end time for linked item (in seconds).
.TP 11
.BI -E end
Specify end time for linked item (in samples).
.TP 11
.B -b
(binary source)Swap bytes in sampled data.
.TP 11
.BI -d DC
(binary source)Subtract value given by "DC" from samples. Performed after byte swapping.
.TP 11
.BI -m mult
(binary source)Shift sampled bit pattern up (positive shift) or down (negative shift)
by the number of bits specified in "mult".  Performed after byte swapping and DC offset correction.
Values of shift greater than 15 are reserved for manipulation of 8-bit values internally.
.TP 11
.BI -h headerlen
(binary source)Skip 'headerlen' bytes before starting the link.  Useful
for skipping fixed length headers prefixed onto raw binary - providing
you know how long they are.
.TP 11
.B -r
Use a relative path to the source file.  Default: absolute path.
.SH VERSION/AUTHOR
3.6 - Mark Huckvale
.SH BUGS
Item offsets are not preserved in links to individual items across files.
*/
/*--------------------------------------------------------------------------*/

#define PROGNAME "slink"
#define PROGVERS "3.8"
char *progname=PROGNAME;

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <string.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#ifdef DOS
#include <io.h>
#endif
#include <math.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include "sfs.h"
#include "sfsdata.h"

typedef unsigned int uint32;

/* byte reversal macros */
#define US(s) ((unsigned short)s)
#define UL(l) ((uint32)l)
#define REVERSESHORT(s) s = ((US(s) & 0xFF00)>>8)|((US(s) & 0x00FF)<<8)
#define REVERSELONG(l) l = ((UL(l) & 0xFF000000)>>24)|((UL(l)&0x00FF0000)>>8)|((UL(l)&0x0000FF00)<<8)|((UL(l)&0x000000FF)<<24)

/* format lookup */
int	binproc(),riffproc(),vocproc(),auproc();
int	aiffproc(),ilsproc(),htkproc(),pclxproc();
struct format_rec {
	char	*name;
	int	(*fproc)();
} formattab[]={
	{ "RAW",	binproc },
	{ "RIFF",	riffproc },
	{ "WAV",	riffproc },
	{ "VOC",	vocproc },
	{ "AU",		auproc },
	{ "AIFF",	aiffproc },
	{ "ILS",	ilsproc },
	{ "HTK",	htkproc },
	{ "PCLX",	pclxproc },
};
#define NUMFORMAT (sizeof(formattab)/sizeof(struct format_rec))

/* global data */
char            	srcfilename[SFSMAXFILENAME];
char            	dstfilename[SFSMAXFILENAME];
char            	relfilename[SFSMAXFILENAME];
char			absfilename[SFSMAXFILENAME];
char			tmpfilename[SFSMAXFILENAME];
struct main_header	srchead;
struct item_header 	item,lnitem;
struct link_header 	ilink,lnlink;
static double		btime= -1, etime= -1;
int			ssamp= -1, esamp= -1;
double			sfreq= -1;
char			*netname=NULL;
static int		swapbyte=0;
int			dcoffset=0;
int			shift=0;
int			numf;
int			chan=1,nchan=0;
int			format=0;
int32			headerlen=0;
int			dorelpath=0;
int			dostereo=0;

struct item_sel_rec {
	int	it;
	char	*ty;
} itlist[MAXITEM];
int itcnt;

#ifdef __STDC__
void relatepath(char *,char *,char *);
void absolutepath(char *,char *,char *);
#else
void relatepath();
void absolutepath();
#endif

/* main program */
void main(argc,argv)
int argc;
char *argv[];
{
        /* option decoding variables */
        extern int      optind;
        extern char     *optarg;
        int             errflg=0;
        char            c;
	/* file variables */
        int             fil1= -1,fil2;
	/* matching variables */
	int		it= -1;
	char		*ty;
	/* misc. */
	int		i,lncount;
	double		atof();
	struct stat	filestat;
	int		unstruct=0;
	int		ismultiple=1;	/* multiple xfer mode */
	int		isbinary=0;	/* binary xfer mode */
	int		isnetwork=0;	/* network binary xfer */
	int		samefile=0;	/* single file mode */
	int32		binoffset=0;

        /* decode switches */
        while ( (c = getopt(argc,argv,"Ii:s:S:e:E:f:c:n:d:m:bt:h:rC:")) != EOF) switch (c) {
                case 'I' :      /* Identify */
                        fprintf(stderr,"%s: Link data set to SFS item V%s\n",PROGNAME,PROGVERS);
                        exit(0);
                        break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) != 0)
				error("bad 'i' option : %s",optarg);
			itlist[itcnt].it=it;
			itlist[itcnt].ty=ty;
			itcnt++;
			ismultiple=0;
			break;
		case 's':	/* start time */
			btime = atof(optarg);
			ismultiple=0;
			break;
		case 'e':	/* end time */
			etime = atof(optarg);
			ismultiple=0;
			break;
		case 'S':	/* start samp */
			ssamp = atoi(optarg);
			ismultiple=0;
			break;
		case 'E':	/* end samp */
			esamp = atoi(optarg);
			ismultiple=0;
			break;
		case 'f' :	/* sampling freq */
			sfreq = atof(optarg);
			isbinary++;
			ismultiple=0;
			break;
		case 'c' :	/* channel info */
			if (sscanf(optarg,"%d/%d",&chan,&nchan)!=2)
				error("bad channel specification: %s",optarg);
			isbinary++;
			ismultiple=0;
			break;
		case 'C' :	/* stereo linkage */
			dostereo = atoi(optarg);
			it=SP_TYPE;
			break;
		case 'n' :	/* network name */
			netname = optarg;
			isnetwork++;
			isbinary++;
			ismultiple=0;
			break;
		case 'b' :	/* swap bytes */
			swapbyte++;
			isbinary++;
			ismultiple=0;
			break;
		case 'd' :	/* DC offset */
			dcoffset=atoi(optarg);
			isbinary++;
			ismultiple=0;
			break;
		case 'm' :	/* binary shift */
			shift = atoi(optarg);
			isbinary++;
			ismultiple=0;
			break;
		case 'h' :	/* header length */
			headerlen = atol(optarg);
			isbinary++;
			ismultiple=0;
			break;
		case 't' :	/* file type */
			for (format=0;format<NUMFORMAT;format++)
				if (strcmp(optarg,formattab[format].name)==0)
					break;
			if (format >= NUMFORMAT) {
				fprintf(stderr,"known file format types are:");
				for (i=0;i<NUMFORMAT;i++)
					fprintf(stderr," %s",formattab[i].name);
				fprintf(stderr,"\n");
				error("unknown file format type '%s'",optarg);
			}
			isbinary++;
			ismultiple=0;
			break;
		case 'r' :	/* do relative path for link */
			dorelpath++;
			break;
		default :      /* unknown */
                        errflg++;
        }
        if (errflg || (argc<2))
                error("usage: %s (-I) (-i item) (-f freq) (-t filetype) (-c chan/#chan|-C chantype) (-s|-S start) (-e|-E end) (-n networkname) (-b) (-d DC) (-m mult) (-h headerlen) srcfile (dstfile)",PROGNAME);

        /* get srcfilename */
        if (optind < argc) {
        	strcpy(absfilename,pathname(sfsfile(argv[optind])));
        	if (dorelpath)
			strcpy(srcfilename,argv[optind]);
		else
			strcpy(srcfilename,absfilename);
        }
        else
                error("no primary file specified",NULL);
	optind++;

        /* get dstfilename */
        if (optind < argc)
                strcpy(dstfilename,argv[optind]);
        else {
		strcpy(dstfilename,srcfilename);
		samefile++;
	}

	/* sort out relative file positions */
	if (dorelpath)
		relatepath(relfilename,srcfilename,dstfilename);
	else
		strcpy(relfilename,srcfilename);

        /* open source datafile */
        if (!isbinary && ((fil1 = sfsopen(srcfilename,"r",&srchead)) < 0)) {
                if (fil1 == -1)
                        error("cannot open %s",srcfilename);
                else {
			isbinary++;
			ismultiple=0;
		}
        }

	/* check input item specified */
	if (isbinary || !ismultiple) {
		if (it < 0)
			error("no input item specified",NULL);
		if ((it==SP_TYPE) || (it==LX_TYPE) || (it==TX_TYPE))
			unstruct++;
		if (isbinary && !unstruct)
			error("only binary SP and LX supported",NULL);

		/* open output SFS file */
	        if ((fil2 = sfsopen(dstfilename,"w",NULL)) < 0) {
	                if (fil2 == -1)
	                        error("cannot open %s",dstfilename);
	                else
	                        error("access error on %s",dstfilename);
	        }
		sfsclose(fil2);
	}
	else /* multiple copy mode */ {
		/* output file must not exist */
		if (sfsopen(dstfilename,"r",NULL) != -1)
			error("destination file exists: '%s'",dstfilename);
		/* create new output file */
		if ((fil2=sfsopen(dstfilename,"c",&srchead)) < 0)
			error("unable to create '%s'",dstfilename);
		sfsclose(fil2);
	}

	/* perform linking */
	if (ismultiple) {
		lncount=0;
		while (sfsnextitem(fil1,&item)) if (item.datatype > 0) {

			/* take copy of item header */
			sfsheader(&lnitem,item.datatype,item.floating,item.datasize,item.framesize,item.frameduration,item.offset,item.windowsize,item.overlap,item.lxsync);

			/* do item history */
			sprintf(lnitem.history,"%s(file=%s,item=%d.%02d,history=%s)",
				PROGNAME,
				srcfilename,
				item.datatype,
				item.subtype,
				item.history);
			strcpy(lnitem.params,item.params);

			/* check input item not linked itself ! */
			if (item.datapresent==2) {

				/* get linked link */
				lseek(fil1,sfsdata[fil1]->datastart,0);
				read(fil1,&ilink,sizeof(struct link_header));
				sfsdata[fil1]->currpos=lseek(fil1,0L,1);

				/* take copy of link header */
				lnlink = ilink;
				if ((ilink.filename[0]=='/') || (ilink.filename[0]=='\\')
#ifdef WIN32
					|| (isalpha(ilink.filename[0]) && (ilink.filename[1]==':'))
#endif
				   ) {
					/* OK the link points to an absolute file
						- just use that as is */
				}
				else {
					/* argh! the source file is relatively linked */
					if (dorelpath) {
						absolutepath(tmpfilename,absfilename,ilink.filename);
						relatepath(lnlink.filename,tmpfilename,dstfilename);
					}
					else
						absolutepath(lnlink.filename,absfilename,ilink.filename);
				}

			}
			else {
				/* build link header */
				memset(&lnlink,0,sizeof(struct link_header));
				strcpy(lnlink.filename,relfilename);
				lnlink.filetype = SFS_TYPE;
				lnlink.datatype = item.datatype;
				lnlink.subtype = item.subtype;
				lnlink.linkdate = item.processdate;
				lnlink.machine = item.machine;
			}

			/* write link to file */
			if (!sfswritelink(&lnitem,item.numframes,&lnlink,dstfilename))
				error("write failed on temporary file",NULL);
			lncount++;

			/* update file periodically */
			if ((lncount%10)==9)
				if (!sfsupdate(dstfilename))
					error("update error on '%s'",dstfilename);
		}
	}
	else if (!isbinary) {
	    /* SFS items -- find them */
	    for (i=0;i<itcnt;i++) {
	    	it = itlist[i].it;
	    	ty = itlist[i].ty;

		if (sfsitem(fil1,it,ty,&item) == 0)
			error("cannot find input item in %s",srcfilename);

		sfreq = 1.0/item.frameduration;
		numf = item.numframes;

		/* convert start and end positions to samples */
		if (btime > 0)
			ssamp = (int)(btime*sfreq);
		if ((ssamp < 0) || !unstruct)
			ssamp = 0;
		if (etime > 0)
			esamp = (int)(etime*sfreq);
		if ((esamp < ssamp) || (esamp > numf) || !unstruct)
			esamp = numf;

		/* create new item header */
		sfsheader(&lnitem,item.datatype,
			item.floating,
			item.datasize,
			item.framesize,
			item.frameduration,
			(samefile)?item.offset+ssamp*item.frameduration:0.0,
			item.windowsize,
			item.overlap,
			item.lxsync);
		if (samefile)
			sprintf(lnitem.history,"%s(%d.%02d;start=%d,end=%d)",
				PROGNAME,
				item.datatype,
				item.subtype,
				ssamp,esamp);
		else {
			if (strlen(item.history)>200) item.history[200]='\0';
			sprintf(lnitem.history,"%s(file=%s,item=%d.%02d,start=%d,end=%d,history=%s)",
				PROGNAME,
				srcfilename,
				item.datatype,
				item.subtype,
				ssamp,esamp,
				item.history);
		}
		strcpy(lnitem.params,item.params);

		/* check input item not linked itself ! */
		if (item.datapresent==2) {
			/* get linked link */
			lseek(fil1,sfsdata[fil1]->datastart,0);
			read(fil1,&ilink,sizeof(struct link_header));
			sfsdata[fil1]->currpos=lseek(fil1,0L,1);

			/* take copy of link header */
			lnlink = ilink;
			if ((ilink.filename[0]=='/') || (ilink.filename[0]=='\\')
#ifdef WIN32
				|| (isalpha(ilink.filename[0]) && (ilink.filename[1]==':'))
#endif
			   ) {
				/* OK the link points to an absolute file
					- just use that as is */
			}
			else {
				/* argh! the source file is relatively linked */
				if (dorelpath) {
					absolutepath(tmpfilename,absfilename,ilink.filename);
					relatepath(lnlink.filename,tmpfilename,dstfilename);
				}
				else
					absolutepath(lnlink.filename,absfilename,ilink.filename);
			}
			lnlink.offset += ssamp*item.datasize*item.framesize*(ilink.multiplex+1);

		}
		else {
			/* build link header */
			memset(&lnlink,0,sizeof(struct link_header));
			strcpy(lnlink.filename,relfilename);
			lnlink.filetype = SFS_TYPE;
			lnlink.datatype = item.datatype;
			lnlink.subtype = item.subtype;
			lnlink.offset = ssamp*item.datasize*item.framesize;
			lnlink.linkdate = item.processdate;
			lnlink.machine = item.machine;
		}

		/* write link to file */
		if (!sfswritelink(&lnitem,esamp-ssamp,&lnlink,dstfilename))
			error("write failed on temporary file",NULL);

		if ((i%10)==9) {
			/* update output file peridoically */
			if (!sfsupdate(dstfilename))
				error("update error on '%s'",dstfilename);
		}
	    }
	}
	else /* binary file */ {

		/* get size of binary file */
		if (isnetwork) {
			int32 size,time;
			netaccess(netname,srcfilename,&size,&time);
			filestat.st_size=size;
			filestat.st_mtime=time;
		}
		else {
			if (access(srcfilename,4) < 0)
				error("cannot find binary file: %s",srcfilename);
			stat(srcfilename,&filestat);
		}

		/* get parameters from foreign file type */
		if ((*formattab[format].fproc)(srcfilename,filestat.st_size,
				&nchan,&swapbyte,&dcoffset,&shift,&sfreq,
				&binoffset,&numf)!=0)
			error("could not process '%s'",srcfilename);

		/* convert start and end positions to samples */
		if (btime > 0)
			ssamp = (int)(btime*sfreq);
		if ((ssamp < 0) || !unstruct)
			ssamp = 0;
		if (etime > 0)
			esamp = (int)(etime*sfreq);
		if ((esamp < ssamp) || (esamp > numf) || !unstruct)
			esamp = numf;
		if (nchan==0) nchan=1;

		/* special operation for Stereo */
		if ((nchan==2)&&(dostereo!=0)) {
			/* create first item header */
			chan=1;
			it = dostereo/10;
			if ((it!=1)&&(it!=2)) it=1;
			sfsheader(&lnitem,it,0,2,1,1.0/sfreq,0.0,1,0,0);
			sprintf(lnitem.history,"%s(file=%s,headerlen=%d,start=%d,end=%d,freq=%g,channels=1/2%s,dc=%d,mult=%d)",
				PROGNAME,
				srcfilename,
				binoffset,
				ssamp,esamp,
				sfreq,
				(swapbyte)?",swapped":"",
				dcoffset,shift);

			/* create link header */
			memset(&lnlink,0,sizeof(struct link_header));
			strcpy(lnlink.filename,relfilename);
			strcpy(lnlink.filepath,"");
			lnlink.filetype |= BINARY_TYPE;
			lnlink.swab=swapbyte;
			lnlink.dcoffset=dcoffset;
			lnlink.shift=shift;
			if (shift < 16)
			 	lnlink.offset = binoffset + ssamp * nchan * lnitem.datasize * lnitem.framesize +
					(chan - 1)*2;
			else
			 	lnlink.offset = binoffset + ssamp * nchan * lnitem.framesize +
					(chan - 1);

			lnlink.multiplex=nchan-1;
			lnlink.linkdate = filestat.st_mtime;
			lnlink.machine = SFSMACHINE;

			/* add linked item to destination file */
			if (!sfswritelink(&lnitem,esamp-ssamp,&lnlink,dstfilename))
				error("write failed on temporary file",NULL);

			/* create second item header */
			chan=2;
			it = dostereo%10;
			if ((it!=1)&&(it!=2)) it=1;
			sfsheader(&lnitem,it,0,2,1,1.0/sfreq,0.0,1,0,0);
			sprintf(lnitem.history,"%s(file=%s,headerlen=%d,start=%d,end=%d,freq=%g,channels=2/2%s,dc=%d,mult=%d)",
				PROGNAME,
				srcfilename,
				binoffset,
				ssamp,esamp,
				sfreq,
				(swapbyte)?",swapped":"",
				dcoffset,shift);

			/* create link header */
			memset(&lnlink,0,sizeof(struct link_header));
			strcpy(lnlink.filename,relfilename);
			strcpy(lnlink.filepath,"");
			lnlink.filetype |= BINARY_TYPE;
			lnlink.swab=swapbyte;
			lnlink.dcoffset=dcoffset;
			lnlink.shift=shift;
			if (shift < 16)
			 	lnlink.offset = binoffset + ssamp * nchan * lnitem.datasize * lnitem.framesize +
					(chan - 1)*2;
			else
			 	lnlink.offset = binoffset + ssamp * nchan * lnitem.framesize +
					(chan - 1);

			lnlink.multiplex=nchan-1;
			lnlink.linkdate = filestat.st_mtime;
			lnlink.machine = SFSMACHINE;

			/* add linked item to destination file */
			if (!sfswritelink(&lnitem,esamp-ssamp,&lnlink,dstfilename))
				error("write failed on temporary file",NULL);
		}
		else {
			/* create output item header */
			if (it==TX_TYPE)
				sfsheader(&lnitem,it,0,4,1,1.0/sfreq,0.0,0,0,1);
			else
				sfsheader(&lnitem,it,0,2,1,1.0/sfreq,0.0,1,0,0);
			if (isnetwork)
				sprintf(lnitem.history,"%s(node=%s,file=%s,headerlen=%d,start=%d,end=%d,freq=%g,channels=%d/%d%s,dc=%d,mult=%d)",
					PROGNAME,
					netname,
					srcfilename,
					binoffset,
					ssamp,esamp,
					sfreq,chan,nchan,
					(swapbyte)?",swapped":"",
					dcoffset,shift);
			else
				sprintf(lnitem.history,"%s(file=%s,headerlen=%d,start=%d,end=%d,freq=%g,channels=%d/%d%s,dc=%d,mult=%d)",
					PROGNAME,
					srcfilename,
					binoffset,
					ssamp,esamp,
					sfreq,chan,nchan,
					(swapbyte)?",swapped":"",
					dcoffset,shift);

			/* create link header */
			memset(&lnlink,0,sizeof(struct link_header));
			if (isnetwork) {
				strcpy(lnlink.filename,relfilename);
				strcpy(lnlink.filepath,netname);
				lnlink.filetype = NETWORK_TYPE;
			}
			else {
				strcpy(lnlink.filename,relfilename);
				strcpy(lnlink.filepath,"");
			}
			lnlink.filetype |= BINARY_TYPE;
			lnlink.swab=swapbyte;
			lnlink.dcoffset=dcoffset;
			lnlink.shift=shift;
			if (shift < 16)
			 	lnlink.offset = binoffset + ssamp * nchan * lnitem.datasize * lnitem.framesize +
					(chan - 1)*2;
			else
			 	lnlink.offset = binoffset + ssamp * nchan * lnitem.framesize +
					(chan - 1);

			lnlink.multiplex=nchan-1;
			lnlink.linkdate = filestat.st_mtime;
			lnlink.machine = SFSMACHINE;

			/* add linked item to destination file */
			if (!sfswritelink(&lnitem,esamp-ssamp,&lnlink,dstfilename))
				error("write failed on temporary file",NULL);
		}
	}

	/* update output file */
	if (!sfsupdate(dstfilename))
		error("update error on '%s'",dstfilename);

	/* that's all folks */
	exit(0);
}

/* create relative file path */
void relatepath(rel,src,dst)
char *rel;
char *src;
char *dst;
{
	char	asrc[SFSMAXFILENAME];
	char	adst[SFSMAXFILENAME];
	char	*p;
	int	i;

	/* get absolute pathnames */
	strcpy(asrc,pathname(src));
	strcpy(adst,pathname(dst));

	/* remove pathname element */
	if ((p=strrchr(asrc,'/')) || (p=strrchr(asrc,'\\')))
		*p='\0';
	if ((p=strrchr(adst,'/')) || (p=strrchr(adst,'\\')))
		*p='\0';

	/* eliminate common prefix */
	for (i=0;asrc[i] && adst[i];i++)
		if (asrc[i]!=adst[i]) break;
	strcpy(rel,"");
	if ((asrc[i]!='\0') || (adst[i]!='\0')) {
		while ((i>0) && (asrc[i-1]!='/') && (asrc[i-1]!='\\')) i--;
		strcpy(asrc,asrc+i);
		strcpy(adst,adst+i);

		if ((adst[0]!='/') && (adst[0]!='\\')) strcat(rel,"../");
		for (i=0;adst[i];i++)
			if ((adst[i]=='/') || (adst[i]=='\\'))
				strcat(rel,"../");
		strcat(rel,asrc);
		strcat(rel,"/");
	}
	if ((p=strrchr(src,'/')) || (p=strrchr(src,'\\')))
		strcat(rel,p+1);
	else
		strcat(rel,src);

/*	printf("relpath(%s,%s)=>%s\n",src,dst,rel); */
}

/* make absolute path */
void absolutepath(asrc,src,dst)
char *asrc;
char *src;
char *dst;
{
	char	*p;

	/* take copy of source */
	strcpy(asrc,src);

	/* remove pathname element */
	if ((p=strrchr(asrc,'/')) || (p=strrchr(asrc,'\\')))
		*(p+1)='\0';
	else
		*asrc = '\0';

	/* add in destination */
	strcat(asrc,dst);

/*	printf("abspath(%s,%s) => %s\n",src,dst,asrc); */
}

/* process binary file */
int	binproc(fname,fsize,nchan,swapb,dcoff,shift,sfreq,offset,numf)
char	*fname;
int32	fsize;
int	*nchan;
int	*swapb;
int	*dcoff;
int	*shift;
double	*sfreq;
int32	*offset;
int	*numf;
{
	/* shift can suggest 8-bit */
	if (*nchan==0) *nchan=1;
	if (*shift >=16)
		*numf= (fsize-headerlen) / *nchan;
	else
		*numf= (fsize-headerlen) / (2 * *nchan);

	/* sampling freq must be specified */
	if (*sfreq < 0)
		fprintf(stderr,"sampling frequency unspecified for binary file: '%s'",srcfilename);

	/* offset is any given header length */
	*offset = headerlen;

	return(0);
}

/* process RIFF file */
int	riffproc(fname,fsize,nchan,swapb,dcoff,shift,sfreq,offset,numf)
char	*fname;
int32	fsize;
int	*nchan;
int	*swapb;
int	*dcoff;
int	*shift;
double	*sfreq;
int32	*offset;
int	*numf;
{
	FILE	*ip;
	struct riff_format_rec {
		unsigned short	format;
		unsigned short	nchan;
		uint32	srate;
		uint32	brate;
		unsigned short	balign;
		unsigned short	nbits;
		unsigned short	xsize;
		char	junk[256];
	} rifform;
	char	riffstr[8];
	int32	size;

	/* open file and check params */
	if ((ip=fopen(fname,"rb"))==NULL) {
		fprintf(stderr,"could not find '%s'\n",fname);
		return(1);
	}
	if (fread(riffstr,1,4,ip)!=4) {
		fprintf(stderr,"read error on '%s'\n",fname);
		fclose(ip);
		return(1);
	}
	if (strncmp(riffstr,"RIFF",4)!=0) {
		fprintf(stderr,"not a RIFF file '%s'\n",fname);
		fclose(ip);
		return(1);
	}
	if (fread(&size,4,1,ip)!=1) {
		fprintf(stderr,"read error on '%s'\n",fname);
		fclose(ip);
		return(1);
	}
	if (fread(riffstr,1,4,ip)!=4) {
		fprintf(stderr,"read error on '%s'\n",fname);
		fclose(ip);
		return(1);
	}
	if (strncmp(riffstr,"WAVE",4)!=0) {
		fprintf(stderr,"not a RIFF wave file '%s'\n",fname);
		fclose(ip);
		return(1);
	}

	/* look for format chunk */
	do {
		if (fread(riffstr,1,4,ip)!=4) {
			fprintf(stderr,"read error on '%s'\n",fname);
			fclose(ip);
			return(1);
		}
		if (strncmp(riffstr,"fmt ",4)==0) break;
		if (fread(&size,4,1,ip)!=1) {
			fprintf(stderr,"read error on '%s'\n",fname);
			fclose(ip);
			return(1);
		}
#if SFSMACHINE==0
		REVERSELONG(size);
#endif
		if (fseek(ip,size,1)!=0) {
			fprintf(stderr,"seek error on '%s'\n",fname);
			fclose(ip);
			return(1);
		}
	} while (1);

	/* read format chunk */
	fread(&size,4,1,ip);
#if SFSMACHINE==0
	REVERSELONG(size);
#endif
	if (size > sizeof(struct riff_format_rec)) {
		fprintf(stderr,"%s: format error on '%s'\n",PROGNAME,fname);
		fprintf(stderr,"%s: possibly not linear PCM data\n",PROGNAME);
		fclose(ip);
		return(1);
	}
	fread(&rifform,size,1,ip);
#if SFSMACHINE==0
	REVERSESHORT(rifform.format);
	REVERSESHORT(rifform.nchan);
	REVERSELONG(rifform.srate);
	REVERSELONG(rifform.brate);
	REVERSESHORT(rifform.balign);
	REVERSESHORT(rifform.nbits);
#endif

	/* set up parameters */
	if (*nchan==0) *nchan = rifform.nchan;
	if (*sfreq < 0) *sfreq = rifform.srate;
	if (rifform.nbits <= 8)
		*shift += 24;

	/* look for data chunk */
	do {
		if (fread(riffstr,1,4,ip)!=4) {
			fprintf(stderr,"read error on '%s'\n",fname);
			fclose(ip);
			return(1);
		}
		if (strncmp(riffstr,"data",4)==0) break;
		if (fread(&size,4,1,ip)!=1) {
			fprintf(stderr,"read error on '%s'\n",fname);
			fclose(ip);
			return(1);
		}
#if SFSMACHINE==0
		REVERSELONG(size);
#endif
		if (fseek(ip,size,1)!=0) {
			fprintf(stderr,"seek error on '%s'\n",fname);
			fclose(ip);
			return(1);
		}
	} while (1);

	/* get # samples to replay */
	fread(numf,4,1,ip);
#if SFSMACHINE==0
	REVERSELONG(*numf);
#endif
	if (*shift < 16)
		*numf /= 2 * *nchan;
	else {
		*numf /= *nchan;
		*dcoff = 128;
	}

	/* offset is current pos */
	*offset = ftell(ip);

#if SFSMACHINE==0
	if (*shift < 16) *swapb=1;
#endif

	fclose(ip);
	return(0);
}

/* process VOC file */
int	vocproc(fname,fsize,nchan,swapb,dcoff,shift,sfreq,offset,numf)
char	*fname;
int32	fsize;
int	*nchan;
int	*swapb;
int	*dcoff;
int	*shift;
double	*sfreq;
int32	*offset;
int	*numf;
{
	FILE	*ip;
	union {
		struct type1_rec {
			unsigned char 	tc1;
			unsigned char 	pack1;
		} type1;
		struct type8_rec {
			unsigned short	tc8;
			unsigned char	pack8;
			unsigned char	mode8;
		} type8;
	} vocrec;
	short	dstart;
	int32	count;
	int	extend=0;
	char	head[32];

	/* get filename and format */
	if ((ip=fopen(fname,"rb"))==NULL) {
		fprintf(stderr,"could not open '%s'\n",fname);
		return(1);
	}

	/* read header */
	if (fread(head,20,1,ip)!=1) {
		fprintf(stderr,"read error on '%s'\n",fname);
		fclose(ip);
		return(1);
	}
	if (strncmp(head,"Creative Voice File",19)!=0) {
		fprintf(stderr,"'%s' is not a .VOC file\n",fname);
		fclose(ip);
		return(1);
	}
	if (fread(&dstart,2,1,ip)!=1) {
		fprintf(stderr,"read error on '%s'\n",fname);
		fclose(ip);
		return(1);
	}
#if SFSMACHINE==0
	REVERSESHORT(dstart);
#endif
	fseek(ip,(long)dstart,0);

	/* process file */
	if (fread(&count,1,4,ip)!=4) {
		fprintf(stderr,"read error on '%s'\n",fname);
		fclose(ip);
		return(1);
	}
#if SFSMACHINE==0
	REVERSELONG(count);
#endif
	while (count & 0xFF) {
		switch (count & 0xFF) {
		case 8:		/* extended voice data */
			fread(&vocrec.type8,4,1,ip);
#if SFSMACHINE==0
			REVERSESHORT(vocrec.type8.tc8);
#endif
			if (*sfreq < 0)
				*sfreq = 256000000L/(65532L-(int32)vocrec.type8.tc8);
			if (*nchan==0) *nchan = 1 + vocrec.type8.mode8;
			if (vocrec.type8.pack8 <= 3)
				*shift += 24;
			extend=1;
			break;
		case 1:		/* voice data */
			fread(&vocrec.type1,2,1,ip);
			if (extend==0) {
				/* no extension block, use this */
				if (*sfreq < 0)
					*sfreq = 1000000L/(256L-(int32)vocrec.type1.tc1);
				if (vocrec.type1.pack1 <= 3)
					*shift += 24;
				if (*nchan==0) *nchan = 1;
			}
			*numf = (count >> 8) - 2;
			if (*shift < 16)
				*numf /= 2 * *nchan;
			else {
				*numf /= *nchan;
				*dcoff = 128;
			}

			*offset = ftell(ip);
#if SFSMACHINE==0
			if (*shift < 16)  *swapb = 1;
#endif
			fclose(ip);
			return(0);
		default:
			fprintf(stderr,"unprocessed VOC record, type %d\n",count & 0xFF);
		}
		fread(&count,1,4,ip);
#if SFSMACHINE==0
		REVERSELONG(count);
#endif
	}

	fclose(ip);
	fprintf(stderr,"No voice data found in '%s'\n",fname);
	return(1);
}

/* AU file format definitions */
typedef struct {
	uint32	magic;		/* magic number */
	uint32	hdr_size;	/* size of this header */
	uint32	data_size;	/* length of data (optional) */
	uint32	encoding;	/* data encoding format */
	uint32	sample_rate;	/* samples per second */
	uint32	channels;	/* number of interleaved channels */
} Audio_filehdr;

/* Define the magic number */
#define	AUDIO_FILE_MAGIC		((uint32)0x2e736e64)

/* Define the encoding fields */
#define	AUDIO_FILE_ENCODING_MULAW_8	(1)	/* 8-bit ISDN u-law */
#define	AUDIO_FILE_ENCODING_LINEAR_8	(2)	/* 8-bit linear PCM */
#define	AUDIO_FILE_ENCODING_LINEAR_16	(3)	/* 16-bit linear PCM */

/* process AU file */
int	auproc(fname,fsize,nchan,swapb,dcoff,shift,sfreq,offset,numf)
char	*fname;
int32	fsize;
int	*nchan;
int	*swapb;
int	*dcoff;
int	*shift;
double	*sfreq;
int32	*offset;
int	*numf;
{
	FILE		*ip;
	Audio_filehdr	head;

	/* get filename and format */
	if ((ip=fopen(fname,"rb"))==NULL) {
		fprintf(stderr,"could not open '%s'\n",fname);
		return(1);
	}

	/* read header */
	if (fread((char *)&head,sizeof(Audio_filehdr),1,ip)!=1) {
		fprintf(stderr,"read error on '%s'\n",fname);
		fclose(ip);
		return(1);
	}
#if SFSMACHINE!=0
	REVERSELONG(head.magic);
	REVERSELONG(head.hdr_size);
	REVERSELONG(head.data_size);
	REVERSELONG(head.encoding);
	REVERSELONG(head.sample_rate);
	REVERSELONG(head.channels);
#endif
	if (head.magic != AUDIO_FILE_MAGIC) {
		fprintf(stderr,"file is not AU format: '%s'\n",fname);
		fclose(ip);
		return(1);
	}
	if (head.encoding==AUDIO_FILE_ENCODING_MULAW_8) {
		fprintf(stderr,"WARNING: 8-bit data is logarithmically encoded in '%s'\n",fname);
		*shift += 16;
	}
	else if (head.encoding==AUDIO_FILE_ENCODING_LINEAR_8)
		*shift += 24;
	else if (head.encoding!=AUDIO_FILE_ENCODING_LINEAR_16) {
		fprintf(stderr,"file contains incompatible signal format: '%s'\n",fname);
		fclose(ip);
		return(1);
	}
	if (*sfreq < 0) *sfreq = head.sample_rate;
	if (*nchan==0) *nchan = head.channels;
	if (*shift < 16)
		*numf = head.data_size / (2 * *nchan);
	else
		*numf = head.data_size / *nchan;

	*offset = head.hdr_size;

#if SFSMACHINE!=0
	if (*shift < 16) *swapb=1;
#endif

	fclose(ip);
	return(0);
}

double	convertlongdouble(bytes)
char	*bytes;
{
	double		f;
	int32		expon;
	uint32 	himant, lomant;

	expon = ((bytes[0] & 0x7F) << 8) | (bytes[1] & 0xFF);
	himant = ((uint32)(bytes[2] & 0xFF) << 24) |
	         ((uint32)(bytes[3] & 0xFF) << 16) |
		 ((uint32)(bytes[4] & 0xFF) << 8)  |
		 ((uint32)(bytes[5] & 0xFF));
 	lomant = ((uint32)(bytes[6] & 0xFF) << 24) |
		 ((uint32)(bytes[7] & 0xFF) << 16) |
		 ((uint32)(bytes[8] & 0xFF) << 8)  |
		 ((uint32)(bytes[9] & 0xFF));

	if ((expon == 0) && (himant == 0) && (lomant == 0))
		/* zero */
		f = 0;
	else if (expon == 0x7FFF)
		/* Infinity or NaN */
#ifdef HUGE_VAL
		f = HUGE_VAL;
#else
		f = 1E50;
#endif
	else {
		/* normal number */
		expon -= 16383;
		f = ldexp((double)himant, expon-=31);
		f += ldexp((double)lomant, expon-=32);
	}

	/* add sign */
	if (bytes[0] & 0x80)
		return -f;
	else
		return f;
}

/* process AIFF file */
int	aiffproc(fname,fsize,nchan,swapb,dcoff,shift,sfreq,offset,numf)
char	*fname;
int32	fsize;
int	*nchan;
int	*swapb;
int	*dcoff;
int	*shift;
double	*sfreq;
int32	*offset;
int	*numf;
{
	FILE	*ip;
	struct aiff_header {
		char	formid[4];
		int32	size;
		char	aiffid[4];
	} aiffhead;
	struct aiff_chunk_rec {
		char	chunkid[4];
		int32	size;
	} chk;
	int32		nextpos;
	short		nbit;
	int		i;
	char		buf[10];

	/* open file and check params */
	if ((ip=fopen(fname,"rb"))==NULL) {
		fprintf(stderr,"could not find '%s'\n",fname);
		return(1);
	}
	if (fread(&aiffhead,sizeof(struct aiff_header),1,ip)!=1) {
		fprintf(stderr,"read error on '%s'\n",fname);
		fclose(ip);
		return(1);
	}
	if (strncmp(aiffhead.formid,"FORM",4) ||
	    strncmp(aiffhead.aiffid,"AIFF",4)) {
		fprintf(stderr,"not a AIFF file '%s'\n",fname);
		fclose(ip);
		return(1);
	}
	while (fread(&chk,sizeof(struct aiff_chunk_rec),1,ip)==1) {
#if SFSMACHINE!=0
		REVERSELONG(chk.size);
#endif
		nextpos = ftell(ip) + ((chk.size+1L)&~1L);
		if (strncmp(chk.chunkid,"COMM",4)==0) {
			/* basic parameters */
			*nchan = getc(ip);
			*nchan = *nchan * 256 + getc(ip);
			*numf = getc(ip);
			*numf = *numf * 256 + getc(ip);
			*numf = *numf * 256 + getc(ip);
			*numf = *numf * 256 + getc(ip);
			nbit = getc(ip);
			nbit = nbit * 256 + getc(ip);
			if (nbit <= 8)
				shift += 24;
			for (i=0;i<10;i++)
				buf[i]=getc(ip);

			/* convert extended double to normal double */
			*sfreq = convertlongdouble(buf);
		}
		else if (strncmp(chk.chunkid,"SSND",4)==0) {
			/* signal */
			*offset = ftell(ip) + 8;
		}
		/* else skip */
		fseek(ip,nextpos,0);
	}

#if SFSMACHINE!=0
	if (*shift < 16) *swapb=1;
#endif

	fclose(ip);
	return(0);
}

/* process ILS file */
int	ilsproc(fname,fsize,nchan,swapb,dcoff,shift,sfreq,offset,numf)
char	*fname;
int32	fsize;
int	*nchan;
int	*swapb;
int	*dcoff;
int	*shift;
double	*sfreq;
int32	*offset;
int	*numf;
{
	FILE	*ip;
	int	ilshead[128];
	int	context,numpadding;
#if SFSMACHINE!=0
	int	i;
#endif

	/* open file and check params */
	if ((ip=fopen(fname,"rb"))==NULL) {
		fprintf(stderr,"could not find '%s'\n",fname);
		return(1);
	}
	if (fread(ilshead,4,128,ip)!=128) {
		fprintf(stderr,"read error on '%s'\n",fname);
		fclose(ip);
		return(1);
	}
#if SFSMACHINE!=0
	for (i=0;i<128;i++)
		REVERSELONG(ilshead[i]);
#endif
	if (ilshead[62] != -32000) {
		fprintf(stderr,"'%s' is incorrect ILS file type",fname);
		fclose(ip);
		return(1);
	}
	*sfreq = ilshead[60];				/* ILS HEAD 61 */
	*sfreq = ilshead[61] * pow(10.0,*sfreq);	/* ILS HEAD 62 */

	*numf = 256*ilshead[5] + ilshead[11];		/* ILS HEAD 6 & 12 */

	context=ilshead[3];
	numpadding=ilshead[7]-1;			/* ILS HEAD 8 */

	*numf -= context*numpadding;
	*offset = ftell(ip) + context*numpadding;
	if (*nchan==0) *nchan = 1;

#if SFSMACHINE!=0
	*swapb=1;
#endif

	fclose(ip);
	return(0);
}

/* process HTK file */
int	htkproc(fname,fsize,nchan,swapb,dcoff,shift,sfreq,offset,numf)
char	*fname;
int32	fsize;
int	*nchan;
int	*swapb;
int	*dcoff;
int	*shift;
double	*sfreq;
int32	*offset;
int	*numf;
{
	FILE	*ip;
	struct htk_header {
		int32	nsamp;
		int32	nperiod;
		short	ssize;
		short	skind;
	} hhead;

	/* open file and check params */
	if ((ip=fopen(fname,"rb"))==NULL) {
		fprintf(stderr,"could not find '%s'\n",fname);
		return(1);
	}
	if (fread(&hhead,sizeof(struct htk_header),1,ip)!=1) {
		fprintf(stderr,"read error on '%s'\n",fname);
		fclose(ip);
		return(1);
	}
#if SFSMACHINE!=0
	REVERSELONG(hhead.nsamp);
	REVERSELONG(hhead.nperiod);
	REVERSESHORT(hhead.ssize);
	REVERSESHORT(hhead.skind);
#endif
	if (hhead.skind!=0) {
		fprintf(stderr,"'%s' is not an HTK sampled data file\n",fname);
		fclose(ip);
		return(1);
	}
	*numf = hhead.nsamp;
	*sfreq = 10000000.0/(double)hhead.nperiod;
	if (hhead.ssize==1)
		*shift += 24;
	*offset = ftell(ip);
	if (*nchan==0) *nchan = 1;

#if SFSMACHINE!=0
	if (*shift < 16) *swapb=1;
#endif

	fclose(ip);
	return(0);
}

/* process PCLX file */
int	pclxproc(fname,fsize,nchan,swapb,dcoff,shift,sfreq,offset,numf)
char	*fname;
int32	fsize;
int	*nchan;
int	*swapb;
int	*dcoff;
int	*shift;
double	*sfreq;
int32	*offset;
int	*numf;
{
	FILE	*ip;

	/* open file and check params */
	if ((ip=fopen(fname,"rb"))==NULL) {
		fprintf(stderr,"could not find '%s'\n",fname);
		return(1);
	}
	*offset = 0xB4C;
	*numf = (fsize - *offset)/4;
	*sfreq = 1000000.0;
	*nchan = 1;
	*swapb = 0;
	*dcoff = 0;
	*shift = 0;

#if SFSMACHINE==0
	*swapb=1;
#endif

	fclose(ip);
	return(0);
}
