/* epson -- DIG routines for EPSON Stylus Pro output */

/* Mark Huckvale - University College London */

/* version 1.0 - November 1996 */

#include "SFSCONFG.h"
#ifdef	EVe

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <ctype.h>
#include <time.h>
#include <malloc.h>
#include <string.h>
#include "dig.h"
#include "digdata.h"

/* Page Layout stuff */

/* Dots per inch of target device */
#define  DPI            ((int32)90)
	/* use 90dpi for lines/text, 360dpi for colours/grey scale */

/* Dimensions of paper, and of border around drawing area, 
   assuming 90dpi and landscape orientation. */
#define	BORDERWIDTH	20
#define	BORDERHEIGHT   	45
#define	A3PAPERWIDTH    (1410-2*BORDERWIDTH)
#define	A3PAPERHEIGHT	(1046-2*BORDERHEIGHT)
#define	A4PAPERWIDTH    (956-2*BORDERWIDTH)
#define	A4PAPERHEIGHT	(742-2*BORDERHEIGHT)
#define	A5PAPERWIDTH    (682-2*BORDERWIDTH)
#define	A5PAPERHEIGHT	(523-2*BORDERHEIGHT)
static int PAPERWIDTH=A4PAPERWIDTH;
static int PAPERHEIGHT=A4PAPERHEIGHT;

/* default font */
#include "pc6x8.h"
#define DEFCHARWIDTH	6
#define DEFCHARHEIGHT	8

/* operating modes */
#define EP_INITIALISED 	1
#define EP_A5		4
#define EP_TOFILE	8
#define EP_A3		16

/* local storage of parameters */
struct {
	int	mode;		/* bitmap modes */
	int	subtype;	/* subtype as supplied from digmap */
	int	myfile;		/* PS system opened output file */
	int	col;
	int	width,height;
	unsigned char **bmap;	/* bitmap array */
	int	currow,curcol;
	int	topipe;		/* output is to pipe */
} digDEVe;

/* shorthand for allowed colours */
#define W	0	/* white */
#define B	1	/* black */
#define M	2	/* magenta */
#define C	4	/* cyan */
#define V	6	/* violet */
#define Y	8	/* yellow */
#define R	10	/* red */
#define G	12	/* green */

/* maxel designs for colours */
struct maxel_rec {
	unsigned char	col[4][4];
} colourtab[33]={
{ /* white */		{{W,W,W,W}, {W,W,W,W}, {W,W,W,W}, {W,W,W,W}} },
{ /* red */		{{R,R,R,R}, {R,R,R,R}, {R,R,R,R}, {R,R,R,R}} },
{ /* light-violet */	{{W,V,W,V}, {V,W,V,W}, {W,V,W,V}, {V,W,V,W}} },
{ /* dark-red-yellow */	{{G,R,G,R}, {R,G,R,G}, {G,R,G,R}, {R,G,R,G}} },
{ /* light-green */	{{W,G,W,G}, {G,W,G,W}, {W,G,W,G}, {G,W,G,W}} },
{ /* dark blue */	{{B,C,B,C}, {C,B,C,B}, {B,C,B,C}, {C,B,C,B}} },
{ /* green */		{{G,G,G,G}, {G,G,G,G}, {G,G,G,G}, {G,G,G,G}} },
{ /* light-violet */	{{W,V,W,V}, {V,W,V,W}, {W,V,W,V}, {V,W,V,W}} },
{ /* yellow green */	{{G,Y,G,Y}, {Y,G,Y,G}, {G,Y,G,Y}, {Y,G,Y,G}} },
{ /* dark magenta */	{{G,M,G,M}, {M,G,M,G}, {G,M,G,M}, {M,G,M,G}} },
{ /* orange */		{{R,Y,R,Y}, {Y,R,Y,R}, {R,Y,R,Y}, {Y,R,Y,R}} },
{ /* violet */		{{V,V,V,V}, {V,V,V,V}, {V,V,V,V}, {V,V,V,V}} },
{ /* dark-cyan */	{{G,C,G,C}, {G,C,G,C}, {C,G,C,G}, {G,C,G,C}} },
{ /* magenta */		{{M,M,M,M}, {M,M,M,M}, {M,M,M,M}, {M,M,M,M}} },
{ /* cyan */		{{C,C,C,C}, {C,C,C,C}, {C,C,C,C}, {C,C,C,C}} },
{ /* black */		{{B,B,B,B}, {B,B,B,B}, {B,B,B,B}, {B,B,B,B}} },
{ /* grey00 */		{{W,W,W,W}, {W,W,W,W}, {W,W,W,W}, {W,W,W,W}} },
{ /* grey01 */		{{W,W,W,W}, {W,W,B,W}, {W,W,W,W}, {W,W,W,W}} },
{ /* grey02 */		{{W,W,W,W}, {W,W,B,W}, {W,B,W,W}, {W,W,W,W}} },
{ /* grey03 */		{{W,W,W,W}, {W,W,B,W}, {W,B,B,W}, {W,W,W,W}} },
{ /* grey04 */		{{W,W,W,W}, {W,B,B,W}, {W,B,B,W}, {W,W,W,W}} },
{ /* grey05 */		{{W,B,W,W}, {W,B,B,W}, {W,B,B,W}, {W,W,W,W}} },
{ /* grey06 */		{{W,B,B,W}, {W,B,B,W}, {W,B,B,W}, {W,W,W,W}} },
{ /* grey07 */		{{W,B,B,B}, {W,B,B,W}, {W,B,B,W}, {W,W,W,W}} },
{ /* grey08 */		{{W,B,B,B}, {W,B,B,B}, {W,B,B,W}, {W,W,W,W}} },
{ /* grey09 */		{{W,B,B,B}, {W,B,B,B}, {W,B,B,B}, {W,W,W,W}} },
{ /* grey10 */		{{W,B,B,B}, {W,B,B,B}, {W,B,B,B}, {W,W,W,B}} },
{ /* grey11 */		{{W,B,B,B}, {W,B,B,B}, {W,B,B,B}, {W,W,B,B}} },
{ /* grey12 */		{{W,B,B,B}, {W,B,B,B}, {W,B,B,B}, {W,B,B,B}} },
{ /* grey13 */		{{W,B,B,B}, {W,B,B,B}, {W,B,B,B}, {B,B,B,B}} },
{ /* grey14 */		{{W,B,B,B}, {W,B,B,B}, {B,B,B,B}, {B,B,B,B}} },
{ /* grey15 */		{{W,B,B,B}, {B,B,B,B}, {B,B,B,B}, {B,B,B,B}} },
{ /* grey16 */		{{B,B,B,B}, {B,B,B,B}, {B,B,B,B}, {B,B,B,B}} },
};

/* standard colours */
static int bundlemap[117]={
/* 00 */	0,
/* 01-14 */	1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14,
/* 15-29 */	15, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14,
/* 30-35 */	15, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14,
/* 45-35 */	15, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14,
/* 60-35 */	15, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14,
/* 75-35 */	15, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14,
/* 90-99 */	15, 1, 2, 3, 4, 5, 6, 7, 8, 9,
/* 100-116 */	16, 17, 18, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 29, 30, 31, 32,
};

/* forward declarations */
void ep_print_bitmap(FILE *op);
static void ep_colour();

/* get graphics parameters */
int ep_params(subtype,digdev)
int	subtype;
struct digdevice *digdev;
{
	char	*p;
	char	cbuf[32];

	/* record operating mode */
	digDEVe.subtype = subtype;
	if (subtype >= 200) {
		digDEVe.mode |= EP_TOFILE;
		subtype -= 100;
	}
	if (subtype >= 100) {
		digDEVe.mode |= EP_A3;
		subtype -= 100;
		PAPERWIDTH = A3PAPERWIDTH;
		PAPERHEIGHT = A3PAPERHEIGHT;
	}
	if (subtype >= 50) {
		digDEVe.mode |= EP_A5;
		PAPERWIDTH = A5PAPERWIDTH;
		PAPERHEIGHT = A5PAPERHEIGHT;
	}
	
	/* set up image dimensions */
	if ((p=getenv("GSIZE"))!=NULL) {
		strncpy(cbuf,p,31);
		digDEVe.width = atoi(strtok(cbuf,"x"));
		digDEVe.height = atoi(strtok(NULL,""));
		if (digDEVe.width < 10) digDEVe.width = PAPERWIDTH;
		if (digDEVe.height < 10) digDEVe.height = PAPERHEIGHT;
	}
	else {
		digDEVe.width = PAPERWIDTH;
		digDEVe.height = PAPERHEIGHT;
	}

	/* force dimensions to be multiple of 8 */
	digDEVe.width &= ~7;
	digDEVe.height &= ~7;
	
	digdev->nhoriz = digDEVe.width;
	digdev->nvert = digDEVe.height;

	/* return other image characteristics */
	digdev->aspect = 1.0;
	digdev->chwidth = DEFCHARWIDTH;
	digdev->chheight = DEFCHARHEIGHT;
	digdev->greylevels = 17;
	digdev->greypixwidth = 1;
	digdev->greypixheight = 1;

	return(0);
}

/* open output file */
int ep_outfile(fp)
FILE	**fp;
{
	char	*p,ofilename[256];

	if ((p=getenv("GFILE"))!=NULL)
		strncpy(ofilename,p,255);
	else
		strcpy(ofilename,"epson.dat");

	if (digDEVe.mode & EP_TOFILE) {
		if ((*fp = fopen(ofilename,"w"))==NULL)
			error("could not open '%s'",ofilename);
		digDEVe.myfile=1;
	}
#ifdef DOS
	else {
		int ptr = (digDEVe.subtype % 50)/2;
#ifdef _MSC_VER
		FILE *stdprn = fopen("PRN","wb");
#endif
		digDEVe.myfile=0;
		
		if (ptr==0) {
			*fp = stdprn;
		}
		else if (ptr==1) {
			if ((*fp = fopen("PRN","wb"))==NULL)
				*fp = stdprn;
			else
				digDEVe.myfile=1;
		}
		else if (ptr==2) {
			if ((*fp = fopen("LPT1","wb"))==NULL)
				*fp = stdprn;
			else
				digDEVe.myfile=1;
		}
		else if (ptr==3) {
			if ((*fp = fopen("LPT2","wb"))==NULL)
				*fp = stdprn;
			else
				digDEVe.myfile=1;
		}
#ifdef _MSC_VER
		if (digDEVe.myfile==1) fclose(stdprn);
#endif
	}
#else
	else if (digDEVe.myfile==0) {
		char	pipename[SFSMAXFILENAME];
		sprintf(pipename,"%s/bin/epson %d",sfsbase(),digDEVe.subtype % 50);
		if ((*fp = popen(pipename,"w"))==NULL)
			error("could not open pipe to '%s'\n",pipename);
		digDEVe.myfile=1;
		digDEVe.topipe=1;
	}
#endif
	return(0);
}

/* open output */
int ep_open()
{
	int	i;

	if (digDEVe.mode & EP_INITIALISED) return(0);

	/* allocate memory for bitmap */
	if ((digDEVe.bmap = (unsigned char **)calloc(digDEVe.height,sizeof(unsigned char *)))==NULL)
		error("could not get memory for bitmap");
	for (i=0;i<digDEVe.height;i++)
		if ((digDEVe.bmap[i]=(unsigned char *)calloc(digDEVe.width,1))==NULL)
			error("could not get memory for bitmap");

	digDEVe.mode |= EP_INITIALISED;
	return(0);
}

/* interrupt */
void ep_inter()
{
	int	i;
	if (digDEVe.mode & EP_INITIALISED) {
		for (i=0;i<digDEVe.height;i++)
			free(digDEVe.bmap[i]);
		free(digDEVe.bmap);
	}
}

/* close and save bitmap */
void ep_close()
{
	if (digDEVe.mode & EP_INITIALISED) {

		/* save bitmap */
		ep_print_bitmap(digdata.outfile);

		/* free bitmap */
		ep_inter();
	        
		/* shut down output stream */
		if (digDEVe.myfile && (digDEVe.mode & EP_TOFILE))
			fclose(digdata.outfile);
#ifndef DOS
		else if (digDEVe.topipe)
			pclose(digdata.outfile);
#endif
	}
	digDEVe.mode = 0;
	digDEVe.myfile=0;
	digDEVe.topipe=0;
}

/* clear screen */
int ep_clear()
{
	int	i;
	for (i=0;i<digDEVe.height;i++)
		memset(digDEVe.bmap[i],0,digDEVe.width);
	return(0);
}

/* set up colour from bundle */
static void ep_colour(bundle)
int	bundle;
{
	digDEVe.col = bundlemap[bundle];
}

/* plot line on bitmap */
static void ep_plotline(int x1,int y1,int x2,int y2)
{
	int	xt,yt;
	int	i;
	float	slope,val;

	if ((x1==x2) && (y1==y2)) {
		/* plot point */
		digDEVe.bmap[y1][x1]=digDEVe.col;
		return;
	}
	xt = x2 - x1;
	yt = y2 - y1;
	if (xt==0) {
		/* vertical line */
		if (y1 > y2) { yt = y1; y1 = y2; y2 = yt; }
		for (i=y1;i<=y2;i++) digDEVe.bmap[i][x1]=digDEVe.col;
	}
	else if (yt==0) {
		/* horizontal line */
		if (x1 > x2) { xt = x1; x1 = x2; x2 = xt; }
		for (i=x1;i<=x2;i++) digDEVe.bmap[y1][i]=digDEVe.col;
	}
	else if (abs(xt) > abs(yt)) {
		/* use x as a base */
		slope = (float)yt/(float)xt;
		if (x2 < x1) {
			val = y2+0.5;
			for (i=x2;i<=x1;i++,val+=slope)
				digDEVe.bmap[(int)val][i]=digDEVe.col;
		}
		else {
			val = y1+0.5;
			for (i=x1;i<=x2;i++,val+=slope)
				digDEVe.bmap[(int)val][i]=digDEVe.col;
		}
	}
	else {
		/* use y as a base */
		slope = (float)xt/(float)yt;
		if (y2 < y1) {
			val = x2+0.5;
			for (i=y2;i<=y1;i++,val+=slope)
				digDEVe.bmap[i][(int)val]=digDEVe.col;
		}
		else {
			val = x1+0.5;
			for (i=y1;i<=y2;i++,val+=slope)
				digDEVe.bmap[i][(int)val]=digDEVe.col;
		}
	}
}

/* plot poly-line */
void ep_polyline(bundle,buff,len)
int bundle;
short *buff;
int32	len;
{
	int	x1,y1,x2,y2;
        
	/* set up plot colour */
	ep_colour(bundle);

	/* move to first point */
	x1 = *buff++;
	y1 = *buff++;
	len-=(int32)2;

	/* plot lines */
	while (len > (int32)0) {
		x2 = *buff++;
		y2 = *buff++;
		len -= (int32)2;
		ep_plotline(x1,y1,x2,y2);
		x1 = x2;
		y1 = y2;
	}
}

/* draw character */
static void ep_char(int x,int y,int ch)
{
	unsigned char	*cbuf;
	int		i,j;
	unsigned char	mask;

	/* locate character in stored font */
	cbuf = pc6x8_font[ch];

	/* row by row */
	for (i=DEFCHARHEIGHT-1;i>=0;i--) {
		/* get row mask */
		mask = *cbuf++;
		/* display row */
		for (j=0;j<DEFCHARWIDTH;j++) {
			if (mask & 0x80)
				digDEVe.bmap[y+i][x+j] = digDEVe.col;
			mask <<= 1;
		}
	}

}

/* draw text */
void ep_text(bundle,x,y,str)
int bundle;
int x;
int y;
char *str;
{
	/* set up plot colour */
	ep_colour(bundle);

	/* write string character by character */
	while (*str) {
		if (x > (digDEVe.width-DEFCHARWIDTH)) break;
		ep_char(x,y,*str);
		str++;
		x += DEFCHARWIDTH;
	}
}

/* draw a filled rectangle */
void ep_fillrect(bundle,x1,y1,x2,y2)
int bundle;
int x1;
int y1;
int x2;
int y2;
{
	int	i,j;

	/* set up plot colour */
	ep_colour(bundle);

	for (i=x1;i<=x2;i++)
		for (j=y1;j<=y2;j++)
			digDEVe.bmap[j][i] = digDEVe.col;
	        
}

/* grey-level display */
void ep_gscale(x,y,height,buff)
int x;
int y;
int height;
unsigned char *buff;
{
	int	i;

	for (i=height-1;i>=0;i--,buff++)
		digDEVe.bmap[y+i][x] = bundlemap[100 + *buff];

}

/* wait for key routine - NULL don't wait */
void ep_pause()
{
}       

#define MOVXDOT	0xE5

static unsigned char lbuf[10000];
static unsigned char *lptr;

/* send RLE code for n repeats of byte b */
static void xferbyt(int b,int n)
{
	while (n > 0) {
		if (n==1) {
			*lptr++ = 0;
			*lptr++ = b;
			n--;
		}
		else if (n < 127) {
			*lptr++ = 257 - n;
			*lptr++ = b;
			n = 0;
		}
		else {
			*lptr++ = 257 - 127;
			*lptr++ = b;
			n -= 127;
		}
	}
}

static void xferline(FILE *op)
{
	int	n = lptr-lbuf;
	int	i;
	
	if (n <= 15)
		fputc(0x20+n,op);
	else if (n <= 255) {
		fputc(0x31,op); fputc(n,op);
	}
	else {
		fputc(0x32,op); fputc(n%256,op); fputc(n/256,op);
	}
	for (i=0;i<n;i++) fputc(lbuf[i],op);
	lptr = lbuf;
}

static unsigned char COL[4]={0x08,0x04,0x02,0x01};
static unsigned char CMYK[10000][4];

/* print bitmap in Epson ESC/P commands */
void ep_print_bitmap(FILE *op)
{
	int	row,col,clr,sub,cur;
	int	pos,j;
	int	count,lastbyt;

	/* reset printer */
	fprintf(op,"\033@");

	/* set units to 360dpi */
	fprintf(op,"\033(U\001"); fputc(0,op); fputc(10,op);

	/* enter graphics mode */
	fprintf(op,"\033(G\001"); fputc(0,op); fputc(1,op);

	/* Compressed raster graphics TIFF mode in 360x360 dpi */
	fprintf(op,"\033.\002"); fputc(10,op); fputc(10,op); fputc(1,op); fputc(0,op); fputc(0,op);

	/* move down by border width */
	fputc(0x72,op); fputc((4*BORDERWIDTH)%256,op); fputc((4*BORDERWIDTH)/256,op);
	
	/* positioning to nearest dot */
	fputc(MOVXDOT,op);

	/* move to next raster line */
	fputc(0x61,op);

	for (row=0;row<digDEVe.width;row++) {

		/* for each of 4 sub-rows */
		for (sub=0;sub<4;sub++) {
			
			/* convert row into CMYK byte pattern */
			for (col=0,pos=BORDERHEIGHT/2;col<digDEVe.height;col+=2,pos++) {
				cur = digDEVe.bmap[col][row];
				for (j=0;j<4;j++) {
					CMYK[pos][0] = (CMYK[pos][0]<<1) | ((colourtab[cur].col[sub][j] & COL[0])?1:0);
					CMYK[pos][1] = (CMYK[pos][1]<<1) | ((colourtab[cur].col[sub][j] & COL[1])?1:0);
					CMYK[pos][2] = (CMYK[pos][2]<<1) | ((colourtab[cur].col[sub][j] & COL[2])?1:0);
					CMYK[pos][3] = (CMYK[pos][3]<<1) | ((colourtab[cur].col[sub][j] & COL[3])?1:0);
				}
				cur = digDEVe.bmap[col+1][row];
				for (j=0;j<4;j++) {
					CMYK[pos][0] = (CMYK[pos][0]<<1) | ((colourtab[cur].col[sub][j] & COL[0])?1:0);
					CMYK[pos][1] = (CMYK[pos][1]<<1) | ((colourtab[cur].col[sub][j] & COL[1])?1:0);
					CMYK[pos][2] = (CMYK[pos][2]<<1) | ((colourtab[cur].col[sub][j] & COL[2])?1:0);
					CMYK[pos][3] = (CMYK[pos][3]<<1) | ((colourtab[cur].col[sub][j] & COL[3])?1:0);
				}
			}

			/* for each colour */
			for (clr=0;clr<4;clr++) {

				/* set colour */
				fputc(0x80+(COL[clr]>>1),op);
			
				/* convert CMYK pattern into RLE */
				lptr=lbuf;
				lastbyt = CMYK[0][clr];
				count=1;
				for (j=1;j<pos;j++) {
					if (CMYK[j][clr]==lastbyt)
						count++;
					else {
						xferbyt(lastbyt,count);
						lastbyt = CMYK[j][clr];
						count=1;
					}
				}
				xferbyt(lastbyt,count);
				xferline(op);
			}

			/* print row */
			fputc(MOVXDOT,op);

			/* move to next raster line */
			fputc(0x61,op);
		}
	}

	/* exit TIFF mode */
	fputc(0xE3,op);
	
	/* reset printer */
/*	fprintf(op,"\033@"); */
	fprintf(op,"\r\n\f");

	fflush(op);
}
#endif


