/* x11dig -- DIG routines for primitive X11 graphics */

/* Mark Huckvale - University College London */

/* version 1.0 - May 1993 */

#include "SFSCONFG.h"
#ifdef EVX		/* <- letter chosen for device type */

#include <stdio.h>
#include <math.h>
#include <ctype.h>
#ifdef _MSC_VER
#define XLIB_NT
#include <windows.h>
#include "xnt.h"

/* patches for XLIB_NT */
#define XDefaultRootWindow 	DefaultRootWindow
#define XWhitePixel		WhitePixel
#define XSetIOErrorHandler	SetIOErrorHandler
#define XDefaultGC		DefaultGC

#else
#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#endif
#include "dig.h"
#include "digdata.h"

/* default sizes */
#define DEFAULT_XSTART	50		/* initial window position */
#define DEFAULT_YSTART	50
#define DEFAULT_WIDTH	780		/* initial window size */
#define DEFAULT_HEIGHT	560
#define DEFAULT_FONT	"6x10"		/* choice of font size */
#define MIN(x,y) (((x)<(y))?(x):(y))
#define MAX(x,y) (((x)>(y))?(x):(y))
#define ABS(x)	(((x)>0)?(x):-(x))

static int x11_window_width=DEFAULT_WIDTH;
static int x11_window_height=DEFAULT_HEIGHT;

/* global data */
static struct {
	/* DIG stuff */
	short		mode;		/* 0=?, 1=initialised */
	short		rop;		/* raster operation */
	short		currbun;	/* current bundle */
	short 		xmouse,ymouse,smouse; /* mouse status */
	/* X stuff */
	Display		*xdisp;		/* main X display */
	int		xscrn;		/* default X screen */
	GC		xdfgc;		/* default graphics context */
	Window		xwin;		/* current window */
	Font		xfont;		/* 'hardware' font */
	XFontStruct	*xfsp;		/* font pointer */
} digDEVX;

/* colours */
#define NUM_COLOUR	32
static struct {
        unsigned long pixel;
	unsigned short	r,g,b;
} colourtab[NUM_COLOUR]={
	{ 0L, 250,250,250},	/* 0 = background */
	{ 0L, 200, 50, 50},	/* 1 */
	{ 0L, 100, 75,100},	/* 2 */
	{ 0L, 100, 75, 50},	/* 3 */
	{ 0L, 150,200,  0},	/* 4 */
	{ 0L,  25, 25,100},	/* 5 = was red - mapped to dark grey/blue */
	{ 0L,  50,200,100},	/* 6 */
	{ 0L,  50,100,200},	/* 7 */
	{ 0L, 150,150,  0},	/* 8 */
	{ 0L,  50, 50, 50},	/* 9 = was white - mapped to dark grey */
	{ 0L, 200,100,  0},	/* 10 */
	{ 0L, 100,  0,100},	/* 11 */
	{ 0L,   0,100,100},	/* 12 */
	{ 0L, 200,100,100},	/* 13 */
	{ 0L,  50,100,200},	/* 14 */
	{ 0L,   0,  0,  0},	/* 15 = foreground */
	{ 0L, 250,250,250},	/* 16 */
	{ 0L, 248,248,248},	/* 17 */
	{ 0L, 244,244,244},	/* 18 */
	{ 0L, 238,238,238},	/* 19 */
	{ 0L, 230,230,230},	/* 20 */
	{ 0L, 220,220,220},	/* 21 */
 	{ 0L, 208,208,208},	/* 22 */
	{ 0L, 194,194,194},	/* 23 */
	{ 0L, 178,178,178},	/* 24 */
	{ 0L, 160,160,160},	/* 25 */
	{ 0L, 140,140,140},	/* 26 */
	{ 0L, 118,118,118},	/* 27 */
	{ 0L,  94, 94, 94},	/* 28 */
	{ 0L,  68, 68, 68},	/* 29 */
	{ 0L,  40, 40, 40},	/* 30 */
	{ 0L,  10, 10, 10},	/* 31 */
};

/* IO error handler */
static int	x11digint(disp)
Display		*disp;
{
	/* IO error - stop */
	fprintf(stderr,"SFS/DIG: closed\n");
	exit(0);
	return(0);
}

/* open graphics */
int x11dig_open()
{
	XSetWindowAttributes	watts;
	unsigned long		wmask;
#ifndef XLIB_NT
	XEvent			event;
#endif
	XColor			col;
	XWMHints		wmhints;
	int			i;
	XGCValues		gc_values;
	
	if (!(digDEVX.mode)) {
		/* change to graphics mode */
#ifdef _MSC_VER
		if ((digDEVX.xdisp = XOpenDisplay("NTXlib"))==NULL)
#else
		if ((digDEVX.xdisp = XOpenDisplay(NULL))==NULL)
#endif
			error("failed to open X display");
		digDEVX.xscrn = XDefaultScreen(digDEVX.xdisp);
		digDEVX.xdfgc = XDefaultGC(digDEVX.xdisp,digDEVX.xscrn);

		/* set up error handling */
#ifdef XLIB_NT
		XSetErrorHandler(x11digint);
#else
		XSetIOErrorHandler(x11digint);
#endif	
		/* set up window attributes */
		watts.background_pixel = XWhitePixel(digDEVX.xdisp,digDEVX.xscrn);
		watts.backing_store = WhenMapped;
		watts.event_mask = ExposureMask | ButtonPressMask | ButtonReleaseMask |
			KeyPressMask | StructureNotifyMask;
		wmask = CWBackingStore | CWEventMask | CWBackPixel;

		/* create and display window */
		digDEVX.xwin = XCreateWindow(digDEVX.xdisp,XDefaultRootWindow(digDEVX.xdisp),
			DEFAULT_XSTART,DEFAULT_YSTART,
			x11_window_width,x11_window_height,2,
			CopyFromParent, InputOutput, CopyFromParent,
			wmask, &watts);
		XStoreName(digDEVX.xdisp,digDEVX.xwin,"SFS/DIG");

		/* get keyboard */
		wmhints.flags = InputHint;
		wmhints.input = True;
		XSetWMHints(digDEVX.xdisp,digDEVX.xwin,&wmhints);

		/* map to screen */
		XMapWindow(digDEVX.xdisp,digDEVX.xwin);

#ifdef XLIB_NT
		/* get a graphics context */
		digDEVX.xdfgc = XCreateGC(digDEVX.xdisp, digDEVX.xwin, (long)0, &gc_values);
#endif
		/* wait for it to appear */
#ifndef XLIB_NT
		while ( (XCheckWindowEvent(digDEVX.xdisp,digDEVX.xwin,ExposureMask,&event))) /* wait */;
		while (!(XCheckWindowEvent(digDEVX.xdisp,digDEVX.xwin,ExposureMask,&event))) /* wait */;
#endif

		/* load font */
		digDEVX.xfsp = XLoadQueryFont(digDEVX.xdisp,DEFAULT_FONT);
		digDEVX.xfont = digDEVX.xfsp->fid;

		/* set up colours */
		for (i=0;i<NUM_COLOUR;i++) {
#ifdef XLIB_NT
			col.red = colourtab[i].r;
			col.green = colourtab[i].g;
			col.blue = colourtab[i].b;
#else
			col.red = colourtab[i].r << 8;
			col.green = colourtab[i].g << 8;
			col.blue = colourtab[i].b << 8;
#endif
			if (XAllocColor(digDEVX.xdisp,DefaultColormap(digDEVX.xdisp,0),&col)==0)
				fprintf(stderr,"Failed to allocate colour (%d,%d,%d)\n",
					col.red,col.green,col.blue);
			colourtab[i].pixel = col.pixel;
		}
		XSetFunction(digDEVX.xdisp,digDEVX.xdfgc,GXcopy);

		(digDEVX.mode)++;
		digDEVX.currbun=24;
		digDEVX.rop=0;
	}
	return(0);
}

/* device characteristics */
int x11dig_params(subtype,digdev)
int	subtype;
struct digdevice *digdev;
{
	XWindowAttributes	info;
	char	*p;
	char	cbuf[32];

	if ((p=getenv("GSIZE"))!=NULL) {
		strncpy(cbuf,p,31);
		x11_window_width = atoi(strtok(cbuf,"x"));
		x11_window_height = atoi(strtok(NULL,""));
		if ((x11_window_height < 10) || (x11_window_width < 10))
			error("dig: GSIZE (WIDTHxHEIGHT) too small");
	}
	else if (subtype==1) {
		x11_window_width = DEFAULT_WIDTH/2;
		x11_window_height = DEFAULT_HEIGHT/2;
	}

	if (!(digDEVX.mode)) {		/* may need to open */
		x11dig_open();
	}

	XSync(digDEVX.xdisp,False);
	XGetWindowAttributes(digDEVX.xdisp,digDEVX.xwin,&info);
	digdev->nhoriz = info.width;
	digdev->nvert  = info.height;
	digdev->aspect = (float)1.0;
#ifdef XLIB_NT
	digdev->chwidth = 7;
	digdev->chheight = 8;
#else
	digdev->chwidth = 6;
	digdev->chheight = 10;
#endif
	digdev->greylevels = 16;
	digdev->greypixwidth = 1;
	digdev->greypixheight = 1;

	return(0);
}

/* close graphics - return to text mode */
int x11dig_close()
{
	/* return to text mode */
	if (digDEVX.mode) {
		XUnmapWindow(digDEVX.xdisp,digDEVX.xwin);
		XDestroyWindow(digDEVX.xdisp,digDEVX.xwin);
	}
	digDEVX.mode = 0;

	return(0);
}

/* interrupt graphics */
int x11dig_inter()
{
	/* deal with interrupted drawing */
	x11dig_close();
	fprintf(stderr,"Interrupt\n");
	fflush(stderr);

	return(0);
}

/* clear screen */
int x11dig_clear()
{
	/* clear screen to background */
	XClearWindow(digDEVX.xdisp,digDEVX.xwin);

	return(0);
}

/* set up colour from bundle */
static int x11dig_colour(bundle)
int	bundle;
{
	unsigned long pix;
	int	mbundle;
	
	/* bundles are 0=background,1-100 various colours/line styles */
	if (bundle < 5)
		pix = colourtab[bundle].pixel;
	else if (bundle >= 100)
		pix = colourtab[bundle-84].pixel;
	else {
		mbundle = bundle % 15;
		if (mbundle==0) 
			pix = colourtab[15].pixel;
		else
			pix = colourtab[mbundle].pixel;
	}
#if 0
	if (digDEVX.rop)
		XSetForeground(digDEVX.xdisp,digDEVX.xdfgc,pix ^ XWhitePixel(digDEVX.xdisp,digDEVX.xscrn));
	else
#endif
	if (bundle)
		XSetForeground(digDEVX.xdisp,digDEVX.xdfgc,pix);	
	else
		XSetForeground(digDEVX.xdisp,digDEVX.xdfgc,XWhitePixel(digDEVX.xdisp,digDEVX.xscrn));	

	return(0);
}

/* block line draw into sections of maximum complexity */
#define MAXLSEGS	2048

/* draw a poly-line */
int x11dig_polyline(bundle,buff,len)
int	bundle;
short	*buff;
int	len;
{
	int	i,l;
	XPoint	xpts[MAXLSEGS];

	/* sort out colours */
	x11dig_colour(bundle);

	while (len > 2) {
		l = MIN(len/2,MAXLSEGS);
		
		/* position to first point */
		xpts[0].x = *buff++;
		xpts[0].y = digdata.nvert - *buff++ - 1;
		len -= 2;

		for (i=1;i<l;i++) {
			xpts[i].x = *buff++;
			xpts[i].y = digdata.nvert - *buff++ - 1;
			len -= 2;
		}

		/* plot line */
		XDrawLines(digDEVX.xdisp,digDEVX.xwin,digDEVX.xdfgc,xpts,l,CoordModeOrigin);

		/* next segment starts at last point */
		buff--;
		buff--;
		len += 2;
	}

	return(0);
}

/* draw 'hardware' text */
int x11dig_text(bundle,x,y,str)
int	bundle;
int	x,y;
char	*str;
{
#ifndef XLIB_NT
	XTextItem	txt;
#endif

	/* sort out colours */
	x11dig_colour(bundle);

	/* write string */
#ifdef XLIB_NT
	XDrawString(digDEVX.xdisp,digDEVX.xwin,digDEVX.xdfgc,
		x,digdata.nvert-y-1-digDEVX.xfsp->descent,str,strlen(str));
#else
	txt.chars = str;
	txt.nchars = strlen(str);
	txt.delta = 1;
	txt.font = digDEVX.xfont;
	XDrawText(digDEVX.xdisp,digDEVX.xwin,digDEVX.xdfgc,
		x,digdata.nvert-y-1-digDEVX.xfsp->descent,&txt,1);
#endif
	return(0);
}

/* draw a filled rectangle */
int x11dig_fillrect(bundle,x1,y1,x2,y2)
int	bundle;
int	x1,y1,x2,y2;
{
	/* sort out colours */
	x11dig_colour(bundle);

	/* draw filled box */
	XFillRectangle(digDEVX.xdisp,digDEVX.xwin,digDEVX.xdfgc,
		MIN(x1,x2),
		digdata.nvert - MAX(y1,y2) -1,
		ABS(x1-x2)+1,
		ABS(y1-y2)+1);

	return(0);
}

/* change line/text raster operation */
int x11dig_rasterop(mode)
int	mode;
{
	int	ret;

	/* set new raster op 0=write, 1=xor */
	ret=digDEVX.rop;
	if (mode)
		XSetFunction(digDEVX.xdisp,digDEVX.xdfgc,GXxor);
	else
		XSetFunction(digDEVX.xdisp,digDEVX.xdfgc,GXcopy);
	digDEVX.rop=mode;
	return(ret);
}

/* draw a circle */
int x11dig_circle(bundle,xc,yc,r)
int	bundle;
int	xc,yc,r;
{
	/* sort out colours */
	x11dig_colour(bundle);

	/* draw a circle */

	return(0);
}

/* draw a filled circle */
int x11dig_fcircle(bundle,xc,yc,r)
int	bundle;
int	xc,yc,r;
{
	/* sort out colours */
	x11dig_colour(bundle);

	/* draw a filled circle */

	return(0);
}

/* initialise pointing device */
int x11dig_pointer(flag,xm,ym)
int	flag;
int	xm,ym;
{
	if (flag) {
		/* turn mouse on */
		
	}
	else {
		/* turn mouse off */
	}

	return(0);
}

/* get keyboard & mouse input */
int x11dig_mouse(but,xm,ym,ich)
int	*but;
int	*xm;
int	*ym;
int	*ich;
{
	XEvent	event;
	KeySym	keysym;
	XComposeStatus status;
	char	chbuf[8];
	int	nch;
	static int	mmask[4] = {0,4,2,1};

	*xm = digDEVX.xmouse;
	*ym = digDEVX.ymouse;
	*but = digDEVX.smouse;
	*ich = 0;

	/* get a window event */
	do {

		XNextEvent(digDEVX.xdisp,&event);

		switch (event.type) {
		case ButtonPress:
			*xm = digDEVX.xmouse = event.xbutton.x;
			*ym = digDEVX.ymouse = digdata.nvert - event.xbutton.y -1;
			*but = digDEVX.smouse | mmask[event.xbutton.button];
			digDEVX.smouse = *but;
			return(DIG_MOUSE);

		case ButtonRelease:
			*xm = digDEVX.xmouse = event.xbutton.x;
			*ym = digDEVX.ymouse = digdata.nvert - event.xbutton.y -1;
			*but = digDEVX.smouse & ~mmask[event.xbutton.button];
			digDEVX.smouse = *but;
			return(DIG_MOUSE);

		case KeyPress:
			nch = XLookupString((XKeyEvent *)&event,chbuf,sizeof(chbuf),&keysym,&status);
				/* note should return all chars in turn */
			*ich = chbuf[nch-1];	/* return last char */
			return(DIG_KEY);

		case Expose:
			/* only act on last expose request in queue */
			if (event.xexpose.count==0) {
				while (XCheckTypedEvent(digDEVX.xdisp,Expose,&event))
					/* skip */;
/* cjh_printf("DIG_REDRAW\n"); */
				return(DIG_REDRAW);
			}
			break;

		case DestroyNotify:
			return(DIG_QUIT);

		default:
			break;
		}
	} while (1);

}

/* display text on prompt line */
int x11dig_prompt(str)
char	*str;
{
	XClearArea(digDEVX.xdisp,digDEVX.xwin,0,0,
		digdata.nhoriz-1,digdata.chheight,0);
	x11dig_text(24,0,digdata.nvert-digdata.chheight,str);
	XFlush(digDEVX.xdisp);

	return(0);
}

/* grey-level display */
int x11dig_gscale(x,y,height,buff)
int	x,y;
int	height;
unsigned char	*buff;
{
	int		i,lasty;
	unsigned char 	*p,lastc;

	/* replace greyscales by colours */
	p=buff;
	lastc=*p++;
	y = digdata.nvert - y -1;
	y -= height - 1;
	lasty = y;
	for (i = 1; i < height; i++, p++) {
		if (*p != lastc) {
			XSetForeground(digDEVX.xdisp,digDEVX.xdfgc,colourtab[16+lastc].pixel);
			XDrawLine(digDEVX.xdisp,digDEVX.xwin,digDEVX.xdfgc,x,lasty,x,y+i);
			lasty = y+i;
			lastc = *p;
		}
	}
	XSetForeground(digDEVX.xdisp,digDEVX.xdfgc,colourtab[16+lastc].pixel);
	XDrawLine(digDEVX.xdisp,digDEVX.xwin,digDEVX.xdfgc,x,lasty,x,y+height-1);

	return(0);
}

/* pause for user input */
int x11dig_pause()
{
	XEvent	event;
	int	bstate=0;
	char	chbuf[8];
	int	nch;
	
	do {
		XNextEvent(digDEVX.xdisp,&event);

		switch (event.type) {
		case ButtonPress:
			bstate=1;
			break;

		case ButtonRelease:
			if (bstate==1)
				return(DIG_QUIT);
			break;

		case KeyPress:
			nch = XLookupString((XKeyEvent *)&event,chbuf,sizeof(chbuf),NULL,NULL);
				/* note should return all chars in turn */
			return(chbuf[nch-1]);	/* return last char */

		case Expose:
			/* only act on last expose request in queue */
			if (event.xexpose.count==0) {
				while (XCheckTypedEvent(digDEVX.xdisp,Expose,&event))
					/* skip */;
/* cjh_printf("DIG_REDRAW\n"); */
				return(DIG_REDRAW);
			}
			break;

		case DestroyNotify:
			return(DIG_QUIT);

		default:
			break;
		}
	} while (1);

}

#endif
