/* DSP routines from Mike Brookes, Imperial College London */

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "libic.h"

#define NAME "dftr"

void dftr(data,spect,len)
REAL *data;
COMPLEX *spect;
int len;
/*
	Fourier Transform of real data with arbitrary length

	data[len] -> spect[1 + len/2]

        data and spect can be the same array if desired

This routine uses the Chirp-Z algorithm. If W = exp(j2PI/2N), we need
to calculate X(k) = SUM{x(n) * W**(-2kn)} =
      W**-kk * SUM{x(n)*W**-nn * W**(k-n)(k-n)} since
-2kn = (k-n)(k-n) - kk - nn.

Thus the fourier transform consists of multiplying by W**-nn, convolving
with W**nn and multiplying by W**-kk. An ordinary fft is used to perform
the convolution; the data must be zero extended to avoid wraparound.
The array W**nn and its transform are saved incase the next transform
is of the same length.
*/
{
    COMPLEX *tdata,*chirp,*Tchirp,*tend;
    COMPLEX *pspect,*pchirp,*ptdata,*qtdata,*pTchirp;
    REAL *pdata;
    static COMPLEX czero = {0.0,0.0};
    REAL datar,datai,chirpr,chirpi;
    int tlen,halftlen;

#ifdef IAG
    int i;		/* for diagnostics only */
#endif

/* Create the chirp function  */

    makechirp(len,&halftlen,&chirp);
    tlen = halftlen + halftlen;
    Tchirp = chirp + len;
    tdata = Tchirp + halftlen + 1;
    tend = tdata + tlen;

#ifdef IAG
printf("Chirp Function\n\n");
for (i=0; i<len; i++) printf("%8d %10g %10g\n",i,chirp[i].r,chirp[i].i);

printf("Transform\n\n");
for (i=0; i<(halftlen+1); i++) 
    printf("%8d %10g %10g\n",i,Tchirp[i].r,Tchirp[i].i);
#endif


/* Multiply Original data by conjugate of Chirp function */

#ifdef IAG
printf("Original data\n\n");
for (i=0; i<len; i++) 
    printf("%8d %10g\n",i,data[i]);
#endif

    pchirp = chirp;
    pdata = data;
    ptdata = tdata;
    while (pchirp < Tchirp) {
	datar = *pdata++;
	chirpr = pchirp->r;
	chirpi = (pchirp++)->i;
	ptdata->r = chirpr*datar;
	(ptdata++)->i = chirpi*datar;
	}
    while (ptdata < tend) *ptdata++ = czero;

#ifdef IAG
printf("Multiplied data\n\n");
for (i=0; i<tlen; i++) 
    printf("%8d %10g %10g\n",i,tdata[i].r,tdata[i].i);
#endif

    fftc(tdata,tdata,tlen);

#ifdef IAG
printf("Transformed data\n\n");
for (i=0; i<tlen; i++) 
    printf("%8d %10g %10g\n",i,tdata[i].r,tdata[i].i);
#endif

/* Now do the convolution */

    ptdata = tdata;
    qtdata = tend;
    pTchirp = Tchirp;
    chirpr = pTchirp->r; chirpi = (pTchirp++)->i;	/* DC Term */
    datar = ptdata->r; datai = ptdata->i;
    ptdata->r = chirpr*datar + chirpi*datai;
    (ptdata++)->i = chirpr*datai - chirpi*datar;
    while (ptdata < --qtdata) {
	chirpr = pTchirp->r; chirpi = (pTchirp++)->i;
	datar = ptdata->r; datai = ptdata->i;
	ptdata->r = chirpr*datar + chirpi*datai;
	(ptdata++)->i = chirpr*datai - chirpi*datar;

	datar = qtdata->r; datai = qtdata->i;
	qtdata->r = chirpr*datar + chirpi*datai;
	qtdata->i = chirpr*datai - chirpi*datar;
	}
    chirpr = pTchirp->r; chirpi = pTchirp->i;	/* Nyquist Term */
    datar = ptdata->r; datai = ptdata->i;
    ptdata->r = chirpr*datar + chirpi*datai;
    ptdata->i = chirpr*datai - chirpi*datar;

#ifdef IAG
printf("Times Tchirp\n\n");
for (i=0; i<tlen; i++) 
    printf("%8d %10g %10g\n",i,tdata[i].r,tdata[i].i);
#endif

    fftci(tdata,tdata,tlen);

#ifdef IAG
printf("Inverse Transform\n\n");
for (i=0; i<tlen; i++) 
    printf("%8d %10g %10g\n",i,tdata[i].r,tdata[i].i);
#endif

/* Now copy back into the destination array */

    ptdata = tdata;
    qtdata = tdata+(len >> 1);
    pspect = spect;
    pchirp = chirp;
    while (ptdata <= qtdata) {
	datar = ptdata->r;
	datai = (ptdata++)->i;
	chirpr = pchirp->r;
	chirpi = (pchirp++)->i;
	pspect->r = chirpr*datar - chirpi*datai;
	(pspect++)->i = chirpr*datai + chirpi*datar;
	}

#ifdef IAG
printf("Output spectrum\n\n");
for (i=0; i<len; i++) 
    printf("%8d %10g %10g\n",i,spect[i].r,spect[i].i);
#endif

    return;
    }

#ifdef EMO

#include <stdio.h>
#include <math.h>
#define LEN 32
#define TWOPI (2.0*PI)

main ()
{
    REAL data[LEN];
    COMPLEX spect[1 + LEN/2];
    REAL sum,theta;
    int i,len,hlen;

    for (;;) {
	do {
	    printf("Enter transform length: ");
	    scanf("%d",&len);
	    } while ((len<2) || (len>LEN));
	hlen = len/2;
	printf("Test for dftr: 1.1 + 2cos(f) + 3sin(3f) + 4cos(%df)\n",hlen);
	for (i=0; i<len; i++) {
	    theta = (TWOPI * i) / len;
	    data[i] = 1.1 + 2.0*cos(theta) + 3.0*sin(3.0*theta) + 
		      4.0*cos(theta*hlen);
	    }

	dftr(data,spect,len);

	for (i=0; i<=hlen; i++) {
	    spect[i].r /= len;
	    spect[i].i /= len;
	    if ((spect[i].r*spect[i].r + spect[i].i*spect[i].i) > 0.01) {
		printf("%3d %8.2f %8.2f\n",i,spect[i].r,spect[i].i);
		}
	    }

	}
    }
#endif
