/* DSP routines from Mike Brookes, Imperial College London */

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "libic.h"

/*
 Implementation of split-radix fft from Duhamel ASSP 35 (6) p818 1987 
 This is a decimation in time version so does rearrangement first 
 subscripting tidied up 
 some signs changed because the one in the paper calculates the complex conj 
 lookup table for sin & cos 

 On input x[2**m] contains the real input data in bit reversed order
   on output x[i] i=0,2**(m-1) contains the real parts
   and x[i] i=2**m,2**(m-1) contains the imaginary parts
   of the dft. Only 2**(m-1)+1 values are needed because
   the dft has hermitian symmetry.


   Note that the imaginary parts are in reverse order.
*/

#define TWOPI (2.0*3.1415926535)
#define SQHALF (0.707106778)

void rsplfft(x,n,m)
REAL *x;
int m,n;
{
    int n1,n2,n4;	/* n = 2**m */
    int j;
    int i0,i1,i2,i3;
    int is,id;
    int ia0,ia1,ia2,ia3;
    int ib0,ib1,ib2,ib3;

    REAL t0,t1,t2,r1;
    REAL cc1,ss1,cc3,ss3;
    REAL c2,c3,d2,d3;
    REAL *psine,*psine3,*sines;
    int cosoff,sinc,inc,inc3;

    sintable(m,&sines,&sinc);
    cosoff = sinc*n/4;
    n1 = n-1;

#ifdef IAG
printf(" (Bit reversed) original data\n");
for (is=0; is<n; is++) printf("%d %g\n",is,x[is]);
#endif

/* Length 2 transforms */

    is = 0;
    id = 4;
    do {
	for (i0 = is; i0 <n; i0+=id) {
	    i1 = i0+1;
	    r1 = x[i0];
	    x[i0] = r1 + x[i1];
	    x[i1] = r1 - x[i1];
#ifdef IAG
printf("2: %d=%g, %d=%g\n",i0,x[i0],i1,x[i1]);
#endif
	    }
	is = id + id - 2;
	id *= 4;
	} while (is < n);

/* Other butterflies */

    for (n2=4; n2<=n; n2+=n2) {
	n4 = n2/4;

/* without mult */

	is = 0;
	id = n2 + n2;
	do {
	    for (i0=is; i0 < n1; i0 += id) {
		i1 = i0+n4;
		i2 = i1+n4;
		i3 = i2+n4;
		t0 = x[i2] + x[i3];
		x[i3] = x[i3] - x[i2];
		x[i2] = x[i0] - t0;
		x[i0] = x[i0] + t0;
#ifdef IAG
printf("4R: %d=%g, (%d,%d)=(%g,%g), %d=%g\n",i0,x[i0],i1,i3,x[i1],x[i3],i2,x[i2]);
#endif
		}
	    is = 2*id-n2;
	    id = 4*id;
	    } while (is < n);
	if (n4 >= 2) {	
	    
/* with 2 real mults */

	    is = n4/2 ;
	    id = 2*n2;
	    do {
		for (i0 = is; i0 < n1; i0 += id) {
		    i1 = i0+n4;
		    i2 = i1+n4;
		    i3 = i2+n4;
		    t1 = (x[i2] - x[i3])*SQHALF;
		    t2 = (x[i2] + x[i3])*SQHALF;
		    x[i2] = - t2 - x[i1];
		    x[i3] =  -t2 + x[i1];
		    x[i1] = x[i0] - t1;
		    x[i0] = x[i0] + t1;
#ifdef IAG
printf("4r: (%d,%d)=(%g,%g), (%d,%d)=(%g,%g)\n",i0,i3,x[i0],x[i3],i1,i2,x[i1],x[i2]);
#endif
		    }
		is = 2*id - n2 + n4/2 ;
		id *= 4;
		} while (is < n);
	    inc = (n*sinc)/n2;
	    inc3 = 3*inc;
	    if (n4 >= 4) {
		psine = psine3 = sines;
		for (j=1; j< n4/2; j++) {
		    psine += inc;
		    psine3 += inc3;
		    cc1 = psine[cosoff];
		    ss1 = *psine;
		    cc3 = psine3[cosoff];
		    ss3 = *psine3;
		    is = j;
		    id = n2+n2;
		    do {
		         for (ia0=is; ia0 < n1; ia0 += id) {
			     
/* with 6 real mults */

			    ib1 = ia0 + n4;
			    ia1 = ib1 - j - j ;
			    ib0 = ia1 + n4;
			    ia2 = ib1 + n4;
			    ia3 = ia2 + n4;
			    ib2 = ib0 + n4;
			    ib3 = ib2 + n4;
#ifdef IAG
printf("%g %g %g %g %g %g %g %g\n",x[ia0],x[ia1],x[ib1],x[ib0],x[ia2],x[ib2],x[ia3],x[ib3]);
printf("cs1 = (%g,%g), cs3 = (%g,%g)\n",cc1,ss1,cc3,ss3);
#endif
			    c2 = x[ia2]*cc1 + x[ib2]*ss1;
			    d2 = x[ia2]*ss1 - x[ib2]*cc1;
			    c3 = x[ia3]*cc3 + x[ib3]*ss3;
			    d3 = x[ia3]*ss3 - x[ib3]*cc3;
#ifdef IAG
printf("%g %g %g %g ",c2,d2,c3,d3);
#endif
			    t1 = c2 + c3;
			    c3 = c2 - c3;
			    t2 = d2 - d3;
			    d3 = d2 + d3;
#ifdef IAG
printf("%g %g %g %g\n",t1,c3,t2,d3);
#endif
			    x[ia2] = -x[ib0] - d3;
			    x[ib2] = -x[ib1] - c3;
			    x[ia3] = x[ib1] - c3;
			    x[ib3] = x[ib0] - d3;
			    x[ib1] = x[ia1] - t2;
			    x[ib0] = x[ia0] - t1;
			    x[ia0] = x[ia0] + t1;
			    x[ia1] = x[ia1] + t2;
#ifdef IAG
printf("8: (%d,%d)=(%g,%g), (%d,%d)=(%g,%g)\n",ia0,ib3,x[ia0],x[ib3],ia1,ia3,x[ia1],x[ia3]);
printf("   (%d,%d)=(%g,%g), (%d,%d)=(%g,%g)\n",ib1,ib2,x[ib1],x[ib2],ib0,ia2,x[ib0],x[ia2]);
#endif
			    }
			is = 2*id - n2 + j;
			id *= 4;
			} while (is < n);
		    }
		}
	    }
	}
    return;
    }
#ifdef EMO

#include <stdio.h>

main()
{
    REAL x[64];
    int i;
    REAL theta;

/* do a 8 point transform */

    printf("8-point transform: sin(f)\n\n");
    for (i=0; i<8; i++) {
	theta = TWOPI * i / 8.0;
	x[i] = sin(theta);
	}
    rbitrev(x,8);
    rsplfft(x,8,3);
    for (i=0; i<8; i++) {
	if(fabs(x[i]) > 0.0001)  printf("%5d %8g\n",i,x[i]);
	}

/* do a 16 point transform */

    printf("16-point transform: sin(f)\n\n");
    for (i=0; i<16; i++) {
	theta = TWOPI * i / 16.0;
	x[i] = sin(theta);
	}
    rbitrev(x,16);
    rsplfft(x,16,4);
    for (i=0; i<16; i++) {
	if(fabs(x[i]) > 0.0001) printf("%5d %8g\n",i,x[i]);
	}

/* first try a 32 point transform */

    printf("32-point transform: 1.5 + 2cos(f) + 2.5sin(5f)\n\n");
    for (i=0; i<32; i++) {
	theta = TWOPI * i / 32.0;
	x[i] = 1.5 + 2.0*cos(theta) + 2.5*sin(5.0*theta);
	}
    rbitrev(x,32);
    rsplfft(x,32,5);
    for (i=0; i<32; i++) {
	if(fabs(x[i]) > 0.0001) printf("%5d %8g\n",i,x[i]);
	}

/* now try a 64 point transform */

    printf("64-point transform: 1.5 + 2cos(f) + 2.5sin(7f)\n\n");
    for (i=0; i<64; i++) {
	theta = TWOPI * i / 64.0;
	x[i] = 1.5 + 2.0*cos(theta) + 2.5*sin(7.0*theta);
	}
    rbitrev(x,64);
    rsplfft(x,64,6);
    for (i=0; i<64; i++) {
	if(fabs(x[i]) > 0.0001) printf("%5d %8g\n",i,x[i]);
	}
    }
#endif
