/* dosdac -- support for DAC boards on DOS/PC implementation */

/* M.A.Huckvale - University College London */

/* version 1.0 - September 1993 */

/*
 * This module contains code to drive various DAC boards
 * from protected mode using the GNU in-line assembler
 * The routines here are called from the SFSDAC module
 * with a common calling convention.
 *
 * To add support for a new board you need to:
 *  1. add a new environment variable setting and subtype
 *     to SFSDAC DAC selection table
 *  2. add code in this module for the given sub-type
 *     to send 12 and 16 bit waveforms to the device
 */

/* define to include in-line ASM (DJGPP only) */
#ifdef __GNUC__
#define GO_FASTER
#endif

/* include files */
#include "SFSCONFG.h"
#include <stdio.h>
#include <string.h>
#include <math.h>
#include <fcntl.h>
#include "sfs.h"
#define MIN(x,y)	(((x)<(y))?(x):(y))

#ifdef DOS

#include <pc.h>

extern int dac_available_channels;	/* # available channels */
extern double dac_selected_rate;	/* rate chosen */

/* device-specific data */
/*-------------------- UCL Parallel Printer Box -------------------------*/
#ifdef DAC_DOS_PP
unsigned short ppbase = 0x378;

#define PP_DATA		(0)		/* offset of data register */
#define PP_INPUT	(1)		/* offset of input register */
#define PP_CONTROL	(2)		/* offset of control register */
#define PP_TIMECTL	(0x5)		/* on board 8254, select control */
#define PP_TIMEDAT0	(0x9)		/* on board 8254, select timer 0 */
#define PP_TIMEDAT1	(0xD)		/* on board 8254, select timer 1 */
#define PP_STROBECTL	(0x2)		/* 8254 timer write strobe */
#define PP_STROBEDAT	(0x1)		/* dac write strobe */
#define PP_STANDARD	(0x0)		/* dac standard value */
#define PP_INIT0	(0x36)		/* 8254 counter 0 = 16bit/square/binary */
#define PP_INIT1	(0x76)		/* 8254 counter 1 = 16bit/square/binary */
#define PP_READY	(0x80)		/* input timer ready */

/* PPDAC crystal */
#define PP_FREQHI	(0x7A)		/* 8MHz/65536 */
#define PP_FREQLO	(0x1200)	/* 8MHz%65536 */
#define PP_FILFREQHI	(0x3)		/* 200kHz/65536 */
#define PP_FILFREQLO	(0x0D40)	/* 200kHz%65536 */
#endif

/*-------------------- UCL EB12 DAC card -------------------------*/
#ifdef DAC_DOS_EB12
unsigned short ebbase = 0x300;

#define EB_DATALSB	(0)		/* offset of LSB data register */
#define EB_DATAMSB	(1)		/* offset of MSB data register */
#define EB_INPUT	(2)		/* offset of input register */
#define EB_TIMCTL	(7)		/* offset of timer control register */
#define EB_TIMCOUNT0	(4)		/* offset of timer counter 0 register */
#define EB_TIMCOUNT1	(5)		/* offset of timer counter 1 register */
#define EB_TIMINIT0	(0x36)		/* 8254 counter 0 = 16bit/square/binary */
#define EB_TIMINIT1	(0x76)		/* 8254 counter 1 = 16bit/square/binary */
#define EB_TIM0OFF	(0x32)		/* 8254 counter 0 = 16bit/one-shot/binary */
#define EB_READY	(0x80)		/* input timer ready */

/* EBDAC crystal */
#define EB_FREQHI	(0x7A)		/* 8MHz/65536 */
#define EB_FREQLO	(0x1200)	/* 8MHz%65536 */
#define EB_FILFREQHI	(0x3)		/* 200kHz/65536 */
#define EB_FILFREQLO	(0x0D40)	/* 200kHz%65536 */
#endif

/*------------------ Data Translation DT2811 -----------------------------*/
#ifdef DAC_DOS_DT2811
unsigned short dtbase = 0x218;
/* DT2811 sampling rate table */
struct dt_rate_rec {
	int	minf;		/* transition frequency */
	int	code;		/* code to send to baord */
	int	freq;		/* actual frequency of board */
} dt_rate_tab[]={
	17500,	0x019,	20000,
	13500,	0x021,	15000,
	11000,	0x029,	12000,
	8000,	0x031,	10000,
	0,	0x002,	6000	/* guard value */
};

/* DT2811 register offsets */
#define DT_ADC_CONTROL	(0)
#define DT_ADC_GAIN	(1)
#define DT_DAC_LO_BYTE	(2)
#define DT_DAC_HI_BYTE	(3)
#define DT_ADC_LO_BYTE	(2)
#define DT_ADC_HI_BYTE	(3)
#define DT_TIMER	(7)

#endif

/*-------------------- Laryngograph PC/LX -----------------------------*/
#ifdef DAC_DOS_PCLX
/* base address of board */
unsigned short int pclxbase=0x2c0;

/* addresses of control registers */
#define PCLX_DATA_LO	(2)
#define PCLX_DATA_HI	(3)
#define PCLX_ADDR_LO	(4)
#define PCLX_ADDR_HI	(5)
#define PCLX_SAMPLE	(6)

/* store a value in the board memory */
#define PCLX_POKE(addr,data) { \
	outportb(pclxbase+PCLX_ADDR_LO,(addr & 0xFF)); \
	outportb(pclxbase+PCLX_ADDR_HI,(addr >> 8)); \
	outportb(pclxbase+PCLX_DATA_LO,(data & 0xFF)); \
	outportb(pclxbase+PCLX_DATA_HI,(data >> 8)); \
	}

/* sampling frequency selection table */
struct pclx_rate_rec {
	int	minf;		/* transition frequency */
	int	code;		/* code to send to baord */
	int	freq;		/* actual frequency of board */
} pclx_rate_tab[]={
	30000,	0,	40000,
	15000,	4,	20000,
	7500,	8,	10000,
	3750,	12,	5000,
	0,	0,	40000	/* guard value */
};

/* minimum size to replay */
#define PCLX_MIN_COUNT	1000

/* number of samples required to flush out signal */
#define PCLX_FLUSH_COUNT 1000
#endif

#ifdef DAC_DOS_BLAST8
/* routines for Protected Mode replay of (top 8 bits of) 16-bit data */
int	SB8Init(void);
int	SB8Play(short *buff,int numf,int srate,int nchan,int nbits);
void	SB8Close(void);
#endif
#ifdef DAC_DOS_BLAST16
/* routines for Protected Mode replay of 16-bit data */
int	SB16Init(void);
int	SB16Play(short *buff,int numf,int srate,int nchan,int nbits);
void	SB16Close(void);
#endif

static	short	sample;

/* DOSDAC despatch routine */
int32	dos_dac_playback(subtype,buff,numf,srate,nbits,nchannels,ntimes)
int	subtype;	/* code for supported DAC device */
short	*buff;		/* sample buffer */
int32	numf;		/* number of samples */
double	srate;		/* sample rate */
int	nbits;		/* number of bits
			   (12 or 16 only - always right shifted in word */
int	nchannels;	/* # channels to replay, NOTE: if 2 channels sent,
			   - should 1 channel devices play mono or one channel ?
			   - have subsequent dac_(get/set)config() routines ?
			 */
int	ntimes;		/* number of times to replay */
{
	char	*p,*getenv();
	int	i,rep;
	short	*wave;
#ifdef DAC_DOS_PCLX
	int	count;
#endif

	switch (subtype) {
#ifdef DAC_DOS_PP
	case 1:
		/* UCL parallel printer box */
		if (p=getenv("PPBASE")) ppbase=atoi(p);
		dac_available_channels=1;

		/* initialise timer 0 */
		outportb(ppbase+PP_CONTROL,PP_TIMECTL);
		outportb(ppbase,PP_INIT0);
		outportb(ppbase+PP_CONTROL,PP_TIMECTL | PP_STROBECTL);
		outportb(ppbase+PP_CONTROL,PP_TIMECTL);

		/* initialise timer 1 */
		outportb(ppbase+PP_CONTROL,PP_TIMECTL);
		outportb(ppbase,PP_INIT1);
		outportb(ppbase+PP_CONTROL,PP_TIMECTL | PP_STROBECTL);
		outportb(ppbase+PP_CONTROL,PP_TIMECTL);

		/* calculate sample clock */
		outportb(ppbase+PP_CONTROL,PP_TIMEDAT0);
		outportb(ppbase,(8000000/(int)srate) & 0xFF);
		outportb(ppbase+PP_CONTROL,PP_TIMEDAT0 | PP_STROBECTL);
		outportb(ppbase+PP_CONTROL,PP_TIMEDAT0);
		outportb(ppbase,((8000000/(int)srate) & 0xFF00) >> 8);
		outportb(ppbase+PP_CONTROL,PP_TIMEDAT0 | PP_STROBECTL);
		outportb(ppbase+PP_CONTROL,PP_TIMEDAT0);
		dac_selected_rate = (double)(8000000/(int)(8000000/(int)srate));

		/* calculate filter clock */
		outportb(ppbase+PP_CONTROL,PP_TIMEDAT1);
		outportb(ppbase,(200000/(int)srate) & 0xFF);
		outportb(ppbase+PP_CONTROL,PP_TIMEDAT1 | PP_STROBECTL);
		outportb(ppbase+PP_CONTROL,PP_TIMEDAT1);
		outportb(ppbase,((200000/(int)srate) & 0xFF00) >> 8);
		outportb(ppbase+PP_CONTROL,PP_TIMEDAT1 | PP_STROBECTL);
		outportb(ppbase+PP_CONTROL,PP_TIMEDAT1);

		/* turn off interrupts */
#ifdef _MSC_VER
		__asm {
			cli
		}
#else
		asm volatile ("cli");
#endif
		/* loop over number of repetitions */
		for (rep=0;rep<ntimes;rep++) {

			/* loop over number of samples in buffer */
			wave = buff;
			for (i=0;i < numf;i++) {

				/* wait timer is ready */
				while (inportb(ppbase+PP_INPUT) & PP_READY) /* wait */;

				/* get sample */
				sample = (nbits==12) ? (*wave++ << 4) : *wave++;

				/* send lo-byte */
				outportb(ppbase,sample & 0x00F0);
				outportb(ppbase+PP_CONTROL,PP_STROBEDAT);
				outportb(ppbase+PP_CONTROL,PP_STANDARD);

				/* send hi-byte */
				outportb(ppbase,((sample & 0xFF00) >> 8) + 0x80);
				outportb(ppbase+PP_CONTROL,PP_STROBEDAT);
				outportb(ppbase+PP_CONTROL,PP_STANDARD);

				/* wait for timer to go low */
				while ((inportb(ppbase+PP_INPUT) & PP_READY)==0) /* wait */;
			}
		}

		/* restore interrupts */
#ifdef _MSC_VER
		__asm {
			sti
		}
#else
		asm volatile ("sti");
#endif
		break;
#endif
#ifdef DAC_DOS_DT2811
	case 2:
		/* Data Translation 2811 board */
		if (p=getenv("DTBASE")) dtbase=atoi(p);
		dac_available_channels=1;

		/* initialise */
		outportb(dtbase+DT_ADC_CONTROL,0);

		/* set handshake */
		inportb(dtbase+DT_ADC_LO_BYTE);
		inportb(dtbase+DT_ADC_HI_BYTE);

		/* specify timing mode and frequency */
		outportb(dtbase+DT_ADC_CONTROL,0x11);
		for (i=0;(int)srate < dt_rate_tab[i].minf;i++) /* loop */;
		outportb(dtbase+DT_TIMER,dt_rate_tab[i].code);
		dac_selected_rate = dt_rate_tab[i].freq;

		/* set gain and channel */
		outportb(dtbase+DT_ADC_GAIN,0);

		/* turn off interrupts */
#ifdef _MSC_VER
		__asm {
			cli
		}
#else
		asm volatile ("cli");
#endif
		/* for each repetition required */
		for (rep=0;rep < ntimes;rep++) {

			/* send samples */
			wave = buff;
			for (i=0;i<numf;i++) {

				/* wait timer is ready */
				while ((inportb(dtbase+DT_ADC_CONTROL) & 0x80)==0) /* wait */;

				/* reset ADC */
				inportb(dtbase+DT_ADC_HI_BYTE);

				/* get sample */
				sample = (nbits==12) ? *wave++ : (*wave++ >> 4);

				/* send low byte */
				outportb(dtbase+DT_DAC_LO_BYTE,(sample & 0xFF));

				/* send high byte */
				outportb(dtbase+DT_DAC_HI_BYTE,((0x08+(sample >> 8)) & 0x0F));

			}
		}

		/* restore interrupts */
#ifdef _MSC_VER
		__asm {
			sti
		}
#else
		asm volatile ("sti");
#endif
		/* reset */		
		outportb(dtbase+DT_ADC_CONTROL,0);
		break;
#endif
#ifdef DAC_DOS_PCLX
	case 3:
		/* Laryngograph PC/LX board */
		if (p=getenv("PCLXBASE")) pclxbase=atoi(p);
		dac_available_channels=2;

		/* reset card */
		outportb(pclxbase,0);
		PCLX_POKE(0x400,65);	/* replay mode */

		/* select sampling rate */
		for (i=0;(int)srate < pclx_rate_tab[i].minf;i++) /* loop */;
		PCLX_POKE(0x401,pclx_rate_tab[i].code);
		dac_selected_rate = pclx_rate_tab[i].freq;

		/* set gain to max */
		PCLX_POKE(0x410,0x7FFF);
		PCLX_POKE(0x411,0x7FFF);

		/* set DSP running */
		outportb(pclxbase,0x001);

		/* for each repetition required */
		for (rep=0;rep < ntimes;rep++) {

			/* calculate # samples to send */
			count = (numf < PCLX_MIN_COUNT) ? PCLX_MIN_COUNT : numf;
	
			/* send samples */
			wave = buff;
			for (i=0;i<count;i++) {

				/* get sample */
				if (i >= numf)
					sample = 0;
				else
					sample = (nbits==12) ? (*wave++ << 4) : *wave++;

				/* wait for timer */
				while ((inportb(pclxbase) & 0x10)==0) /* wait */;

				/* send lo byte channel 0 */
				outportb(pclxbase+PCLX_SAMPLE, sample & 0xFF);

				/* wait for timer */
				while ((inportb(pclxbase) & 0x10)==0) /* wait */;

				/* send hi byte channel 0 */
				outportb(pclxbase+PCLX_SAMPLE, (sample >> 8));

				/* wait for timer */
				while ((inportb(pclxbase) & 0x10)==0) /* wait */;

				/* send lo byte channel 1 */
				outportb(pclxbase+PCLX_SAMPLE, sample & 0xFF);

				/* wait for timer */
				while ((inportb(pclxbase) & 0x10)==0) /* wait */;

				/* send hi byte channel 1 */
				outportb(pclxbase+PCLX_SAMPLE, (sample >> 8));

			}

			/* flush out data */
			for (i=0;i<PCLX_FLUSH_COUNT;i++) {

				/* get sample */
				sample = 0;

				/* wait for timer */
				while ((inportb(pclxbase) & 0x10)==0) /* wait */;

				/* send lo byte channel 0 */
				outportb(pclxbase+PCLX_SAMPLE, sample & 0xFF);

				/* wait for timer */
				while ((inportb(pclxbase) & 0x10)==0) /* wait */;

				/* send hi byte channel 0 */
				outportb(pclxbase+PCLX_SAMPLE, (sample >> 8));

				/* wait for timer */
				while ((inportb(pclxbase) & 0x10)==0) /* wait */;

				/* send lo byte channel 1 */
				outportb(pclxbase+PCLX_SAMPLE, sample & 0xFF);

				/* wait for timer */
				while ((inportb(pclxbase) & 0x10)==0) /* wait */;

				/* send hi byte channel 1 */
				outportb(pclxbase+PCLX_SAMPLE, (sample >> 8));

			}
		}

		/* turn off DSP */
		outportb(pclxbase,0);
		
		break;
		
#endif
#ifdef DAC_DOS_EB12
	case 4:
		/* UCL expansion bus 12-bit */
		if ((p=getenv("EBBASE"))) ebbase=atoi(p);
		dac_available_channels=1;

		/* initialise timer 0 */
		outportb(ebbase+EB_TIMCTL,EB_TIMINIT0);

		/* initialise timer 1 */
		outportb(ebbase+EB_TIMCTL,EB_TIMINIT1);

		/* calculate sample clock */
		outportb(ebbase+EB_TIMCOUNT0,(8000000/(int)srate) & 0xFF);
		outportb(ebbase+EB_TIMCOUNT0,((8000000/(int)srate) & 0xFF00) >> 8);
		dac_selected_rate = (double)(8000000/(int)(8000000/(int)srate));

		/* calculate filter clock */
		outportb(ebbase+EB_TIMCOUNT1,(200000/(int)srate) & 0xFF);
		outportb(ebbase+EB_TIMCOUNT1,((200000/(int)srate) & 0xFF00) >> 8);

		/* turn off interrupts */
#ifdef _MSC_VER
		__asm {
			cli
		}
#else
		asm volatile ("cli");
#endif
		/* loop over number of repetitions */
		for (rep=0;rep<ntimes;rep++) {

			/* loop over number of samples in buffer */
			wave = buff;
			for (i=0;i < numf;i++) {

				/* get sample */
				sample = (nbits==12) ? (*wave++ << 4) : *wave++;

#ifdef GO_FASTER
				/* wait timer is ready */
				asm volatile ("			\n\
					movzwl 	_ebbase,%edx	\n\
					addl 	$2,%edx		\n\
				loop1:				\n\
					inb 	%dx,%al		\n\
					andb 	$-128,%al	\n\
					testb 	%al,%al		\n\
					je 	done1		\n\
					jmp 	loop1		\n\
				done1:"
				);
				/* send lo-byte & hi-byte */
				asm volatile ("			\n\
					movzwl	_ebbase,%edx	\n\
					movzwl	_sample,%eax	\n\
					andb	$240,%al	\n\
					outb	%al,%dx		\n\
					incl	%edx		\n\
					movzwl	_sample,%eax	\n\
					sarw	$8,%ax		\n\
					addw	$128,%ax	\n\
					outb	%al,%dx		\n"
				);
				/* wait for timer to go low */
				asm volatile ("			\n\
					movzwl 	_ebbase,%edx	\n\
					addl 	$2,%edx		\n\
				loop2:				\n\
					inb 	%dx,%al		\n\
					andb 	$-128,%al	\n\
					testb 	%al,%al		\n\
					jne 	done2		\n\
					jmp 	loop2		\n\
				done2:"
				);
#else
				/* wait timer is ready */
				while (inportb(ebbase+EB_INPUT) & EB_READY) /* wait */;

				/* send lo-byte */
				outportb(ebbase+EB_DATALSB,sample & 0x00F0);

				/* send hi-byte */
				outportb(ebbase+EB_DATAMSB,((sample & 0xFF00) >> 8) + 0x80);

				/* wait for timer to go low */
				while ((inportb(ebbase+EB_INPUT) & EB_READY)==0) /* wait */;
#endif
			}
		}

		/* turn off timer */
		outportb(ebbase+EB_TIMCTL,EB_TIM0OFF);

		/* restore interrupts */
#ifdef _MSC_VER
		__asm {
			sti
		}
#else
		asm volatile ("sti");
#endif

		break;
#endif
#ifdef DAC_DOS_BLAST8
	case 5:
		if (SB8Init()) {
			SB8Play(buff,numf,(int)srate,nchannels,nbits);
			SB8Close();
		}
		break;
#endif
#ifdef DAC_DOS_BLAST16
	case 6:
		if (SB16Init()) {
			SB16Play(buff,numf,(int)srate,nchannels,nbits);
			SB16Close();
		}
		break;
#endif

	default:
		/* not found */
		return(-1);
	}
	return(0);
}

#endif
