/* sbr16dma -- SoundBlaster Compatible 16-bit record by DMA */

/* Mark Huckvale - University College London */

/* version 1.0 - October 1995 */

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <string.h>
#include <dos.h>
#include <conio.h>
#include <go32.h>
#include <dpmi.h>
#include "sfsadc.h"

/* routines for Protected Mode recording of 16-bit data */
int	SBR16Init(void);
int	SBR16Record(short *buff,int numf,int srate,int nchan,int flags);
void	SBR16Close(void);

/* defines for SoundBlaster */
#define SB_DSP_WRITEMIXER	0x04
#define SB_DSP_READMIXER	0x05
#define SB_DSP_RESET		0x06
#define SB_DSP_READ_DATA	0x0A
#define SB_DSP_WRITE_DATA	0x0C
#define SB_DSP_WRITE_STATUS	0x0C
#define SB_DSP_DATA_AVAIL	0x0E
#define SB_DSP_DATA_AVAIL16	0x0F

/* DSP Commands */
#define SB_DMA_16_BIT_DAC	0xB6
#define SB_TIME_16BIT		0x41
#define SB_SPEAKER_ON		0xD1
#define SB_SPEAKER_OFF		0xD3
#define SB_PAUSE		0xD5
#define SB_INP_RATE		0x42
#define SB_16BIT_ADC_DMA	0xBE
#define SB_16BIT_SIGNED_MONO	0x10
#define SB_16BIT_SIGNED_STEREO	0x30

/* Card parameters */
unsigned int	sb_ioaddr;
unsigned int	sb_irq;
unsigned int	sb_dma16;
unsigned int	sb_type;

/* write a DSP command to SoundBlaster board */
#define sb_writedsp(x) {						\
	while (inportb(sb_ioaddr + SB_DSP_WRITE_STATUS) & 0x80) /* wait */;	\
	outportb(sb_ioaddr + SB_DSP_WRITE_DATA, (x));			\
}

/* Defines for 16-bit DMA Controller IO addresses */
#define SB_DMA16_BASE		(0xC0 + 4*(sb_dma16-4))
#define SB_DMA16_COUNT		(0xC2 + 4*(sb_dma16-4))
#define SB_DMA16_MASK		(0xD4)
#define SB_DMA16_MODE		(0xD6)
#define SB_DMA16_FF		(0xD8)
#define SB_DMA16_STOP_MASK	(sb_dma16 - 4 + 4)
#define SB_DMA16_START_MASK	(sb_dma16 - 4 + 0)
#define SB_DMA16_INP_MODE	(sb_dma16 - 4 + 0x54)
#define SB_DMA16_OUT_MODE	(sb_dma16 - 4 + 0x58)
static unsigned SB_DMA16_PAGE[8]={0x87,0x83,0x81,0x82,0,0x8B,0x89,0x8A};

/*  GO32 DPMI structures for accessing DOS memory. */
static _go32_dpmi_seginfo dosmem;   /* DOS (conventional) memory buffer */

/* Is a sound currently playing or recorded ? */
volatile int		sb_dma_active;

/* Conventional memory buffers for DMA. */
static char		*sb_buf[2];

/* store current settings */
static int	sb_sample_rate=8000;
static int	sb_stereo=0;

/* DMA chunk size, in bytes, must be power of 2 */
#define DMA_CHUNK (0x4000)	/* NOTE: I have found that small values
					may be allocated above 640k, and
					that this leads to problems */

/* read a value returned from DSP */
static int sb_read_dsp(int iobase)
{
	int	i;

	for (i=0;i<10000;i++)
	        if (inportb(iobase + SB_DSP_DATA_AVAIL) & 0x080)
			break;
	return(inportb(iobase+SB_DSP_READ_DATA));
}

/* reset SoundBlaster card */
static void sb_reset(void)
{
	int j;

	outportb(sb_ioaddr + SB_DSP_RESET,1);

	inportb(sb_ioaddr + SB_DSP_RESET);
	inportb(sb_ioaddr + SB_DSP_RESET);
	inportb(sb_ioaddr + SB_DSP_RESET);
	inportb(sb_ioaddr + SB_DSP_RESET);
    
	outportb(sb_ioaddr + SB_DSP_RESET, 0);
	for (j=0;j<100;j++)
		if (sb_read_dsp(sb_ioaddr) == 0xAA)
			 break;
}

#define SB_NUM_BASES	6
static int sbbase[SB_NUM_BASES]={0x210,0x220,0x230,0x240,0x250,0x260};

/* try and find card if port address not given */
static int sb_findcard(void)
{
	int i,j;

	/* search possible base addresses */
	for (i=0;i<SB_NUM_BASES;i++) {

		/* send a reset */
	        outportb(sbbase[i] + SB_DSP_RESET,1);

	        /* wait for a little while */
		inportb(sbbase[i] + SB_DSP_RESET);
		inportb(sbbase[i] + SB_DSP_RESET);
		inportb(sbbase[i] + SB_DSP_RESET);
		inportb(sbbase[i] + SB_DSP_RESET);

		/* request acknowledgement */
		outportb(sbbase[i] + SB_DSP_RESET, 0);

		/* look for reply */
		for(j=0;j<100;j++)
			if (sb_read_dsp(sbbase[i]) == 0xAA)
				/* got it ! */
				return(sbbase[i]);
	}
	/* nope */
	return(0);
}

/* Read soundblaster card parameters from BLASTER enivronment variable .*/
static int sb_getparams()
{
	char *t, *blaster;

	/* set Interrupt and DMA channel to usual defaults */
	sb_irq = 7;
	sb_dma16 = 5;

	/* assume port address unknown */
	sb_ioaddr = 0;

	/* see if environment variable set */
	if ((t=getenv("BLASTER"))) {

		/* take a copy */
		blaster = strdup(t);

		/* Parse the BLASTER variable */
		t = strtok(blaster, " \t");
		while (t && *t) {
			switch (t[0]) {
			case 'A':
			case 'a':
				/* I/O address */
				sscanf(t + 1, "%x", &sb_ioaddr);
				break;
			case 'I':
			case 'i':
				/* IRQ */
				sb_irq = atoi(t + 1);
				break;
			case 'H': 
			case 'h':
				/* 16bit DMA channel */
				sb_dma16 = atoi(t + 1);
				break;
			default:
				/* ignore everything else */
				break;
			}
			t = strtok(NULL," \t");
		}
		free(blaster);
        }

#ifdef EMO
printf("BLASTER: port=0x%X dma16=%d int=%d\n",sb_ioaddr,sb_dma16,sb_irq);
#endif

        if (sb_ioaddr==0)
        	return(sb_findcard());
        else
        	return(sb_ioaddr);
}

/* get DOS memory buffers */
static int sb_DOS_alloc()
{
	/* get a block in low memory big enough to hold
	   two buffers aligned on multiple of DMA_CHUNK */
	dosmem.size = (DMA_CHUNK*3)/16;
	if (_go32_dpmi_allocate_dos_memory(&dosmem)) {
		dosmem.size = -1;
		return(0);
	}

	(unsigned long) sb_buf[0] = dosmem.rm_segment * 16;
	(unsigned long) sb_buf[0] += (DMA_CHUNK-1);
	(unsigned long) sb_buf[0] &= ~(long)(DMA_CHUNK-1);
	(unsigned long) sb_buf[1] = (unsigned long) sb_buf[0] + DMA_CHUNK;

#ifdef EMO
	printf("DOS buffers at %08lX, %08lX\n",(unsigned long)(sb_buf[0]),
						(unsigned long)(sb_buf[1]));
#endif
	return(1);
}

/* free DOS memory */
static void sb_DOS_free()
{
	if (dosmem.size != -1)
		_go32_dpmi_free_dos_memory(&dosmem);
	dosmem.size = -1;
}

/* set SoundBlaster sample rate */
static void sb_set_sample_rate(unsigned int rate)
{
	if (rate < 5000)
		sb_sample_rate = 5000;
	else if (rate < 44100)
		sb_sample_rate = rate;
	else
		sb_sample_rate = 44100;

}

/* fill a DOS buffer */
static int sb_empty_buffer(int bno,short *buff,int numf)
{
	int	i;
	int	count;
	unsigned char	xbuf[DMA_CHUNK];

        dosmemget((unsigned long) sb_buf[bno], DMA_CHUNK, xbuf);

	/* copy a chunk from an unsigned byte buffer */
	count = (2*numf <= DMA_CHUNK) ? 2*numf : DMA_CHUNK;
	for (i=0;i<count;i+=2,buff++) {
		*buff = (short)((((unsigned short)(xbuf[i+1]))<<8) | xbuf[i]);
	}

	return(count/2);
}

/* close down card and DMA */
static void sb_halt_record()
{
	if (sb_dma_active) {
		sb_writedsp(SB_PAUSE);
		outportb(SB_DMA16_MASK, SB_DMA16_STOP_MASK);
	}
	sb_dma_active=0;
}
	
/* set DOS buffer recording repetitively */
static void sb_record_buffer()
{
	int     t;

	/* set DMA mode to write memory */
	outportb(SB_DMA16_MASK, SB_DMA16_STOP_MASK);
	outportb(SB_DMA16_FF, 0);
	outportb(SB_DMA16_MODE, SB_DMA16_INP_MODE);

	/* set DMA transfer address */
	t = (int) (((unsigned long) sb_buf[0] & 0x1FFFE) >> 1);
	outportb(SB_DMA16_BASE, t & 0xFF);
	outportb(SB_DMA16_BASE, t >> 8);
	t = (int) (((unsigned long) sb_buf[0] & 0xE0000) >> 16);
	outportb(SB_DMA16_PAGE[sb_dma16], t);

	/* set transfer length */
	outportb(SB_DMA16_COUNT, (DMA_CHUNK-1) & 0xFF);
	outportb(SB_DMA16_COUNT, (DMA_CHUNK-1) >> 8);

	/* OK set it going */
	outportb(SB_DMA16_MASK, SB_DMA16_START_MASK);

	/* set SoundBlaster to receive data */
	sb_writedsp(SB_INP_RATE);		/* specify sampling rate */
	sb_writedsp((sb_sample_rate & 0xFF00)>>8);
	sb_writedsp((sb_sample_rate & 0x00FF));
	sb_writedsp(SB_16BIT_ADC_DMA);		/* 16-bit acquisition */
	if (sb_stereo) {
		sb_writedsp(SB_16BIT_SIGNED_STEREO);	/* 16-bit stereo signed */
	}
	else {
		sb_writedsp(SB_16BIT_SIGNED_MONO);	/* 16-bit mono signed */
	}

	sb_writedsp((DMA_CHUNK/2-1) & 0xFF);   /* write length */
	sb_writedsp((DMA_CHUNK/2-1) >> 8);

	/* a sound is playing now. */
	sb_dma_active = 1;

	atexit(sb_halt_record);
}

/* wait for DMA counter to indicate half buffer done */
static int sb_wait_counter(int bno)
{
	int	len;
	int	done;

	len = inportb(SB_DMA16_COUNT);
	len |= inportb(SB_DMA16_COUNT) << 8;
	done = DMA_CHUNK - len - 1;

	if ((bno==1) && (done >= DMA_CHUNK/2)) {
		/* OK done buffer 0 - unload it */
		inportb(sb_ioaddr + SB_DSP_DATA_AVAIL16);
		return(0);
	}
	else if ((bno==0) && (done < DMA_CHUNK/2)) {
		/* OK done buffer 1 - unload it */
		inportb(sb_ioaddr + SB_DSP_DATA_AVAIL16);
		return(0);
	}
	return(1);
}

/* SoundBlaster initialise - interface routine */
int	SBR16Init(void)
{
	if (sb_getparams()==0)
		return(0);
	if (sb_DOS_alloc()==0)
		return(0);
	return(1);
}

/* record signed 16-bit data buffer */
int	SBR16Record(short *obuff,int numf,int srate,int nchan,int flags)
{
	int	count,len;
	int	bno;
	short	*buff = obuff;
	int	opos=0;
	int	owin=0;
	int	startpos=0;
	int	endpos=-1;
	int	i;
#ifdef EMO
	int	lasti=0;
#endif

	sb_reset();
	sb_set_sample_rate(srate);
	sb_stereo = (nchan>1);

	count = numf;
	bno = 1;

	/* start off replay */
	sb_record_buffer();

	/* while buffers waiting */
	while (count > 0) {
#ifdef EMO
		printf("recording buffer %d, samples left %d\n",1-bno,count);
#endif
		while (sb_wait_counter(bno)) /* loop */;
		bno = 1 - bno;		/* swap to next buffer */

		if (count > 0) {
			if (flags==0) {
				/* completely fill buffer */
				len = sb_empty_buffer(bno,buff,count);
				count -= len;
				buff += len;
			}
			else {
				/* do endpointing */
				len = sb_empty_buffer(bno,&obuff[opos],numf-opos);
				opos += len;
				count -= len;
				while ((owin+ep_windowsize) < opos) {
					switch (i=getendpoint(&obuff[owin])) {
					case EP_RESET:
					case EP_SILENCE:
						/* HACK by MAH 961028 */
						startpos = owin /* +ep_stepsize */;
						break;
					case EP_MAYBEEND:
						/* HACK by MAH 961028 */
						endpos = owin + ep_windowsize;
					case EP_SIGNAL:
						break;
					case EP_NOTEND:
						endpos = -1;
						break;
					case EP_ENDOFUTT:
						goto done;
					case EP_NOSTARTSILENCE:
					case EP_NONE:
						break;
					}
#ifdef EMO
if (i!=lasti)
printf("ep_state=%d\n",i);
lasti=i;
#endif
					owin += ep_stepsize;
				}
				if (startpos!=0) {
					/* reset buffer to start */
					memcpy(&obuff[0],&obuff[startpos],(opos-startpos)*sizeof(short));
					opos -= startpos;
					owin -= startpos;
					endpos -= startpos;
					count = numf-opos;
					startpos = 0;
				}
			}
		}
		if (kbhit())
			if (getch() == 27) 
				break;
	}
done:
	sb_halt_record();
	sb_reset();
	if (flags) {
		if (endpos <= 0)
			return(opos);
		else {
			if ((endpos + ep_windowsize) < opos)
				return(endpos+ep_windowsize);
			else
				return(opos);
		}
	}	
	else
		return(numf-count);
}

void	SBR16Close(void)
{
	sb_DOS_free();
}

#ifdef EMO
short	sbuf[100000];
void	main(int argc,char *argv[])
{
	int	i;
	FILE	*op;

	if (SBR16Init()) {
		printf("Speak to start .... \n");
		initparams(20000.0);

		i = SBR16Record(sbuf,100000,20000,1,1);
		printf("recorded %d samples\n",i);
		SBR16Close();

		op=fopen("sbr16dma.dat","wb");
		fwrite(sbuf,2,i,op);
		fclose(op);

		for (i=0;i<50;i++) printf("%d, ",sbuf[i]);
		printf("\n");
	}
	exit(0);
}
#endif
