/* anlist -- list an annotation item to standard output in form required by anload */

/* M.A.Huckvale - November 1988 */

/* version 1.0 */
/* version 1.1 - August 1989
	- add sample number outputs
*/
/* version 1.2 - March 1992
	- add -h HTK format output
*/
/* version 1.3 - July 1992
	- add -m anmap annotation mapping on output
*/
/* version 1.4 - September 1992
	- add -c flag to collapse adjacent identical output annotations
*/
	
/*-------------------------------------------------------------------*/
/**MAN
.TH ANLIST SFS1 UCL
.SH NAME
anlist -- list an annotation item to a text description
.SH SYNOPSIS
.B anlist
(-i item) (-s rate) (-h) (-m anmapfile) (-c) file
.SH DESCRIPTION
.I anlist
dumps the contents of an annotation item onto the standard output
in a form suitable for editting and reloading with
.I anload(SFS1).
The standard format of the output is one annotation per line, with two
entries: firstly the time in seconds, secondly the text label.
.PP
.B Options
.TP 11
.B -I
Identify the program version.
.TP 11
.BI -i item
Select input item.
.TP 11
.BI -s rate
Output sample numbers pre-supposing given sampling rate.  This output
is not compatible with
.I anload.
.TP 11
.B -h
Output annotations in HTK format, with start and stop times
expressed in 100ns units.
.TP 11
.BI -m anmapfile
Map any annotations that match lines in the supplied file.  Format as for
.I anmap(SFS1)
prograrm, namely two fields per line, input and output.
.TP 11
.B -c
Collapse together adjacent annotations sharing the same output label.
.SH INPUT ITEMS
.IP AN 11
Any annotation item
.SH VERSION/AUTHOR
1.4 - Mark Huckvale
.SH BUGS
The sample mode of anlist is not compatible with the sample mode of
anload.
*/
/*---------------------------------------------------------------*/

/* program name and version */
#define	PROGNAME "anlist"
#define PROGVERS "1.4"
char	*progname = PROGNAME;

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <malloc.h>
#include <string.h>
#include "sfs.h"		/* database filing system structures */
char	*strsave();

/* global data */
char			filename[SFSMAXFILENAME];	/* dbase file name */
struct item_header 	anitem;			/* annotation item header  */
struct an_rec		*an;			/* annotation record */
char			*antype="0";		/* annotation type */
double			srate= -1;
int			htkformat=0;		/* output in HTK format */
int			domap=0;		/* do annotation mapping */
int			dojoin=0;		/* join adjacent duplicates */

/* annotation mapping */
char	**anin,**anout;
int	nummap;
char	lastan[256];

/* map file line */
char	mline[256];

/* save string in dynamic memory */
char	*strsave(s)
char	*s;
{
	char	*p;
	int	len=strlen(s)+1;
	if ((p=malloc(len))==NULL)
		error("cannot save string",NULL);
	strcpy(p,s);
	return(p);
}

/* binary string search */
int strfind(s,t,num)
char	*s;
char	*t[];
int	num;
{
	int	i,j,k;
	int	c;
	
	if (num==0)
		return(-1);
	else {
		i=0;
		j=num-1;
		do {
			k=(i+j)/2;
			if ((c=strcmp(s,t[k])) > 0)
				i=k+1;
			else
				j=k-1;
		} while (c && (i <= j));
		if (c)
			return(-1);
		else
			return(k);
	}
}

/* maintain string table */
void strtable(s,t,num)
char	*s;
char	*t[];
int	*num;
{
	int	i;
	
	/* see if string in table */
	if (strfind(s,t,*num) < 0) {
		/* add to table */
		i = *num;
		while ((i>0) && (strcmp(s,t[i-1])<0)) {
			t[i] = t[i-1];
			i--;
		}
		t[i]=strsave(s);
		(*num)++;
	}
}

/* get entry from map file */
int	getmapentry(ip,ilab,olab)
FILE	*ip;
char	*ilab;
char	*olab;
{
	char	*p;

	if (fgets(mline,256,ip)) {
		/* get input */
		p=strtok(mline," \t\n");
		if (p && *p)
			strcpy(ilab,p);
		else
			*ilab='\0';
		/* get output */
		p=strtok(NULL," \t\n");
		if (p && *p)
			strcpy(olab,p);
		else
			*olab='\0';
		return(1);
	}
	return(0);
}

void processmap(mapname)
char	*mapname;
{
	char	ilab[256],olab[256];		/* temporary storage */
	FILE	*ip;
	int	idx;
	
	/* open map file */
	if ((ip=fopen(mapname,"r"))==NULL)
		error("cannot open map file '%s'",mapname);

	/* get number of annotations from map */
	nummap=0;
	while (getmapentry(ip,ilab,olab)) nummap++;

	/* get space for annotation table */
	if ((anin=(char **)calloc(nummap,sizeof(char *)))==NULL)
		error("cannot get memory for annotation table",NULL);
	if ((anout=(char **)calloc(nummap,sizeof(char *)))==NULL)
		error("cannot get memory for annotation table",NULL);
	
	/* get list of input annotations from map */
	nummap=0;
	rewind(ip);
	while (getmapentry(ip,ilab,olab))
		strtable(ilab,anin,&nummap);

	/* save output annotations */
	rewind(ip);
	while (getmapentry(ip,ilab,olab)) {
		idx=strfind(ilab,anin,nummap);
		anout[idx]=strsave(olab);
	}
	fprintf(stderr,"%d annotation mappings loaded from '%s'\n",nummap,mapname);
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;			/* option index */
	extern char	*optarg;		/* option argument ptr */
	int		errflg = 0;		/* option error flag */
	int32		it;			/* item type */
	char		*ty="0";		/* sub-type = last */
	int		c;			/* option switch */
	double		atof();

	/* processing variables */
	int		i,j;
	int		fid;			/* input file descriptor */
	double		tim,etim;
	int		idx;
	
	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:s:hm:c")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: List AN item V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* input specification */
			if (itspec(optarg,&it,&ty)==0) {
				if (it==AN_TYPE)
					antype=ty;
				else
					error("unsuitable item specification: '%s'",optarg);
			}
			else
				error("illegal item specification: '%s'",optarg);
			break;
		case 's' :	/* sampling rate */
			srate = atof(optarg);
			if (srate <= 0)
				error("bad sampling rate '%s'",optarg);
			break;
		case 'h' :	/* HTK format output */
			htkformat++;
			break;
		case 'm' :	/* map annotations */
			processmap(optarg);
			domap++;
			break;
		case 'c' :	/* join adjacent duplicates */
			dojoin++;
			break;
		case '?' :	/* unknown */
			errflg++;
	}

	/* check command line */
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-s rate) (-h) (-m anmap) (-c) file",PROGNAME);

	/* get data filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open data file */
	if ((fid=sfsopen(filename,"r",NULL)) < 0) {
		if (fid==-1)
			error("cannot find file %s",filename);
		else
			error("access error on %s",filename);
	}

	/* locate input item */
	if (!sfsitem(fid,AN_TYPE,antype,&anitem))
		error("unable to find input annotations in '%s'",filename);

	/* get output record buffer */
	if ((an=(struct an_rec *)sfsbuffer(&anitem,anitem.numframes))==NULL)
		error("could not get memory buffer",NULL);

	/* read in all annotations */
	if (sfsread(fid,0,anitem.numframes,an)!=anitem.numframes)
		error("read error on input");

	/* perform mapping */
	if (domap) for (i=0;i<anitem.numframes;i++) {
		if ((idx=strfind(an[i].label,anin,nummap)) >= 0)
			an[i].label = anout[idx];
	}

	/* collapse adjacencies */
	if (dojoin) {
		for (i=1,j=0;i<anitem.numframes;i++) {
			if (strcmp(an[j].label,an[i].label)==0)
				an[j].size += an[i].size;
			else {
				j++;
				an[j].posn = an[i].posn;
				an[j].size = an[i].size;
				an[j].label = an[i].label;
			}
		}
		anitem.numframes = j+1;
	}

	/* print annotations */
	for (i=0;i<anitem.numframes;i++) {
		tim = anitem.offset + an[i].posn*anitem.frameduration;
		if (an[i].label[0]) {
			if (htkformat) {
				etim = tim + an[i].size*anitem.frameduration;
				if ((an[i].label[0]=='-') || (an[i].label[0]=='/') ||
					((an[i].label[0]>='0') && (an[i].label[0]<='9')))
					printf("%d %d L%s\n",
						(int)(0.5+tim*10000000L),
						(int)(0.5+etim*10000000L),
						an[i].label);
				else
					printf("%d %d %s\n",
						(int)(0.5+tim*10000000L),
						(int)(0.5+etim*10000000L),
						an[i].label);
			}
			else if (srate > 0)
				printf("%8d %s\n",(int)(0.5+tim*srate),an[i].label);
			else
				printf("%10.5f %s\n",tim,an[i].label);
		}
	}

	/* that's all folks */
	sfsclose(fid);
	exit(0);

}

