/* remove -- remove an individual data item from a file */

/* M.A.Huckvale - December 1985 */

/* version 1.2 - checks file needs changing before copying */
/* version 2.0 - new format history and access */
/* version 2.1 - fix bug in inter-relationships */
/* version 3.0 - SFS revision */
/* version 3.1 - November 1987
	- query on pattern match
	- -n flag
*/
/* version 3.2 - January 1988
	- header validation for corrupt files
*/
/* version 3.3 - february 1988
	- ensure file mode & ownership across backup
*/
/* version 3.4 - May 1994
	- fix bug in deletion of files with mixed machine type
*/
/* version 3.5 - April 1995
	- add -b switch (delete 'but')
*/

/*--------------------------------------------------------------------------*/
/**MAN
.TH REMOVE SFS1 UCL
.SH NAME
remove - delete dataset(s) from file
.SH SYNOPSIS
.B remove
(-i item) (-a item) (-b item) (-e) (-s) (-n) file
.SH DESCRIPTION
.I remove
is a program to delete specified datasets from database files.  Use the "-i"
option to delete the first dataset matching a specification.  Use "-a item" to delete 
.I all
datasets matching a specification.  Use "-b item" to remove everything but
the items specified.
Thus "remove -i FX file" would remove
the last fx item while "remove -a FX file" would remove all fx items.
While "remove -bfx." would remove all but the first FX item in the file.
If
no items are specified the program enters a "prompted" mode.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select first item matching item specification.
.TP 11
.BI -a type
Select all items matching item specification.
.TP 11
.BI -b type
Select all items except those matching item specification.
.TP 11
.B -e
Remove 
.I everything
from the file except the main header.
.TP 11
.B -s
Delete selected items and all 
.I subsequent
items that have been processed from them.
.TP 11
.B -n
Do not query deletion of multiple pattern matches.
.SH INPUT ITEMS
.IP all 11
Any dataset may be deleted.
.SH VERSION/AUTHOR
3.5 - Mark Huckvale.
.SH SEE ALSO
scopy(SFS1), summary(SFS1)
*/
/*--------------------------------------------------------------------------*/
#define PROGNAME "remove"
#define PROGVERS "3.5"

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#ifdef DOS
#include <io.h>
#endif
#include <ctype.h>
#include <malloc.h>
#include "sfs.h"
#include "sfsdata.h"

/* item selection data */
#define MAXSELECT 50
struct {
	int32	it;
	int32	ty;
	char	*match;
	int	all;
	int	but;
	int	query;
} itab[MAXSELECT];
int	itcnt=0;

struct item_header item;
char	*tmpfilename=NULL;

int	somebut=0;
int	everything=0;
int	promptmode=1;
int	subseqmode=0;
int	querymode=1;

#ifdef __STDC__
int itemvalid(struct item_header *item);
int relate(int	fid);
int itrequired(struct item_header *item,char *hist,int 	flag);
static int upd_exist(char *s,char *t);
void process(int inp,int out);
int prompt(struct item_header *item);
static int upd_copy(int len,int ip,int op);
#else
int itemvalid();
int relate();
int itrequired();
static int upd_exist();
void process();
int prompt();
static int upd_copy();
#endif

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	/* file variables */
	char		filename[SFSMAXFILENAME]; /* dbase file name */
	int		fid,ofid;
	int		pass1=0;
	char		*exphist,*lab_store();

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:a:b:sen")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: SFS item deletion V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&itab[itcnt].it,&itab[itcnt].match) != 0)
				error("bad 'i' option : %s",optarg);
			if (strcmp(itab[itcnt].match,"0")==0) pass1++;
			itcnt++;
			promptmode=0;
			break;
		case 'b' :	/* but specific item */
			if (itspec(optarg,&itab[itcnt].it,&itab[itcnt].match) != 0)
				error("bad 'b' option : %s",optarg);
			if (strcmp(itab[itcnt].match,"0")==0) pass1++;
			itab[itcnt].but=1;
			itcnt++;
			somebut++;
			promptmode=0;
			break;
		case 'a' :	/* generic item */
			if (itspec(optarg,&itab[itcnt].it,&itab[itcnt].match) != 0)
				error("bad 'a' option : %s",optarg);
			itab[itcnt].all=1;
			itab[itcnt].query=1;
			if (strcmp(itab[itcnt].match,"0")==0) {
				itab[itcnt].match = "*";
				itab[itcnt].query=0;
			}
			itcnt++;
			promptmode=0;
			break;
		case 'e' :	/* remove everything */
			everything++;
			promptmode=0;
			break;
		case 's' :	/* subsequent mode */
			subseqmode++;
			break;
		case 'n' :	/* no questions */
			querymode=0;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	/* check for command line errors */
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-a item) (-b item) (-e) (-s) (-n) dbase_file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,argv[optind]);
	else
		error("no database file specified",NULL);

	/* open datafile */
        if ((fid = sfsopen(filename,"w",NULL)) < 0) {
                if (fid == -1)
                        error("cannot open %s",filename);
                else
                        error("access error on %s",filename);
        }

	/* first pass for specially selected items */
	if (pass1) {
	        while (sfsnextitem(fid,&item)) {
			exphist = lab_store(&item);
			itrequired(&item,exphist,0);
		}
		close(fid);
	        fid = sfsopen(filename,"r",NULL);
	}

	/* build history inter-relationships and delete table */
	if (!relate(fid)) {
		fprintf(stderr,"remove: nothing deleted\n");
		exit(0);
	}

	/* create a new file to hold results of processing */
	if ((ofid = sfschannel(filename,&item)) < 0)
		error("could not open temporary file",NULL);
	tmpfilename = sfsdata[ofid]->ofilename;

	/* process datafile into temporary file */
	process(fid,ofid);

	/* close files */
	sfsclose(fid);
	close(ofid);

	/* transfer results */
	if (!upd_backup(tmpfilename,filename))
		error("backup error on %s",filename);

	/* that's all folks */
	exit(0);
}

/* validate an item header */
int itemvalid(item)
struct item_header *item;
{
	if (	(item->history[0] < 32) ||
		(item->history[0] > 126) ||
		(item->length < 0) ||
		(item->datatype > 10000) ||
		(item->datatype < -10000) ||
		(item->subtype < 0) ||
		(item->subtype > 10000) ||
		(item->frameduration < 0)
	    )
		return(0);	/* fail */
	else
		return(1);	/* pass .. probably */
}

/* build history inter-relationships */
static struct {
	char	*history;
	char	itemno[8];
	int	datatype;
	int	del;
	int	used;
} current[MAXITEM];
static int	curcount=0;

int relate(fid)
int	fid;
{
	struct item_header item;
	int	i,j,flag;
	char	*exphist,*lab_store();

	for (i=0;i<MAXITEM;i++) {
		current[i].used=0;
		current[i].del=0;
	}

	/* scan file to determine relationships */
	flag=0;
	while (sfsnextitem(fid,&item)) {
		if (!itemvalid(&item)) {
			flag++;
			break;
		}
		exphist = lab_store(&item);
		current[curcount].datatype=item.datatype;
		if (item.datatype < 0) {
			/* include item stubs, but mark for deletion */
			item.datatype=abs(item.datatype);
			current[curcount].del++;
		}
		/* save history */
		current[curcount].history=malloc(strlen(item.history)+1);
		strcpy(current[curcount].history,item.history);
		/* check whether item is required to be deleted */
		if (promptmode) {
			current[curcount].del=prompt(&item);
			flag += current[curcount].del;
		}
		else if ((i=itrequired(&item,exphist,1)) >= 0) {
			if ((querymode) && (itab[i].query))
				current[curcount].del=prompt(&item);
			else
				current[curcount].del++;
			flag++;
			current[curcount].datatype=abs(current[curcount].datatype);
		}
		/* generate itemno */
		sprintf(current[curcount].itemno,"%d.%02d",item.datatype,item.subtype);
		/* check for deletion because of subsequent mode */
		if (subseqmode) for (i=0;i<curcount;i++) {
			if ((current[i].del) && (current[i].datatype > 0) &&
			    upd_exist(current[i].itemno,item.history)) {
				current[curcount].del++;
				flag++;
			}
		}
		/* determine antecedents of this item */
		for (i=0;i<curcount;i++) {
			if (upd_exist(current[i].itemno,item.history))
				current[i].used++;
		}
		curcount++;
	}
	/* scan to remove deleted whole branches */
	for (j=curcount-1;j>=0;j--) {
		if ((current[j].del) && (current[j].used==0)) {
			for (i=0;i<j;i++) {
				if (upd_exist(current[i].itemno,current[j].history))
					current[i].used--;
			}
		}
	}
	return(flag);
}

/* check if item required */
int itrequired(item,hist,flag)
struct item_header 	*item;
char			*hist;
int			flag;		/* 0=first pass; 1=second pass */
{
	int	i,ty,req= -1;
	char	junk[16];

	if (everything) return(0);

	for (i=0;i<itcnt;i++) if ((itab[i].it==0) || (itab[i].it==item->datatype)) {
		if (itab[i].ty==item->subtype)
			req=(itab[i].but)?-2:i;
		else if (isdigit(itab[i].match[0]) || (itab[i].match[0]=='-')) {
			ty=atoi(itab[i].match);
			if ((ty==-1) || (ty==item->subtype)) {
				if (itab[i].all==0) itab[i].ty=item->subtype;
				req=(itab[i].but)?-2:i;
			}
			else if ((ty==0) && (flag==0)) {
				itab[i].it=abs(item->datatype);
				itab[i].ty=item->subtype;
			}
		}
		else if (itab[i].ty==0) {
			if (histmatch(itab[i].match,hist,junk,junk,junk,junk,junk,junk,junk,junk,junk,junk)==1) {
				if (itab[i].all==0) itab[i].ty=item->subtype;
				req=(itab[i].but)?-2:i;
			}
		}
	}
	if (somebut && (req==-1)) req=itcnt;
	return(req);
}

/* check existence of itemno string in history */
static int upd_exist(s,t)
char	*s,*t;
{
	register int	i,j,flag;
	int	ls,lt;
	ls=strlen(s);
	lt=strcspn(t,";=")-ls;
	for (i=1;i<=lt;i++) {
		if (((t[i-1]=='(') || (t[i-1]==',')) && (s[0]==t[i])) {
			flag=1;
			for (j=1;j<ls;j++) if (s[j]!=t[i+j]) flag=0;
			if ((t[i+j]!=')') && (t[i+j]!=';') && (t[i+j]!=',')) flag=0;
			if (flag) return(1);
		}
	}
	return(0);
}

/* process data file into temporary file */
void process(inp,out)
int	inp,out;
{
	struct main_header head;
	struct item_header item,nitem;
	int	skip,len;

	/* copy main header */
	lseek(inp,0L,0);
	if (read(inp,&head,sizeof(struct main_header)) != sizeof(struct main_header))
		error("read error on main header",NULL);
	if (write(out,&head,sizeof(struct main_header)) != sizeof(struct main_header)) 
		error("write error on temporary file",NULL);

	/* copy undeleted items */
	curcount=0;
	while (read(inp,&item,sizeof(struct item_header)) == sizeof(struct item_header)) {
	        nitem = item;
	        if (item.machine!=SFSMACHINE) sfsconvi(&nitem);
		if (!itemvalid(&nitem)) break;
		/* operations: 0=write as norm, 1=truncate, 2=delete */
		if (current[curcount].del==0)
			skip=0;		/* no deletion required */
		else if (current[curcount].used==0)
			skip=2;		/* complete deletion required */
		else
			skip=1;		/* leave item stub */
		if (skip==2) {
			lseek(inp,nitem.length,1);
		}
		else if (skip==1) {
			nitem.datatype= -(abs(nitem.datatype));
			len=nitem.length;
			nitem.length=0;
			nitem.datapresent=0;
			nitem.machine=SFSMACHINE;
			if (write(out,&nitem,sizeof(struct item_header)) != sizeof(struct item_header)) 
				error("write error on temporary file",NULL);
			lseek(inp,len,1);
		}
		else /* skip == 0 */ {
			if (write(out,&item,sizeof(struct item_header)) != sizeof(struct item_header)) 
				error("write error on temporary file",NULL);
			if (!upd_copy(nitem.length,inp,out))
				error("write error on temporary file",NULL);
		}
		curcount++;
	}
}

/* prompt -- get y/n response from user */
int prompt(item)
struct item_header *item;
{
	char	ans[80];
	extern char *sfsname[];

	if ((item->datatype > -(MAXDATATYPE+1)) && (item->datatype < MAXDATATYPE+1)) 
		printf("%8s",sfsname[abs(item->datatype)]);
	else
		printf("type %3d",item->datatype);
	if (item->datapresent == 0)
		printf(" -");
	else
		printf("  ");
	if (item->datatype < 10)
		printf(" (%d.%02d)",abs(item->datatype),item->subtype);
	else
		printf("(%2d.%02d)",abs(item->datatype),item->subtype);
	printf(" %5d frames",item->numframes);
	printf(" from %s; delete (y/n) ? : ",item->history);
	fflush(stdout);
	if (gets(ans) == NULL) {
		printf("\n");
		error("aborted",NULL);
	}
	if ((ans[0]=='y') || (ans[0]=='Y'))
		return(1);
	else
		return(0);
}

#define COPYBUFSIZE 2048
#define MIN(x,y) (((x)<(y))?(x):(y))

static int upd_copy(len,ip,op)
int	len;
int	ip,op;
{
	char	buf[COPYBUFSIZE];
	int	count=0;

	while (len > 0) {
		count = read(ip,buf,MIN(COPYBUFSIZE,len));
		if (count==0) return(0);
		if (write(op,buf,count) != count) return(0);
		len -= count;
	}
	return(1);
}

