/* fxmomel - model fx contour using MOMEL algorithm */

/* M.A.Huckvale - University College London */

/* version 1.0 - April 2000 */

#define PROGNAME "fxmomel"
#define PROGVERS "1.0"
char	*progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH FXMOMEL SFS1 UCL
.SH NAME
fxmomel -- model fx contour using MOMEL algorithm
.SH SYNOPSIS
.B fxmomel
(-i item) (-f) file
.SH DESCRIPTION
.I fxmomel
is a program to model a fundamental frequency contour using a
minimum number of turning points on a quadratic spline fit.
The algorithm follows the basic design by Daniel Hirst and
Robert Espesser (Laboratoire Parole et Langage CNRS-ESA 6057,
Aix-en-Provence).
The output is a set of annotations at the times of the turning
points with labels equal to the fundamental frequency values.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version number.
.TP 11
.BI -i item
Select input item number.
.TP 11
.B -f
Also save the modelled fx contour.
.SH INPUT ITEMS
.IP 4.xx 11
Any fundamental frequency item.
.SH OUTPUT ITEMS
.IP 5 11
Momel annotations.
.SH VERSION/AUTHOR
1.0 - Mark Huckvale
*/
/*--------------------------------------------------------------------------*/

/* standard definitions */
#include "SFSCONFG.h"
#include <stdio.h>		/* standard io library */
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <malloc.h>
#include "sfs.h"		/* database structures */

/* global data */
struct item_header fxitem;	/* item header for fx data */
short		*fx;		/* buffer for fx data */
struct item_header ofxitem;	/* output item header */
short		*ofx;
struct item_header anitem;	/* item header for an data */

int verbose=0;

/* MOMEL parameters */
#define	MINFX	50
#define MAXFX	600
#define WIN1	0.3
#define WIN2	0.2
#define TOL1	1.05
#define TOL2	0.05

#define SEUILV 50

#define FAUX 0
#define VRAI 1
#define FSIGMA 1.

#define PAS_TRAME 10.
#define ECART_MIN 50. /*in ms*/
#define RAPP_MIN_FREQ 0.05

/* prototype targets */
struct st_cib {
	double x;
	double y;
};

struct st_cibred {
	double x;
	double y;
	int p;
};

struct st_cib *cib;
struct st_cib *cibout;
int	ncibout;

/* preparation */
void pref(double cfiltre,double hzvois)
{
	int	i;

	for (i = 1; i < fxitem.numframes - 1; i++) {
		if (fx[i] < hzvois)
			fx[i] = 0;
		else {
			if (((double)fx[i] / (fx[i - 1] + 1.) > cfiltre) &&
				((double)fx[i] / (fx[i + 1] + 1.) > cfiltre)) {
				fx[i] = 0;
			}
		}
	}
if (verbose) {
	printf("After pref:\n");
	for (i=0;i<fxitem.numframes;i++) printf("%d\n",fx[i]);
}
}

/* quadratic fitting */
int calcrgp (double *hz, char *pond, int dpx, int fpx, double *pa0, double *pa1, double *pa2, double *hzes)
{
	double muet;
	double pn;
	double val_ix;
	double x2, x3, x4, x2y, xy;
	double sx, sx2, sx3, sx4, sy, sxy, sx2y;
	double spdxy, spdx2, spdx3, spdx4, spdx2y;
	double p;
	double y;
	int ix;

	pn = 0.;
	sx = sx2 = sx3 = sx4 = sy = sxy = sx2y = 0.;

	for (ix = dpx; ix <= fpx; ix++) {
		p = pond[ix];
		if (p != 0.) {
			val_ix = (double) ix;
			y = hz[ix];
			x2 = val_ix * val_ix;
			x3 = x2 * val_ix;
			x4 = x2 * x2;
			xy = val_ix * y;
			x2y = x2 * y;

			pn += p;
			sx += (p * val_ix);
			sx2 += p * x2;
			sx3 += p * x3;
			sx4 += p * x4;
			sy += p * y;
			sxy += p * xy;
			sx2y += p * x2y;
		}
	}
	if (pn < 3.)
		return (1);

	spdxy = sxy - (sx * sy) / pn;
	spdx2 = sx2 - (sx * sx) / pn;
	spdx3 = sx3 - (sx * sx2) / pn;
	spdx4 = sx4 - (sx2 * sx2) / pn;
	spdx2y = sx2y - (sx2 * sy) / pn;

	muet = spdx2 * spdx4 - spdx3 * spdx3;
	if (spdx2 == 0.|| muet == 0.)
		return (1);

/*fprintf(stderr,"avant calcul pa210 muet %g spdx2 %g pn %g\n",muet,spdx2,pn);*/

	*pa2 = (spdx2y * spdx2 - spdxy * spdx3) / muet;
	*pa1 = (spdxy - *pa2 * spdx3) / spdx2;
	*pa0 = (sy - *pa1 * sx - *pa2 * sx2) / pn;

	for (ix = dpx; ix <= fpx; ix++)
		hzes[ix] = *pa0 + (*pa1 + *pa2 * (double) ix) * (double) ix;

	return (0);
}

/* cible - generate turning points */
void cible(int lfen, double hzinf, double hzsup, double	maxec)
{
	double *bhz;
	char *bpond;
	double *bhzes;
	char *bpondloc;
	double *hz, *hzes;
	char *pond;

	double a0, a1, a2;
	double xc, yc;
	double vxc, vyc;

	int ix, x;
	int dpx, fpx;
	int nsup, nsupr;
	int lfens2;
	int retour_rgp;

	/* get memory */
	bhz = (double *)calloc(200+fxitem.numframes,sizeof(double));
	bpond = (char *)calloc(200+fxitem.numframes,sizeof(char));
	bhzes = (double *)calloc(200+fxitem.numframes,sizeof(double));
	bpondloc = (char *)calloc(200+fxitem.numframes,sizeof(char));
	cib = (struct st_cib *)calloc(fxitem.numframes,sizeof(struct st_cib));
	hz = bhz + 100;
	pond = bpond + 100;
	hzes = bhzes + 100;

	/* hz[] contains the Hz values of f0 in doubles */
	for (ix=0;ix<fxitem.numframes;ix++)
		hz[ix] = fx[ix];

	for (ix = 0; ix < fxitem.numframes; ix++) {
		if (hz[ix] > SEUILV)
			pond[ix] = 1;
	}

	lfens2 = lfen / 2;

	if (verbose) printf("prototype targets:\n");

	for (ix = 0; ix < fxitem.numframes; ix++) {
		dpx = ix - lfens2;
		fpx = dpx + lfen;
		nsup = 0;
		nsupr = -1;

		{
		char *pondloc;
		int i;

		pondloc = bpondloc + 100;
		/* recopie */
		for (i = dpx; i <= fpx; i++)
			pondloc[i] = pond[i];

		while (nsup > nsupr) {
			nsupr = nsup;
			nsup = 0;
			retour_rgp = calcrgp (hz, pondloc, dpx, fpx, &a0, &a1, &a2, hzes);
			if (retour_rgp != 0)
				break; /* quit  while */

			for (x = dpx; x <= fpx; x++) {
				if (hz[x] == 0.|| hzes[x] / hz[x] > maxec) {
					pondloc[x] = 0;
					nsup++;
				}
			}
		} /* while */

		} /* block */

		xc = yc = 0.;
		if (retour_rgp == 0 && a2 != 0.) { /* tout va bien */
			vxc = -a1 / (a2 + a2);
			if ((vxc > ix - lfen) && (vxc < ix + lfen)) {
				vyc = a0 + (a1 + a2 * vxc) * vxc;
				if (vyc > hzinf && vyc < hzsup) {
					xc = vxc;
					yc = vyc;
				}
			}
		}


		if (verbose)
			printf("ret %d a0 %g a1 %g a2 %g\n", retour_rgp, a0, a1, a2);


		/* save one target point per hz value */
		cib[ix].x = xc;
		cib[ix].y = yc;

		if (verbose) printf("%g\t%g\n",xc,yc);

	} /* for ix */

	/* free memory */
	free(bhz);
	free(bpond);
	free(bhzes);
	free(bpondloc);
}

/* comparison function for qsort: increasing temporal order */
int cb_compare(const void *va, const void *vb)
{
	struct st_cib *pa = (struct st_cib *) va;
	struct st_cib *pb = (struct st_cib *) vb;

	if (pa->x > pb->x)
		return (1);
	if (pa->x < pb->x )
		return (-1);
	return (0);
}

/* reduction to minimum number of turning points */
void reduc(int lfen, double seuildiff_x, double	seuilrapp_y)
{
	struct st_cibred *cibred;
	struct st_cibred *cibred2;

	struct st_cibred cibred_cour ;

	double *xdist;
	double *ydist;
	double *dist;
	int *xd;

	int i;
	int j;
	int j1, j2;
	int ng, nd;
	int ncible;
	int n, np, xmax;
	int susseuil;

	double px, py;
	double xds, yds;
	double sxg, syg, sxd, syd;

	double seuil;

	int lf;

	/* allocate memory */
	cibred = (struct st_cibred *)calloc(fxitem.numframes,sizeof(struct st_cibred));
	cibred2 = (struct st_cibred *)calloc(fxitem.numframes,sizeof(struct st_cibred));
	cibout = (struct st_cib *)calloc(fxitem.numframes,sizeof(struct st_cib));
	ncibout=0;
	xdist = (double *)calloc(fxitem.numframes,sizeof(double));
	ydist = (double *)calloc(fxitem.numframes,sizeof(double));
	dist = (double *)calloc(fxitem.numframes,sizeof(double));
	xd = (int *)calloc(fxitem.numframes,sizeof(int));

	lf = lfen / 2;
	xds = yds = 0.;
	np = 0;
	for (i = 0; i < fxitem.numframes - 1; i++) {
		j1 = 0;
		if (i > lf)
			j1 = i - lf;
		j2 = fxitem.numframes - 1;
		if (i + lf < fxitem.numframes - 1)
			j2 = i + lf;
		sxg = syg = 0.;
		ng = 0;
		for (j = j1; j < i + 1; j++) /* left */ {
			if (cib[j].y > SEUILV) {
				sxg += cib[j].x;
				syg += cib[j].y;
				ng++;
			}
		}
		sxd = syd = 0.;
		nd = 0;
		for (j = i + 1; j < j2; j++) {
			if (cib[j].y > SEUILV) {
				sxd += cib[j].x;
				syd += cib[j].y;
				nd++;
			}
		}

		xdist[i] = ydist[i] = -1.;
		if (nd * ng > 0) {
			xdist[i] = fabs (sxg / ng - sxd / nd);
			ydist[i] = fabs (syg / ng - syd / nd);
			xds += xdist[i];
			yds += ydist[i];
			np++;
		}
	} /* i loop */

	/* weighted by 1/ mean distance */

	px = np / xds;
	py = np / yds;

	for (i = 0; i < fxitem.numframes; i++) {
		dist[i] = -1;
		if (xdist[i] > 0.) {
			dist[i] = (xdist[i] * px + ydist[i] * py) / (px + py);
		}
	}

	/* seuil (threshold) = mean of weighted distances */

	seuil = 2. / (px + py);

	xd[0] = 0;
	ncible = 0;
	susseuil = FAUX;

	/* look for max peaks where dist > seuil */

	for (i = 0; i < fxitem.numframes; i++) {

		if (susseuil == FAUX) {
			if (dist[i] > seuil) {
				susseuil = VRAI;
				xmax = i;
			}
		}
		if (susseuil == VRAI) {
			if (dist[i] > dist[xmax]) {
				xmax = i;
			}
			if (dist[i] < seuil) {
				ncible++;
				xd[ncible] = xmax;
				susseuil = FAUX;
			}
		}
	} /*  i loop */

	if (susseuil == VRAI) {
		ncible++;
		xd[ncible] = xmax;
	}

	if (xmax < fxitem.numframes) {
		ncible++;
		xd[ncible] = fxitem.numframes;
	}

	/* ncibles = number of partitions
	*  [0 to ncible] beginning of partition xd[i]
	*/

	if (verbose) {
		int k;

		printf(" ncible %d partitions\n", ncible);
		for (k = 0; k < ncible; k++)
			printf("k %d debut %d a %d\n", xd[k], xd[k + 1]);
	}


	/**********************
	*partition on x
	*/

	{
		int ip, j;
		int k, ncibr;
		double sx, sx2, sy, sy2;
		double varx, vary;
		double xm, ym;
		double et2x, et2y;
		double seuilbx, seuilby, seuilhx, seuilhy;
		int parinf, parsup;

		int ncibr_brut ;
		int nred2;

		/* ncibr points to preceding reduced target
		* end with items idexed [0 to ncibr]
		*/
		ncibr = -1;
		for (ip = 0; ip < ncible; ip++) /* on partitions */ {
			parinf = xd[ip]; /* limits of current partition  */
			parsup = xd[ip + 1];
			for (k = 0; k < 1; k++) /* 1 (!) passages */ {
				sx = sx2 = sy = sy2 = 0.;
				n = 0;

				if (verbose)
					printf("parti %d k %d debut %d fin %d\n", ip, k, xd[ip], xd[ip + 1]);

				/* mean sigma */
				for (j = parinf; j < parsup; j++) /* on the population of a partition */ {
					if (cib[j].y > 0.) {
						/*fprintf(stderr,"cible %d x %g y %g\n",j,cib[j].x ,cib[j].y);*/
						sx += cib[j].x;
						sx2 += cib[j].x * cib[j].x;
						sy += cib[j].y;
						sy2 += cib[j].y * cib[j].y;
						n++;
					}
				} /*  j loop */

				if (n > 1) /* for the variance */ {
					xm = (double) sx / (double) n;
					ym = (double) sy / (double) n;
					varx = (double) sx2 / (double) n - xm * xm;
					vary = (double) sy2 / (double) n - ym * ym;
					/*fprintf(stderr," X k%d n %d %g %g %g \n", k,n,(double)sx2/(double)n,xm*xm,varx);
					fprintf(stderr,"Y k%d n %d %g %g %g\n", k,n,(double)sy2/(double)n,ym*ym, vary);
					*/
					if (varx <= 0.) /* if variance == +epsilon */
						varx = 0.1;
					if (vary <= 0.)
						vary = 0.1;

					et2x = FSIGMA * sqrt (varx);
					et2y = FSIGMA * sqrt (vary);
					seuilbx = xm - et2x;
					seuilhx = xm + et2x;
					seuilby = ym - et2y;
					seuilhy = ym + et2y;

					for (j = parinf; j < parsup; j++) /* elimination */ {
						if (cib[j].y > 0. &&
							(cib[j].x < seuilbx || cib[j].x > seuilhx
							|| cib[j].y < seuilby || cib[j].y > seuilhy)) {
							cib[j].x = cib[j].y = 0.;
						}
					} /* boucle j */

				} /* if n>1 */
			} /* boucle k */

			/*recalculate means */
			sx = sy = 0.;
			n = 0;
			for (j = parinf; j < parsup; j++) {
				if (cib[j].y > 0.) {
					sx += cib[j].x;
					sy += cib[j].y;
					n++;
				}
			}
			if (n > 0) {
				cibred_cour.x = sx / n;
				cibred_cour.y = sy / n;
				cibred_cour.p = n;

				if (verbose)
					printf("ncibr %d x %g y %g p %d\n", ncibr,
						cibred_cour.x, cibred_cour.y, cibred_cour.p );
				if (ncibr < 0) {
					ncibr++;
					cibred[ncibr] = cibred_cour;
				}
				else {
					/* if cibred[].x are not strictly increasing
					delete the target with the smaller weight
					*/

					if (cibred_cour.x > cibred[ncibr].x)/* 1 more cibred with t increasing */ {
						ncibr++;
						cibred[ncibr] = cibred_cour;
					}
					else /* t <= preceding one */ {
						if(verbose)
							printf("reduc:cb arriere %g %d %g %d\n",
								cibred_cour.x, cibred_cour.p,
								cibred[ncibr].x, cibred[ncibr].p);

						if (cibred_cour.p > cibred[ncibr].p) {
							/* if current p  >, replace the preceding */
							cibred[ncibr] = cibred_cour;
						}
					}
				}

			} /*if n>0*/

		} /*  ip loop */

		ncibr_brut= ncibr+1; /* ( cibred[0,...ncibr_brut[ */

		/* 2nd pass for targets too close in t and Hz*/

		/* sort cibred by increasing temporal order */
		qsort( (char *) cibred, ncibr_brut, sizeof(struct st_cibred), cb_compare);

		nred2 = 0;
		cibred2[0] = cibred[0];

		for (i = 1; i < ncibr_brut; i++) {
			if (cibred[i].x - cibred2[nred2].x < seuildiff_x) {

				if ( fabs( (double)(cibred[i].y - cibred2[nred2].y) )/ cibred2[nred2].y < seuilrapp_y ) {
					if(verbose)
						printf("reduc:fusion_moyenne %g %g %d %g %g %d\n",
							cibred2[nred2].x,cibred2[nred2].y,cibred2[nred2].p,
							cibred[i].x, cibred[i].y, cibred[i].p );
					cibred2[nred2].x = (cibred2[nred2].x +cibred[i].x)/2. ;
					cibred2[nred2].y = (cibred2[nred2].y +cibred[i].y)/2. ;
					cibred2[nred2].p += cibred[i].p ;
				}
				else {
					if(verbose)
						printf("reduc:choix_poids %g %d %g %d\n",
							cibred2[nred2].x, cibred2[nred2].p,
							cibred[i].x, cibred[i].p );

					if (cibred2[nred2].p < cibred[i].p) {
						cibred2[nred2] = cibred[i] ;
					}
				}
			}
			else {
				nred2++ ;
				cibred2[nred2] = cibred[i] ;
			}
		}


		nred2++ ; /*lnumber of reduced targets*/
		if (verbose)
			printf("reduc: ncibred_brut: %d ncibred2 %d\n", ncibr_brut,nred2);

		for (i=0; i < nred2; i++) {
			cibout[ncibout].x = cibred2[i].x * fxitem.frameduration;
			cibout[ncibout].y = cibred2[i].y;
			ncibout++;
printf ("target=%g %g\n", cibred2[i].x * fxitem.frameduration, cibred2[i].y);
		}

	} /* partition */

	/* free memory */
	free(cibred);
	free(cibred2);
	free(xdist);
	free(ydist);
	free(dist);
	free(xd);
}

int PARV(int x,int H,int h,int T,int t,int k)
{
	double v = H+((double)(h-H)*(x-T)*(x-T))/((double)(k-T)*(t-T));
	return((int)(0.5+v));
}

/* create an Fx contour from modelled turning points */
void qsp()
{
	int	i,x;
	int 	*t, *h, *k;

	/* get output buffer */
	ofx = (short *)calloc(fxitem.numframes,sizeof(short));

	/* get calculation buffers */
	t = (int *)calloc(ncibout,sizeof(int));
	h = (int *)calloc(ncibout,sizeof(int));
	k = (int *)calloc(ncibout,sizeof(int));

	switch(ncibout) {
	case 0 :
		/* no roots */
		break;
	case 1 :
		/* one root */
		for (i=0;i<fxitem.numframes;i++)
			ofx[i] = (short)cibout[0].y;
		break;
	default:

		/* copy times as frame numbers */
		for (i=0;i<ncibout;i++) {
			t[i] = (int)(0.5+cibout[i].x/fxitem.frameduration);
			h[i] = (int)(0.5+cibout[i].y);
			if (i < ncibout-1)
				k[i] = (int)(0.5+(cibout[i].x+cibout[i+1].x)/(2*fxitem.frameduration));
			else
				k[i] = fxitem.numframes;
		}

		/* do start portion */
		for(x=0; x <= k[0]; x++)
			ofx[x] = PARV(x,h[0],h[1],t[0],t[1],k[0]);

		/* do middle portion */
		for(i=1; i < ncibout-1; i++) {
			for(; x <= t[i]; x++)
				ofx[x] = PARV(x,h[i],h[i-1], t[i],t[i-1], k[i-1]);
			for(; x <= k[i]; x++)
				ofx[x] = PARV(x, h[i],h[i+1], t[i],t[i+1], k[i]);
		}

		/* do end portion */
		for( ; x < k[ncibout-1]; x++)
			ofx[x] = PARV(x, h[ncibout-1], h[ncibout-2], t[ncibout-1],t[ncibout-2], k[ncibout-2]);

		break;
	}

	/* free memory */
	free(t);
	free(h);
	free(k);
}


/* main program */
int main(int argc,char **argv)
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item selections */
	char		*ty;
	char		*fxtype = "0";	/* default sub-type = last */
	/* file variables */
	char		filename[SFSMAXFILENAME];	/* dbase file name */
	int		fid;
	/* data variables */
	int		i;
	struct an_rec	*anrec;		/* annotation record */
	int		dofxmodel=0;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:fv")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Model Fx contour with MOMEL V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == FX_TYPE)
					fxtype = ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'f' :	/* save modelled Fx */
			dofxmodel++;
			break;
		case 'v' :	/* verbose */
			verbose=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-f) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* load fx data */
	getitem(filename,FX_TYPE,fxtype,&fxitem,(void *)&fx);

	/* stage 1 - prepare Fx by removing sudden jumps */
	pref(TOL1,MINFX);

	/* stage 2 - generate prototype turning points */
	cible((int)(0.5+WIN1/fxitem.frameduration),MINFX,MAXFX,TOL1);

	/* stage 3 - reduce to minimum # turning points */
	reduc((int)(0.5+WIN2/fxitem.frameduration),5.0,TOL2);

	/* create new item header */
	sfsheader(&anitem,AN_TYPE,-1,1,-1,
		fxitem.frameduration,fxitem.offset,0,0,1);
	sprintf(anitem.history,"%s/AN(%d.%02d)",PROGNAME,fxitem.datatype,fxitem.subtype);

	/* get annotation record */
	anrec=(struct an_rec *)sfsbuffer(&anitem,1);


	/* open output channel */
	if ((fid=sfschannel(filename,&anitem)) < 0)
		error("unable to open output channel to '%s'",filename);

	/* write out targets */
	for (i=0;i<ncibout;i++) {
		sprintf(anrec->label,"Fx=%d",(int)(0.5+cibout[i].y));
		anrec->posn = (int)(cibout[i].x/anitem.frameduration);
		anrec->size = 0;
		sfswrite(fid,1,anrec);
	}

	if (dofxmodel) {
		/* create new item header */
		sfsheader(&ofxitem,FX_TYPE,0,2,1,
			fxitem.frameduration,fxitem.offset,1,0,0);
		sprintf(ofxitem.history,"%s/FX(%d.%02d)",PROGNAME,fxitem.datatype,fxitem.subtype);

		/* open output channel */
		if ((fid=sfschannel(filename,&ofxitem)) < 0)
			error("unable to open output channel to '%s'",filename);

		/* generate contour and write it */
		qsp();
		sfswrite(fid,fxitem.numframes,ofx);
	}

	/* update original file */
	if (!sfsupdate(filename))
		error("update error on %s",filename);

	/* that's all folks ... */
	exit(0);
}
