/* mktrack -- general purpose energy track making program */

/* M.A.Huckvale - University College London */

/* version 1.0 - August 1999 */

/* version 1.1 - December 2003
	- fix delta switches
*/

#define PROGNAME "mktrack"
#define PROGVERS "1.1"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH MKTRACK UCL1 UCL SFS
.SH NAME
mktrack - general purpose energy track program
.SH SYNOPSIS
.B mktrack
(-i item) (-l lowfreq) (-h highfreq) (-n order) (-d) (-r rate) (-D1|-D2) (-t name) file
.SH DESCRIPTION
.I mktrack
is a program to calculate an energy track based on some simple specification
of a band-pass filter.  Options allow a decibel encoding and a first
and second differencing of the filter output.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -i item
Select input item number.
.TP 11
.BI -l lowfreq
Select frequency of high-pass band edge.  Default 0Hz.
.TP 11
.BI -h highfreq
Select frequency of low-pass band edge.  Default 0.5xsampling freq.
.TP 11
.BI -n order
Select filter order.  Default: 4. Maximum 20.
.TP 11
.BI -r rate
Sampling rate for track.  Default: 100Hz.
.TP 11
.B -d
Convert energy to decibels (logarithmic compression).  Default: rms.
.TP 11
.B -D1
Calculate first difference of energy track.
.TP 11
.B -D2
Calculate second difference of energy track.
.TP 11
.BI -t name
Give the parameter track a name.  Default: none.
.SH INPUT ITEMS
.IP SP
Any speech item
.SH OUTPUT ITEMS
.IP TR
Energy track.
.SH HISTORY
.IP lowfreq=
high-pass band edge
.IP highfreq=
low-pass band edge
.IP order=
filter order.
.IP name=
track name
.SH VERSION/AUTHOR
.IP 1.0
Mark Huckvale
.SH BUGS
*/
/*--------------------------------------------------------------------------*/

/* include files */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"
#include "filter.h"

/* global variables */
struct item_header spitem;
#define SPBUFSIZE 4096
short sp[SPBUFSIZE];
struct item_header opitem;

/* options */
int decibel=0;
int delta1=0;
int delta2=0;
double erate=100.0;
char trackname[256];

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it=SP_TYPE;	/* item selection */
	char		*ty="0";	/* item sub type */
	/* file variables */
	char		filename[SFSMAXFILENAME]; /* SFS data file name */
	int		fid;		/* input file descriptor */
	int		ofid;		/* output file descriptor */

	/* processing variables */
	FILTER		*flt=NULL,*sflt=NULL;
	int		fil_type=FILTER_BAND_PASS;
	int		fil_order=4;
	double		fil_low=50000;
	double		fil_high=0;
	double		half_freq;
	int		len,pos,opos;
	char		fil_messg[50];
	int		step;
	float		oval,sval,oenergy;
	float		lastval=0,last2val=0;
	int		i;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:l:h:r:dD:n:t:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Make energy track V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					/* ok */;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'l' :	/* high-pass band edge */
			fil_high = atof(optarg);
			break;
		case 'h' :	/* low-pass band edge */
			fil_low = atof(optarg);
			break;
		case 'n' :	/* order */
			fil_order = atoi(optarg);
			if ((fil_order < 1)||(fil_order > 20))
				error("bad filter order");
			break;
		case 'r':	/* sampling rate */
			erate = atof(optarg);
			if (erate < 1)
				error("bad sampling rate");
			break;
		case 'd':	/* log coding */
			decibel=1;
			break;
		case 'D':	/* differencing */
			if (*optarg=='1')
				delta1=1;
			else if (*optarg=='2')
				delta2=1;
			else
				error("bad -D switch");
			break;
		case 't':	/* name */
			strncpy(trackname,optarg,256);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-l lowfreq) (-h highfreq) (-n order) (-r rate) (-d) (-D1|-D2) (-t name) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* find input item */
	if ((fid=sfsopen(filename,"w",NULL)) < 0)
		error("access error on '%s'",filename);
	if (!sfsitem(fid,it,ty,&spitem))
		error("could not find input item in '%s'",filename);

	/* choose filter type */
	half_freq = (int)(0.5+0.5/spitem.frameduration);
	if (fil_low > half_freq) fil_low=half_freq;
	if (fil_low <= 0) fil_low=half_freq;
	if (fil_high > half_freq) fil_high=half_freq;
	if (fil_high < 0) fil_high=0;
	if ((fil_high==0) && (fil_low < half_freq))
		fil_type = FILTER_LOW_PASS;
	else if ((fil_high > 0) && (fil_high < half_freq) && (fil_low==half_freq))
		fil_type = FILTER_HIGH_PASS;
	else if ((fil_high < half_freq) && (fil_low < half_freq) && (fil_high < fil_low))
		fil_type = FILTER_BAND_PASS;
	else
		fil_type = 0;
/*		error("no/illegal filter specification: lo=%g hi=%g hf=%g",fil_low,fil_high,half_freq); */
/* printf("Filter type = %d from %g to %g\n",fil_type,fil_high,fil_low); */

	/* design filter */
	if (fil_type==FILTER_HIGH_PASS) {
		if ((flt = filter_design(fil_type,fil_order,fil_high,fil_high,1.0/spitem.frameduration))==NULL)
			error("could not design filter");
	}
	else if (fil_type==FILTER_LOW_PASS) {
		if ((flt = filter_design(fil_type,fil_order,fil_low,fil_low,1.0/spitem.frameduration))==NULL)
			error("could not design filter");
	}
	else if (fil_type==FILTER_BAND_PASS) {
		if ((flt = filter_design(fil_type,fil_order,fil_high,fil_low,1.0/spitem.frameduration))==NULL)
			error("could not design filter");
	}

	/* map output rate to whole # samples */
	step = (int)(0.5+1.0/(spitem.frameduration*erate));
	erate = 1.0/(step*spitem.frameduration);

	/* create smoothing filter */
	if ((sflt = filter_design(FILTER_LOW_PASS,4,erate,erate,1.0/spitem.frameduration))==NULL)
		error("could not design smoothing filter");

	/* create output item header */
	sfsheader(&opitem,TR_TYPE,1,4,1,1.0/erate,spitem.offset,1,0,0);
	if (trackname[0])
		sprintf(opitem.history,"%s(%d.%02d;lofreq=%g,hifreq=%g,order=%d,rate=%g%s%s,name=%s)",
			PROGNAME,
			spitem.datatype,
			spitem.subtype,
			fil_high,fil_low,
			fil_order,
			erate,
			(decibel)?",decibel":"",
			(delta2==1)?",delta=2":(delta1==1)?",delta=1":"",
			trackname);
	else
		sprintf(opitem.history,"%s(%d.%02d;lofreq=%g,hifreq=%g,order=%d,rate=%g%s%s)",
			PROGNAME,
			spitem.datatype,
			spitem.subtype,
			fil_high,fil_low,
			fil_order,
			erate,
			(decibel)?",decibel":"",
			(delta2==1)?",delta=2":(delta1==1)?",delta=1":"");

	/* open output channel */
	if ((ofid=sfschannel(filename,&opitem))<0)
		error("could not open output channel");

	/* process the signal */
	pos=0;
	opos=step;
	while ((len=sfsread(fid,pos,SPBUFSIZE,sp)) > 0) {
		for (i=0;i<len;i++) {
			if (flt)
				oval = filter_sample(flt,(float)sp[i]);
			else
				oval = (float)sp[i];
			oenergy = sqrt(oval * oval);
			if (decibel) oenergy = 20.0*log10(oenergy);
			sval = filter_sample(sflt,oenergy);
			pos++;
			if (pos==opos) {
				if (delta2)
					oval=sval-2*lastval+last2val;
				else if (delta1)
					oval=sval-lastval;
				else
					oval=sval;
				sfswrite(ofid,1,&oval);
				opos += step;
				last2val=lastval;
				lastval=sval;
			}
		}
		if (ttytest()) {
			printf("%d/%d frames\r",pos+1,spitem.numframes);
			fflush(stdout);
		}
	}
	if (ttytest()) {
		printf("                        \r");
		fflush(stdout);
	}

	/* update and exit */
	if (!sfsupdate(filename))
		error("update error on '%s'",filename);
	exit(0);
}


