/* noisanal -- estimate % of periodicity in a signal */

/* M.A. Huckvale - University College London */

/* version 1.0 - November 1999 */

/* version 1.1 - June 2001
	- add normalised energy option
*/

#define PROGNAME "noisanal"
#define PROGVERS "1.1"
char *progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH NOISANAL SFS1 UCL
.SH NAME
noisanal - estimate quantity of aperiodicity in signal
.SH SYNOPSIS
.B noisanal
(-i item) (-w windowsize) (-e) (-n) file
.SH DESCRIPTION
.I noisanal
estimates the degree of aperiodicity in a signal as a function of time.
It uses a type of autocorrelation measure to estimate what proportion of the
signal is predictable at each time instant.  The output is a track item
normalised to overall energy.  The measure is probably irrelevant in silent intervals.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select input item number and type.
.TP 11
.BI -w windowsize
Set the analysis window size.  Default 30ms.
.TP 11
.B -e
Output energy track as well as aperiodicity track.
.TP 11
.B -n
Normalise energy track to 0..1
.SH INPUT ITEMS
.IP 1.xx 11
Speech waveform.
.SH OUTPUT ITEMS
.IP 16 11
Periodicity estimate by autocorrelation.
.IP 16 11
Overall energy track (decibels).
.SH VERSION/AUTHOR
1.0 - Mark Huckvale.
.SH SEE ALSO
fxac(SFS1)
*/
/*--------------------------------------------------------------------------*/

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"			/* header structures */

/* FX estimation control */
#define	STEP	0.005	/* 5ms step */
#define WINDOW	0.030	/* 30ms window */
#define LOWFX	50	/* 50 Hz */
#define HIGHFX	400	/* 400 Hz */

/* global data */
struct item_header 	spitem;		/* item header data */
double			wtime=WINDOW;	/* window size */
short			*sp;		/* speech data buffer (1 x 25ms window) */
float			*fsp;		/* float speech buffer */
struct item_header 	tritem;		/* item header data */
struct item_header	enitem;		/* energy item header */
int			sampfreq;	/* current sampling frequency */
float			*acoeff;	/* autocorrelation function */
float			smoothmem[5]={1.0,1.0,1.0,1.0,1.0};
int			normenergy=0;
double			maxenergy=1;

/* auto comparison function */
void autoc(sp,len,acoeff,l1,l2)
float	*sp;
int	len;
float	*acoeff;
int	l1,l2;
{
	register int	i,j;
	int		num;
	float		sum;
	float		*s1,*s2;

	/* for zero delay */
	sum=(float)0.0;
	num=len;
	s1 = sp;
	for (j=0;j<num;j++,s1++) sum += *s1 * *s1;
	acoeff[0] = sum/num;

	/* for each delay in expected freq. range */
	for (i=l1;i<=l2;i++) {
		sum=(float)0.0;
		num=len-i;
		s1 = sp;
		s2 = sp + i;
		for (j=0;j<num;j++,s1++,s2++) sum += (*s1 - *s2)*(*s1 - *s2);
		acoeff[i] = sum/num;
	}

}

/* find dip */
int finddip(acoeff,len,l1,l2)
float	*acoeff;
int	len;
int	l1,l2;
{
	register int	i,pos;
	float		min;

	/* find smallest diff */
	pos = l1;
	min = acoeff[l1];
	for (i=l1+1;i<=l2;i++) {
		if (acoeff[i] < min) {
			min = acoeff[i];
			pos=i;
		}
	}
	return(pos);
}

/* estimate periodicity by auto comparison */
float perproc(sp,len)
float	*sp;			/* input signal buffer */
int	len;			/* buffer length */
{
	int	lofx,hifx;	/* autocorrelation limits */
	int	dip;		/* location of dip */
	
	/* get autocorrelation coefficients */
	hifx = sampfreq/HIGHFX;		/* 400Hz */
	lofx = sampfreq/LOWFX;		/* 60Hz */
	if (lofx > (len-10)) lofx=len-10;
	autoc(sp,len,acoeff,hifx,lofx);

	/* find auto comparison dip */
	dip = finddip(acoeff,len,hifx,lofx);

	/* return estimate of squared difference */
	if (acoeff[dip] > acoeff[0]) acoeff[dip]=acoeff[0];
	return(acoeff[dip]/acoeff[0]);
}

/* median smoother */
float smooth(val)
float val;
{
	int	i,idx;
	float	max;
	float	temp[5];
	
	for (i=0;i<4;i++) smoothmem[i] = smoothmem[i+1];
	smoothmem[4]=val;
	
	for (i=0;i<5;i++) temp[i]=smoothmem[i];

	max = temp[0];
	idx = 0;
	for (i=1;i<5;i++) if (temp[i]>max) { max=temp[i]; idx=i; }
	temp[idx]=0;

	max = temp[0];
	idx = 0;
	for (i=1;i<5;i++) if (temp[i]>max) { max=temp[i]; idx=i; }
	temp[idx]=0;

	max = temp[0];
	idx = 0;
	for (i=1;i<5;i++) if (temp[i]>max) { max=temp[i]; idx=i; }

	return(temp[idx]);
}
	
/* main program */
void main(argc,argv)
int argc;
char *argv[];
{
	/* local variables */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument */
	int		errflg=0;	/* option error flag */
	int		c;		/* option char */
	int		it;
	char		*ty;
	int		iptype=SP_TYPE;
	char		*sptype = "0";	/* input sp sub-type = last */
	char		filename[SFSMAXFILENAME];
					/* database file name */
	int		fid,ofid,ofid2;	/* file descriptors */

	/* processing variables */
	int		stsize;		/* # samples in step */
	int		wisize;		/* # samples in window */
	int		i,f=0;		/* counters */
	int		nframes;	/* # output frames */
	int		j;
	float		val,trval,sumsq;
	int		doenergy=0;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:w:en")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: periodicity by autocorrelation V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* item spec */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE) {
					iptype=SP_TYPE;
					sptype=ty;
				}
				else
					error("unsuitable item specification %s",optarg);
			}
			else
				error("illegal item specification %s",optarg);
			break;
		case 'w' :
			wtime = atof(optarg);
			break;
		case 'e' :
			doenergy=1;
			break;
		case 'n' :
			normenergy=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-w windowtime) (-e) (-n) file\n",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open file */
	if ((fid=sfsopen(filename,"w",NULL)) < 0) {
		if (fid==-1)
			error("'%s': file not found",filename);
		else
			error("access error on '%s'",filename);
	}

	/* locate input speech item */
	if (!sfsitem(fid,iptype,sptype,&spitem))
		error("cannot find input item in '%s'",filename);

	/* get window parameters */
	sampfreq = (int)(0.5 + (1.0/spitem.frameduration));
	stsize = (int)(0.5 + (STEP/spitem.frameduration));
	wisize = (int)(0.5 + (wtime/spitem.frameduration));

	/* estimate # output frames for user info */
	nframes = (spitem.numframes/stsize)-(wisize/stsize)+1;

	/* get input buffer */
	if ((sp=(short *)sfsbuffer(&spitem,wisize)) == NULL)
		error("cannot get buffer for speech",NULL);
	if ((fsp=(float *)calloc(wisize,sizeof(float)))==NULL)
		error("cannot get buffer for speech",NULL);
	if ((acoeff=(float *)calloc(1+sampfreq/LOWFX,sizeof(float)))==NULL)
		error("cannot get buffer for speech",NULL);

	/* generate output item header */
	sfsheader(&tritem,TR_TYPE,1,4,1,stsize*spitem.frameduration,spitem.offset+WINDOW/2,1,0,0);
	sprintf(tritem.history,"%s(%d.%02d,window=%g,aperiodicity)",
		PROGNAME,spitem.datatype,spitem.subtype,wtime);

	/* open output channel */
	if ((ofid=sfschannel(filename,&tritem)) < 0)
		error("cannot open temporary file",NULL);

	if (doenergy) {
		/* generate output item header */
		sfsheader(&enitem,TR_TYPE,1,4,1,stsize*spitem.frameduration,spitem.offset+WINDOW/2,1,0,0);
		if (normenergy)
			sprintf(enitem.history,"%s(%d.%02d,window=%g,normenergy)",
				PROGNAME,spitem.datatype,spitem.subtype,wtime);
		else
			sprintf(enitem.history,"%s(%d.%02d,window=%g,logenergy)",
				PROGNAME,spitem.datatype,spitem.subtype,wtime);

		/* open output channel */
		if ((ofid2=sfschannel(filename,&enitem)) < 0)
			error("cannot open temporary file",NULL);
	}

	/* find max energy window */
	if (normenergy) {
		for (i=0;sfsread(fid,i,wisize,sp)==wisize;i+=stsize) {

			/* copy waveform into float buffer */
			for (j=0,sumsq=0;j<wisize;j++) {
				/* generate low-level random noise */
				val = (rand()%201)-100;	/* +/-10 */
				/* add in to signal */
				fsp[j] = (float)(sp[j]+val);
				sumsq += fsp[j] * fsp[j];
			}

			if (sumsq > maxenergy) maxenergy = sumsq;
		}
	}
	
	/* perform processing */
	for (i=0;sfsread(fid,i,wisize,sp)==wisize;i+=stsize,f++) {

		/* copy waveform into float buffer */
		for (j=0,sumsq=0;j<wisize;j++) {
			/* generate low-level random noise */
			val = (rand()%201)-100;	/* +/-10 */
			/* add in to signal */
			fsp[j] = (float)(sp[j]+val);
			sumsq += fsp[j] * fsp[j];
		}

		/* process window */
		trval = perproc(fsp,wisize);

		/* smooth */
		trval = smooth(trval);

		/* save estimate */
		sfswrite(ofid,1,&trval);

		if (doenergy) {
			if (normenergy)
				val = sumsq/maxenergy;
			else
				val = 20.0*log10(sumsq/wisize);
			sfswrite(ofid2,1,&val);
		}

		/* report progress */
		if (((f%10)==9) && ttytest()) {
			printf("\rFrame %d/%d",f+1,nframes);
			fflush(stdout);
		}
	}
	if (ttytest()) printf("\r                       \r");

	/* and update file */
	if (!sfsupdate(filename))
		error("backup error on '%s'",filename);

	/* ... that's all folks */
	exit(0);
}
