// kaiser.cpp - Kaiser window design
//
// C++ (c) 1996 Mark Huckvale University College London

#include "tools.h"

// 
// Routines to support design of Kaiser window to specification
//

// zeroth order modified Bessel function of the first kind
const int ITERLIMIT=15;
const double CONVERGE=1.0E-8;

double besselI0(double p)
{
	// initialise iterative loop
	p = p/2;
	double n = 1;
	double t = 1;
	double d = 1;

	// iteration
	int k = 1;
	double v;
	do {
		n = n * p;
		d = d * k;
		v = n / d;
		t = t + v * v;
	} while ((++k < ITERLIMIT) && (v > CONVERGE));

	return t;
}

// return half-Kaiser window to specification }
Waveform Kaiser(
	double ripple,		// input stop-band ripple (dB)
	double twidth,		// input transition width
				//  (fraction of sample rate)
	int kmaxlen)		// maximum window size
{
	double alpha;		// window coefficient 
	int hlen;		// half window length
	double d,v;
	int n;

	// set Kaiser window coefficient (design rule)
	if (ripple <= 21)
		alpha = 0;
	else if (ripple < 50)
		alpha = 0.5842*exp(0.4*log(ripple-21))+0.07886*(ripple-21);
	else
		alpha = 0.1102*(ripple - 8.7);

	// set Kaiser window size (design rule)
	if (ripple < 7.95)
		hlen = 0;
	else
		hlen = 1+(int)((ripple - 7.95)/(28.72*twidth));
	if ((hlen+1) > kmaxlen) hlen = kmaxlen - 1;

	// build window
	Waveform kwin(hlen+1,1.0);		// output window

	// calculate window 1..hlen+1
	d = besselI0(alpha);
	kwin[1] = 1.0;
	for (n=1;n<=hlen;n++) {
		v = n/(double)hlen;
		kwin[n+1] = besselI0(alpha*sqrt(1-v*v))/d;
	}

	return kwin;
}
