// lpcauto.cpp - compute linear predictor coefficients by autocorrelation
//
// C++ (c) 1996 Mark Huckvale University College London

#include "tools.h"
#include "ltisys.h"
#include "lpcauto.h"

//	LPCAutocorrelation()	Form autocorrelation vector and then
//				solve the normal equations using the
//				Levinson (Durbin) recursion
//
//	Translated from the routine LPCA written in FORTRAN
//	by T.Parsons "Voice and Speech Processing" McGraw-Hill 1987

int	LPCAutocorrelation(
	Waveform x,		// windowed input signal
	int order,		// predictor order required
	LTISystem& ltis,	// returned predictor coefficients
	double& pe		// returned predictor error
	)			// returns 0=OK, 1=zero power, 2=fail
{
	// compute autocorrelations
	double	*r = new double[order+1];	// temporary array
	int i;
	for (i=0;i<=order;i++) {
		double sum = 0;
		for (int k=1;k<=x.count()-i;k++)
			sum += x[k] * x[k+i];
		r[i] = sum;
	}

	// check power in signal
	if (r[0] == 0) {
		// no signal !!
		delete [] r;
		return 1;
	}

	// compute predictor coefficients
	double	*pc = new double[order+1];	// temporary array
	pe = r[0];		// initialise error to total power
	pc[0] = 1.0;		// first coefficient (b[0]) must = 1

	// for each coefficient in turn
	for (int k=1;k<=order;k++) {

		// find next coeff from pc[] and r[]
		double sum = 0;
		for (i=1;i<=k;i++)
			sum -= pc[k-i] * r[i];
		pc[k] = sum/pe;

		// perform recursion on pc[]
		for (i=1;i<=k/2;i++) {
			double pci  = pc[i] + pc[k] * pc[k-i];
			double pcki = pc[k-i] + pc[k] * pc[i];
			pc[i] = pci;
			pc[k-i] = pcki;
		}

		// calculate residual error
		pe = pe * (1.0 - pc[k]*pc[k]);
		if (pe <= 0) {
			// no power left in signal!
			delete [] r;
			delete [] pc;
			return 2;
		}
	}

	// copy coefficients into LTI System
	ltis = LTISystem(0,order);
	ltis.a[0] = 1.0;
	for (i=1;i<=order;i++)
		ltis.b[i] = pc[i];

	// return OK
	delete [] r;
	delete [] pc;
	return 0;
}
