// ltisys.cpp -- implementation for Linear system class
//
// C++ (c) 1996 Mark Huckvale University College London

// The LTISystem class provides a convenient implementation
// of a linear time-invariant system including estimation
// of frequency response

#include "tools.h"
#include "ltisys.h"

// constructors
LTISystem::LTISystem(int na,int nb)
: a(na+1,1,0), 			// a[0]..a[na]
  b(nb+1,1,0), 			// b[0]..b[nb] (but b[0]=1.0)
  s((na>nb)?na+1:nb+1,1,0)	// s[0]..s[max(na,nb)]
{
	clear();
}
LTISystem::LTISystem(const LTISystem& ltis)
{
	a = ltis.a;
	b = ltis.b;
	s = ltis.s;
}
LTISystem::~LTISystem()
{
}

// assignment
LTISystem& LTISystem::operator= (const LTISystem& ltis)
{
	if (this == &ltis)
		return *this;
	a = ltis.a;
	b = ltis.b;
	s = ltis.s;
	return *this;
}

// clear
void LTISystem::clear()
{
	for (int i=0;i<s.count();i++) s[i]=0;
}

// run sample through system
double LTISystem::operator() (double ival)
{
	int i;

	// shift state memory
	for (i=s.count()-1;i>0;i--) s[i] = s[i-1];

	// compute s[0] from y[] coefficients
	s[0] = ival;
	for (i=1;i<b.count();i++)
		s[0] -= b[i] *s[i];

	// compute output from x[] coefficients
	double out = 0;
	for (i=0;i<a.count();i++)
		out += a[i] * s[i];

	return out;
}

// caclculate frequency response
Complex LTISystem::response(double freq) const
{
	int i;
	WaveComplex omega(s.count(),1,0);
	Complex z=exp(Complex(0,2*PI*freq));

	// initialise polynomial of complex frequency
	omega[0] = 1.0;
	omega[1] = z;
	for (i=2;i<s.count();i++)
		omega[i] = omega[i-1] * z;

	// compute response of numerator
	Complex rnum(0,0);
	for (i=0;i<a.count();i++)
		rnum += a[i] * omega[i];

	// compute response of denominator
	Complex rden=omega[0];
	for (i=1;i<b.count();i++)
		rden += b[i] * omega[i];

	// compute ratio
	if (mag(rden)==0)
		return 1.0E5;		// i.e. infinity
	else
		return rnum/rden;
}
