// Wavecomp.cpp -- implementation for complex floating-point waveform class
//
// (c) 1996 Mark Huckvale University College London

#include <iostream.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include "wavecomp.h"
#include "waveshor.h"
#include "wavedoub.h"

// constructors
WaveComplex::WaveComplex(int length,double rate,int first)
{
	alloc = WAVE_ALLOC_CHUNK;
	while (alloc < length) alloc += WAVE_ALLOC_CHUNK;
	siz = length;
	lobound = first;
	tau = 1.0/rate;
	buf = new Complex[alloc];
	memset(buf,0,alloc*sizeof(Complex));
}
WaveComplex::WaveComplex(const WaveComplex& wv)
{
	alloc = wv.alloc;
	siz = wv.siz;
	lobound = wv.lobound;
	tau = wv.tau;
	buf = new Complex[alloc];
	memcpy(buf,wv.buf,alloc*sizeof(Complex));
}
WaveComplex::~WaveComplex()
{
	delete [] buf;
}

// virtual dump function
void WaveComplex::dump(int len,int first) const
{
	first -= lobound;
	if (first < 0) first=0;
	Complex	*ptr = buf+first;
	if ((len < 0) || (len > (siz-first))) len=(siz-first);
	while (len-- > 0) 
		cout << *ptr++ << ' ';
	cout.flush();
}

// assignment operator
WaveComplex& WaveComplex::operator= (const WaveComplex& wv)
{
	if (&wv == this)
		return *this;		// self-assignment
	delete [] buf;
	alloc = wv.alloc;
	siz = wv.siz;
	lobound = wv.lobound;
	tau = wv.tau;
	buf = new Complex[alloc];
	memcpy(buf,wv.buf,alloc*sizeof(Complex));
	return *this;
}

// sample access
Complex& WaveComplex::operator[] (const int idx)
{
	static Complex junk=0;
	int nidx = idx - lobound;
	if ((nidx < 0) || (nidx >= siz)) {
		junk = Complex(0,0);
		return junk;		// silently ignore range errors
	}
	else
		return buf[nidx];
}
const Complex& WaveComplex::operator[] (const int idx) const
{
	static Complex junk=0;
	int nidx = idx - lobound;
	if ((nidx < 0) || (nidx >= siz)) {
		junk = Complex(0,0);
		return junk;		// silently ignore range errors
	}
	else
		return buf[nidx];
}

// concatenation
WaveComplex& WaveComplex::operator+= (const Complex val)
{
	if (siz >= alloc) {
		Complex * ptr = new Complex[alloc+WAVE_ALLOC_CHUNK];
		memcpy(ptr,buf,alloc*sizeof(Complex));
		alloc += WAVE_ALLOC_CHUNK;
		delete [] buf;
		buf = ptr;
	}
	buf[siz++] = val;
	return *this;
}
WaveComplex& WaveComplex::operator+= (const WaveComplex& wv)
{
	if (fabs(tau-wv.tau)/(tau+wv.tau) > 1.0E-6) {
		cerr << "WaveComplex: sample rate mismatch.\n";
		exit(1);
	}
	if ((siz+wv.siz) >= alloc) {
		while (alloc < (siz+wv.siz)) alloc += WAVE_ALLOC_CHUNK;
		Complex *ptr = new Complex[alloc];
		memcpy(ptr,buf,siz*sizeof(Complex));
		delete [] buf;
		buf = ptr;
	}
	memcpy(buf+siz,wv.buf,wv.siz*sizeof(Complex));
	siz += wv.siz;
	return *this;
}
#ifdef __GNUC__
WaveComplex WaveComplex::operator+ (const WaveComplex& wv) return owv(siz+wv.siz,wv.rate(),lobound)
{
	if (fabs(tau-wv.tau)/(tau+wv.tau) > 1.0E-6) {
		cerr << "WaveComplex: sample rate mismatch.\n";
		exit(1);
	}
	memcpy(owv.buf,buf,siz*sizeof(Complex));
	memcpy(owv.buf+siz,wv.buf,wv.siz*sizeof(Complex));
	return owv;
}
#else
WaveComplex WaveComplex::operator+ (const WaveComplex& wv)
{
	if (fabs(tau-wv.tau)/(tau+wv.tau) > 1.0E-6) {
		cerr << "WaveComplex: sample rate mismatch.\n";
		exit(1);
	}
	WaveComplex owv(siz+wv.siz,wv.rate(),lobound);
	memcpy(owv.buf,buf,siz*sizeof(Complex));
	memcpy(owv.buf+siz,wv.buf,wv.siz*sizeof(Complex));
	return owv;	// unfortunately uses copy constructor
}
#endif

// cut function
WaveComplex WaveComplex::cut(int first,int len) const
{
	first -= lobound;
	if ((first < 0) || (first >= siz)) {
		first=0;
	}
	if (len < 0) len=0;
	if ((first+len) > siz) len = siz-first;
	WaveComplex owv(len,rate(),lobound);
	memcpy(owv.buf,buf+first,len*sizeof(Complex));
	return owv;
}

// conversion to other wave types
WaveDouble WaveComplex::realtoWaveDouble() const
{
	WaveDouble owv(siz,rate(),lobound);
	for (int i=0;i<siz;i++) owv[i+lobound] = buf[i].real();
	return owv;
}
WaveDouble WaveComplex::imagtoWaveDouble() const
{
	WaveDouble owv(siz,rate(),lobound);
	for (int i=0;i<siz;i++) owv[i+lobound] = buf[i].imag();
	return owv;
}
WaveDouble WaveComplex::magtoWaveDouble() const
{
	WaveDouble owv(siz,rate(),lobound);
	for (int i=0;i<siz;i++) owv[i+lobound] = mag(buf[i]);
	return owv;
}
WaveDouble WaveComplex::argtoWaveDouble() const
{
	WaveDouble owv(siz,rate(),lobound);
	for (int i=0;i<siz;i++) 
		if (mag(buf[i]) < 1.0E-6)
			owv[i+lobound] = 0;
		else
			owv[i+lobound] = arg(buf[i]);
	return owv;
}


#ifdef EMO

void wvdump(Wave *wv)
{
	cout << "wv->alloc=" << wv->allocsize() << " ";
	cout << "wv->count=" << wv->count() << " ";
	cout << "wv->rate=" << wv->rate() << "\n";
	cout << "wv->buf="; wv->dump(); cout << "\n";
}

void main()
{
	WaveComplex wv(10);
	int	i;

	for (i=0;i<wv.count();i++) wv[i] = Complex(i/10.0,i/20.0);
	cout << "\nlist 0 to 9\n";
	wvdump(&wv);

	WaveComplex wvnew = wv;
	cout << "\nlist 0 to 9\n";
	wvdump(&wvnew);

	wvnew += Complex(1.0,0.5);
	cout << "\nlist 0 to 10\n";
	wvdump(&wvnew);

	wvnew += wv;
	cout << "\nlist 0 to 10, then 0 to 9\n";
	wvdump(&wvnew);

	wvnew = wv;
	cout << "\nlist 0 to 9\n";
	wvdump(&wvnew);

	wvnew = wv + wv;
	cout << "\nlist 0 to 9, then 0 to 9\n";
	wvdump(&wvnew);

	wvnew = wvnew + wv;
	cout << "\nlist 0 to 9, then 0 to 9, then 0 to 9\n";
	wvdump(&wvnew);

	wvnew = wv.cut(3,4);
	cout << "\nlist 3 to 6\n";
	wvdump(&wvnew);

	WaveDouble wvr = wv.realtoWaveDouble();
	cout << "\nReal version\n";
	wvdump(&wvr);

	WaveDouble wvi = wv.imagtoWaveDouble();
	cout << "\nImaginary version\n";
	wvdump(&wvi);

	WaveDouble wvm = wv.magtoWaveDouble();
	cout << "\nMagnitude version\n";
	wvdump(&wvm);

	WaveDouble wva = wv.argtoWaveDouble();
	cout << "\nArgument version\n";
	wvdump(&wva);

}
#endif

