// zplane.cpp -- implementation of z-plane graph plotting
//
// C++ (c) 1996 Mark Huckvale University College London

#include <stdio.h>
#include "SFSCONFG.h"
#include "sfs.h"
#include "dig.h"
#include "digdata.h"
#include "tools.h"
#include "graph.h"
#include "zplane.h"

int PlotZPlane(const ComplexWaveform& poles,
	       const ComplexWaveform& zeros,
	       Graph& gr, int gno, char *title)
{
	double	xcirc[200],ycirc[200];
	double	xdata[200],ydata[200];
	int	num;
	int	i;
	float	nh,nv;

	if (gr.getstate()==Graph::noinit) gr.open();

	// find pixel dimensions
	nh = digdata.nhoriz / gr.numhoriz();
	nv = (digdata.nvert/1.2) / gr.numvert();
	gr.setxdata(GRAPH_X_ALL,xcirc);
	if (nv > nh) {
		gr.axes(-1.0,1.0,-nv/nh,nv/nh,gno);
	}
	else {
		gr.axes(-nh/nv,nh/nv,-1.0,1.0,gno);
	}

	// calculate unit circle
	for (i=0;i<200;i++) {
		xcirc[i] = cos(2.0*PI*i/200.0);
		ycirc[i] = sin(2.0*PI*i/200.0);
	}

	// set up plot
	gr.setxdata(GRAPH_X_ALL,xcirc);
	gr.setxtitle("Real");
	gr.setytitle("Imag");
	gr.setxzero(GRAPH_ON);
	gr.setyzero(GRAPH_ON);
	gr.settype(GRAPH_LINE);
	gr.setgtitle(title);

	// plot circle
	gr.plot(ycirc,200,gno);

	// plot poles
	num = poles.count();
	if (num > 200) num=200;
	for (i=0;i<num;i++) {
		xdata[i] = poles[i+1].real();
		ydata[i] = poles[i+1].imag();
	}
	gr.setxdata(GRAPH_X_ALL,xdata);
	gr.settype(GRAPH_POINT);
	gr.setchar('X');
	gr.plot(ydata,num,gno);

	// plot zeros
	num = zeros.count();
	if (num > 200) num=200;
	for (i=0;i<num;i++) {
		xdata[i] = zeros[i+1].real();
		ydata[i] = zeros[i+1].imag();
	}
	gr.setxdata(GRAPH_X_ALL,xdata);
	gr.settype(GRAPH_POINT);
	gr.setchar('O');
	gr.plot(ydata,num,gno);

	gr.settype(GRAPH_LINE);
	gr.setchar(0);

	return(0);
}

#ifdef EMO

ComplexWaveform poles(2,1);
ComplexWaveform zeros(1,1);

void doplot(int nh,int nw)
{
	int	i,j;

	Graph gr(nh,nw,"Z-Plane");

	for (i=nh-1;i>=0;i--)
		for (j=0;j<nw;j++)
			PlotZPlane(poles,zeros,gr,1+i*nw+j,"2 Poles and 1 Zero");

	gr.close();
}

int main()
{

	poles[1] = Complex(0.5,0.5);	
	poles[2] = Complex(0.5,-0.5);	
	zeros[1] = Complex(-0.5,0.0);	

	doplot(1,1);
	doplot(2,3);
	doplot(3,2);

}
#endif
