/* spandrv - C-SPAN synthesis support routines */

/* M.A. Huckvale - November 1989 */

/* version 1.0 */

/* version 1.1 - November 1992
	- add -A,B,C,.. parameters for SYNTH()
*/

#define PROGNAME "span"
#define PROGVERS "1.1"
char	*progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH SPAN UCL1 UCL SFS
.SH NAME
span - C-SPAN synthesizer control language compiler
.SH SYNOPSIS
.B span
(-I) (-p) synth.spn
.SH DESCRIPTION
.I span
is a script to compile and link a synthesizer control program written
in the SPAN language (see SPAN manual page).
.PP
The source file "<synth>.spn" must contain a legal C-SPAN program
with its main procedure called SYNTH().  The
.I span
script concatenates a SPAN definition header with the supplied file
to generate a C-language source file which is then compiled and linked 
with the SPAN run-time modules to create an executable program called
"<synth>".  Thus a SPAN file called "test.spn" creates a program called
"test".  To save the macro processed output from SPAN use the switch "-p", 
the C pre-processor output will be produced in a file <synth>.i.
.PP
SPAN programs have a common command-line interface:
.nf

<synth> (-I) (-v) (-n num) (-s sfsfile) (-A param) (-B param) etc

.fi
with the following
options:
.PP
.TP 11
.B -I
Identify program name and version number.
.TP 11
.B -s sfsfile
Save output to SFS file. (Default is to control synthesizer/replay).
.TP 11
.BI -n num
Save/Replay synthesis 'n' (counting 0,1,...).  
Default is to save/replay all syntheses.
.TP 11
.B -v
Verbose. Print control parameters to screen.
.TP 11
.BI -A param etc
Specify parameter 0 (B=1, C=2, etc) for the SYNTH function.  Parameters
are passed as an array of 26 strings.
.SH OUTPUT ITEMS
.IP SYNTH
SPAN synthesizer control parameters
.SH HISTORY
.IP file 11
source file for synthesis.
.IP synth 11
synthesis number
.SH FILES
.IP $(SFSBASE)/include/spandrv.h 11
Include file for compiling SPAN programs.
.IP $(SFSBASE/lib/spandrv.a 11
Library containing SPAN run-time system.
.SH VERSION/AUTHOR
.IP 1.1 11
M.A.Huckvale
.SH SEE ALSO
syout(1), span(5), enspan(1)
.SH BUGS
*/
/*--------------------------------------------------------------------------*/
/* include files */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <malloc.h>
#include "sfs.h"
#include "spandrv.h"

/* operation mode */
int	verbose=0;
int	synthsave=0;
int	synthnum= -1;

/* global data on output file */
char	filename[SFSMAXFILENAME]; /* SFS data file name */
extern char	*sourcename;

/* SYNTH parameters */
char	*letpar[26];

/* functions to push and pop integers on a small stack */
#define STACKSIZE	100
static int	stack[STACKSIZE];
static int	*sp=stack;
void push(val)
int	val;
{
	if (sp < (stack+STACKSIZE))
		*sp++ = val;
	else
		error("SAVE stack overflow");
}
int	pop()
{
	if (sp > stack)
		return(*--sp);
	else
		error("RESTORE stack underflow");
	return(0);
}

/* global data */
struct param_rec {
	int	value;
	int	interp;
	int	time;
	struct param_rec *next;
};

/* global synthesis table */
static struct param_rec parmtab[NUM_PARM];
int	_abstime;

/* other global parameters */
static int	somechange=0;		/* set by setval, reset by flush */
static int	lasttime=0;		/* set by setval, used/reset by flush */
static int	currsynth=0;		/* index of current synthesis */

/* SFS data */
struct item_header	syitem;

/* interpolation function */
static int interpolate(v1,t1,itype,v2,t2,t3)
int	v1,v2;			/* range of values */
int	t1,t2,t3;		/* t1 < t3 < t2 */
int	itype;			/* interpolation type */
{
	int	dur;
	float	slope;
	float	lv1,lv2;

	/* get duration */
	dur = t2-t1;
	if (dur==0) return(v1);

	/* do different interpolations */
	switch (itype) {
	case NUL:		/* no interpolation */
	case FIX:
		return(v1);

	case LIN:		/* linear interpolation */		
		slope = (float)(v2-v1)/dur;
		return((int)(0.5 + v1 + (t3-t1)*slope));

	case LOG:		/* logarithmic interpolation */
		lv1 = (v1) ? log((double)v1) : 0;
		lv2 = (v2) ? log((double)v2) : 0;
		slope = (lv2-lv1)/dur;
		return((int)(0.5+exp(lv1 + (t3-t1)*slope)));
	}
	return(0);
}

/* function to add record to parameter net */
static struct param_rec *createparm(p,t)
int	p;
int	t;
{
	struct param_rec *plast,*pnext,*pnew;

	/* initialise pointers */
	plast = pnext = &parmtab[p];

	/* skip to correct position in parameter list */
	while (pnext && (pnext->time < t)) {
		plast = pnext;
		pnext = pnext->next;
	}

	/* check for found bracket */
	if (pnext) {
		/* found a range */
		if (pnext->time == t) {
			/* found an exact time entry */
			return(pnext);
		}
		else if (plast) {
			/* found a bracket */
			if ((pnew=(struct param_rec *)malloc(sizeof(struct param_rec)))==NULL)
				error("no memory for buffers");
			pnew->value = interpolate(plast->value,plast->time,plast->interp,pnext->value,pnext->time,t);
			pnew->interp = FIX;
			pnew->time = t;
			pnew->next = pnext;
			plast->next = pnew;
			return(pnew);
		}
		else {
			error("negative time ?");
			return(0);
		}
	}
	else {
		/* reached end of param data */
		if ((pnew=(struct param_rec *)malloc(sizeof(struct param_rec)))==NULL)
			error("no memory for buffers");
		pnew->value = plast->value;
		pnew->interp = FIX;
		pnew->time = t;
		pnew->next = NULL;
		plast->next = pnew;
		return(pnew);
	}
}

/* function to determine current value of parameter at given time */
static int	currentparm(p,t)
int	p;
int	t;
{
	struct param_rec *plast,*pnext;

	/* initialise pointers */
	plast = pnext = &parmtab[p];

	/* skip to correct position in parameter list */
	while (pnext && (pnext->time < t)) {
		plast = pnext;
		pnext = pnext->next;
	}

	/* check for found bracket */
	if (pnext) {
		/* found a range */
		if (pnext->time == t)
			/* found an exact time entry */
			return(pnext->value);
		else if (plast)
			/* found a bracket */
			return(interpolate(plast->value,plast->time,plast->interp,pnext->value,pnext->time,t));
		else {
			error("negative time ?");
			return(0);
		}
	}
	else
		/* reached end of param data */
		return(plast->value);
}

/* function to set up values in parameter data structure */
int _setval(p,v,i,t)
int	p,v,i,t;
{
	int	oldvalue;
	struct param_rec	*pr;

	/* check not an equiry on parameter number */
	if (v==-2)
		/* return parameter number, not parameter value */
		return(p);

	/* check ranges */
	if ((p < 0) || (p >= NUM_PARM))
		error("parameter out of range");
	if ((v < NUL) || (v > 32767))
		error("parameter value out of range");
	if ((i < NUL) || (i > LOG))
		error("illegal interpolation operator");
	if ((t < 0) || (t > 120000))
		error("synthesis longer than 2 minute maximum");

	/* check whether enquiry or setting */
	if (!v && !i) {
		/* just an enquiry */
		return(currentparm(p,t));
	}
	else {
		/* create an entry at given time */
		pr = createparm(p,t);

		/* save old value for return */
		oldvalue = pr->value;

		/* set up new value */
		if (v != NUL)
			pr->value = v;

		/* set up new interpolation */
		if (i != NUL)
			pr->interp = i;

		/* record change */
		somechange++;

		/* record latest time */
		if (t > lasttime) lasttime=t;

		return(oldvalue);
	}
}

/* set maximum synthesis length */
void _maxlength(t)
{
	if ((t >= 0) && (t < 120000))
		lasttime=t;
}

/* clear entire synthesis */
void _clearsynth()
{
	int	i;
	struct param_rec	*pr,*nextpr;

	for (i=0;i<NUM_PARM;i++) {
		/* get address of first dynamic entry */
		pr = parmtab[i].next;
		/* delete linked list */
		while (pr) {
			nextpr = pr->next;
			free(pr);
			pr = nextpr;
		}
		/* reset root parameters */
		parmtab[i].value = 0;
		parmtab[i].interp = 0;
		parmtab[i].time = 0;
		parmtab[i].next = NULL;
	}
	/* reset synthesis parameters */
	_abstime=0;
	lasttime=0;
	somechange=0;
}

/* parameters worth having */
static int parmreq[NUM_PARM]={
	1,0,0,1,10,0,1,10,0,1,10,0,1,10,0,1,10,0,1};

/* flush synthesis */
void _spanflush()
{
	int			i,t,fid=-1;
	struct param_rec	*pr[NUM_PARM];
	short			sy[NUM_PARM];
	short			*buff=NULL;
	int			synthlen=0;

	/* don't count NULL syntheses */
	if (!somechange) return;

	/* check single synthesis */
	if ((synthnum >= 0) && (currsynth != synthnum)) {
		currsynth++;
		return;
	}

	/* save or replay ? */
	if (synthsave) {
		/* open SFS output */
		sfsheader(&syitem,SY_TYPE,0,2,NUM_PARM,0.01,0.0,1,0,0);
		sprintf(syitem.history,"span(program=%s,synth=%d)",
				sourcename,currsynth);
		if ((fid=sfschannel(filename,&syitem)) < 0)
			error("unable to open channel to '%s'",filename);
	}
	else {
		/* get buffer for synthesis */
		synthlen=2+(lasttime+9)/10;
		if ((buff=(short *)calloc(synthlen,16*sizeof(short)))==NULL)
			error("unable to get buffer for synthesis");
	}

	/* initialise parameter sweep */
	for (i=0;i<NUM_PARM;i++)
		pr[i] = &parmtab[i];

	/* print header */
	if (verbose)
		printf("  Time    FX   F1   A1   F2   A2   F3   A3   F4   A4   FN   AN   VR\n");

	/* initialise replay buffer */
	if (!synthsave) {
		for (i=0;i<NUM_PARM;i++) 
			sy[i]=parmreq[i]*parmtab[i].value;
		hwsynthcode(sy,buff);
	}

	/* sweep in time to maximum time */
	for (t=0;t<=lasttime;t+=10) {
		if (verbose) printf("%5d: ",t);

		/* clear frame */
		for (i=0;i<NUM_PARM;i++)
			sy[i]=0;

		/* sweep across parameters */
		for (i=0;i<NUM_PARM;i++) if (parmreq[i]) {

			/* need to move on ? */
			while ((pr[i]->next) && (pr[i]->next->time <= t)) {
				/* yes */
				pr[i] = pr[i]->next;
			}

			/* calculate current value */
			if (pr[i]->next) {
				sy[i] = interpolate(
						parmreq[i] * pr[i]->value,
						pr[i]->time,
						pr[i]->interp,
						parmreq[i] * pr[i]->next->value,
						pr[i]->next->time,
						t);
			}
			else {
				sy[i] = parmreq[i] * pr[i]->value;
			}
			if (verbose)
				printf("%5d",sy[i]);
		}
		if (verbose)
			printf("\n");

		/* save or replay ? */
		if (synthsave)
			/* save to SFS file */
			sfswrite(fid,1,sy);
		else
			/* encode into replay buffer */
			hwsynthcode(sy,buff+(t/10)*16+16);
	}

	/* replay the synthesis */
	if (!synthsave) {
		hwsynth(buff,synthlen);
		free(buff);
	}

	somechange=0;
	currsynth++;
	if ((currsynth%20)==0) {
		if (!sfsupdate(filename))
			error("update error on '%s'",filename);
	}
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		fid;		/* input file descriptor */

	/* decode switches */
	while ( (c = getopt(argc,argv,"Isn:vA:B:C:D:E:F:G:H:J:K:L:M:N:O:P:Q:R:S:T:U:V:W:X:Y:Z:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: C-SPAN run-time program V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 's' :	/* save syntheses to file */
			synthsave++;
			break;
		case 'n' :	/* choose synthesis number */
			synthnum=atoi(optarg);
			break;
		case 'v' :	/* verbose option */
			verbose++;
			break;
		case 'A' :	/* SYNTH parameters */
		case 'B' :
		case 'C' :
		case 'D' :
		case 'E' :
		case 'F' :
		case 'G' :
		case 'H' :
		case 'J' :
		case 'K' :
		case 'L' :
		case 'M' :
		case 'N' :
		case 'O' :
		case 'P' :
		case 'Q' :
		case 'R' :
		case 'S' :
		case 'T' :
		case 'U' :
		case 'V' :
		case 'W' :
		case 'X' :
		case 'Y' :
		case 'Z' :
			letpar[c-'A'] = optarg;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg)
		error("usage: %s (-I) (-v) (-n num) (-s sfsfile) (-A param) (-B param etc)",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else if (synthsave)
		error("no output file specified",NULL);

	/* if syntheses to be saved, check output file */
	if (synthsave) {
		if ((fid=sfsopen(filename,"w",NULL)) < 0)
			error("access error on '%s'",filename);
		sfsclose(fid);
	}

	/* run the synthesis code */
	_abstime=0;
	SYNTH(letpar);
	_spanflush();

	/* update sfsfile */
	if (synthsave) {
		if (!sfsupdate(filename))
			error("update error on '%s'",filename);
	}
	exit(0);
}

