function exitcode=modulation(fname,ilist,plist)
% MODULATION - performs fm and am modulation and demodulation
%
%  MODULATION(fname,ilist,plist) processes the speech item 
%  specified in ilist in the SFS file fname, using parameters
%  specified in plist.
%
%  MODULATION is designed to be run by MATLABEX with the
%  following command line parameters:
%
%    -m fcarrier      Modulate using given carrier frequency
%    -d fcarrier      Demodulate using given carrier frequency
%    -t "am"          Amplitude modulation (default)
%    -t "fm"          Frequency modulation
%    -t "pm"          Phase modulation

%
% sort out items
%
if (isempty(ilist)) ilist=[1]; end;
%
% sort out options
%
fcarrier=10000;
mtype='am';
demodulate=0;
%
for i=1:length(plist)
    if (strcmp(plist(i).param,'-m'))
        % modulate 
        fcarrier=str2num(plist(i).value);
        demodulate=0;
    end;
    if (strcmp(plist(i).param,'-d'))
        % demodulate
        fcarrier=str2num(plist(i).value);
        demodulate=1;
    end;
    if (strcmp(plist(i).param,'-t'))
        % modulation type
        mtype=plist(i).value;
    end;
end;
%
% load speech signal
%
[h,d]=sfsgetitem(fname,num2str(ilist(1)));
srate=1.0/h.frameduration;
%
% upsample to 2*(carrier + bandwidth)
%
if (demodulate==0)
   [num,den]=rat(2*(fcarrier+srate/2)/srate,0.001)
   d=resample(d,num,den);
   srate=num*srate/den;
   fprintf('Up-sampled to %g samples/sec\n',srate);
end
%
% make output header
%
oh=sfsnewitem(h.datatype,1.0/srate,h.offset,1,length(d));
%
% do processing
%
if (demodulate)
    y=demod(d,fcarrier,srate,mtype);
    oh.history=sprintf('demodulate(%d.%02d;carrier=%g,type=%s)',...
        h.datatype,h.subtype,fcarrier,mtype);
    
else
    y=modulate(d,fcarrier,srate,mtype,(fcarrier/srate)*2*pi);
    oh.history=sprintf('modulate(%d.%02d;carrier=%g,type=%s)',...
        h.datatype,h.subtype,fcarrier,mtype);
end;
%
% downsample to 2*bandwidth
%
if (demodulate)
   [num,den]=rat((2*(srate/2-fcarrier))/srate,0.001)
   y=resample(y,num,den);
   srate=num*srate/den;
   fprintf('Down-sampled to %g samples/sec\n',srate);
   oh.frameduration=1.0/srate;
end
% 
% save to file
%
sfsputitem(fname,oh,y,length(y));
exitcode=0;
