/* ldabbot -- load Abbot RNN output into coeff/display item */

/* Mark Huckvale - University College London */

/* Version 1.0 - March 2001 */

#define PROGNAME "ldabbot"
#define PROGVERS "1.0"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH LDABBOT SFS1 UCL
.SH NAME
ldabbot -- load Abbot RNN output into coefficient/display item
.SH SYNOPSIS
.B ldabbot
(-d) abbot_rnn.out file
.SH DESCRIPTION
.I ldabbot
is a program to load the output of the Abbot phonetic neural net
into an SFS coefficient or display item.  A suitable shell script
to generate the RNN data file is:
.nf
AbbotAudioCat $1 |
plp -online |
rnnInputNorm -nobuffer |
rnnForwardDemo $ABBOT/etc/wsjcam0_plp_zap.wei - - > abbot_rnn.out
.fi
The program assumes 45 UK English phones, which are re-ordered
and re-labelled in SAMPA notation.  The binary codes are converted to
pseudo-energies for display by the formula 50*(255-c)/255.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.B -d
Save as display item.  Default: coefficient item.
.SH OUTPUT ITEMS
.IP CO
RNN phone probabilities as pseudo energies.
.IP DI
RNN phone probabilities as grey levels.
.SH HISTORY
.IP file
abbot_rnn filename.
.SH VERSION/AUTHOR
.IP 1.0
Mark Huckvale
.SH SEE ALSO
WDLOAD
*/
/*--------------------------------------------------------------------------*/

#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <math.h>
#include "sfs.h"

/* global data */
struct item_header	coitem;
struct co_rec		*co;
struct item_header	diitem;
struct di_rec		*di;
int			dodisp=0;

#define NPHONE	45

struct phone_rec {
	char	*label;
	int	pos;
} phonetab[NPHONE]={
	{ "#",  44 },
	{ "A:", 18 },
	{ "{",  12 },
	{ "V",  11 },
	{ "O:", 17 },
	{ "aU",  4 },
	{ "@",   8 },
	{ "aI",  6 },
	{ "b",  40 },
	{ "tS", 37 },
	{ "d",  39 },
	{ "D",  28 },
	{ "e@",  1 },
	{ "e",  13 },
	{ "3:", 15 },
	{ "eI",  7 },
	{ "f",  31 },
	{ "g",  38 },
	{ "h",  35 },
	{ "I@",  2 },
	{ "I",  14 },
	{ "i:", 19 },
	{ "dZ", 36 },
	{ "k",  41 },
	{ "l",  23 },
	{ "m",  26 },
	{ "n",  25 },
	{ "N",  24 },
	{ "Q",  10 },
	{ "@U",  3 },
	{ "OI",  5 },
	{ "p",  43 },
	{ "r",  21 },
	{ "s",  33 },
	{ "S",  34 },
	{ "t",  42 },
	{ "T",  32 },
	{ "U@",  0 },
	{ "U",   9 },
	{ "u:", 16 },
	{ "v",  27 },
	{ "w",  22 },
	{ "y",  20 },
	{ "z",  29 },
	{ "Z",  30 },
};

float coconvert(int c)
{
	return((float)(50.0*(255-c)/255.0));
}
int diconvert(int c)
{
	if (c > 128)
		return(0);
	else
		return((int)(0.5+15.0*(128-c)/128.0));
}

/* main program */
void main(int argc,char *argv[])
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */

	/* file variables */
	char		ifilename[SFSMAXFILENAME]; /* Abbot data file name */
	char		filename[SFSMAXFILENAME]; /* SFS data file name */
	FILE		*ip;
	int		ofid;		/* input file descriptor */
	int		i,j,t,pos;
	char		tbuf[32];

	/* decode switches */
	while ( (c = getopt(argc,argv,"Id")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Load Abbot RNN output V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'd' :	/* save as display item */
			dodisp=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-d) abbot_rnn.out file",PROGNAME);

	/* get abbot file name */
	if (optind < argc)
		strcpy(ifilename,argv[optind]);
	else
		error("no abbot RNN file specified",NULL);
	optind++;

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no SFS file specified",NULL);

	/* open input file */
	if ((ip=fopen(ifilename,"rb"))==NULL)
		error("could not open '%s'",ifilename);

	/* create output item */
	if (dodisp) {
		sfsheader(&diitem,DI_TYPE,-1,1,NPHONE+sfsstruct[DI_TYPE],
			0.016,0.0,1,0,0);
		sprintf(diitem.history,"%s(file=%s)",PROGNAME,ifilename);
		sprintf(diitem.params,"labels=");
		for (i=0;i<NPHONE;i++) {
			for (j=0;j<NPHONE;j++) {
				if (phonetab[j].pos==i) {
					sprintf(tbuf,"%s|",phonetab[j].label);
					strcat(diitem.params,tbuf);
				}
			}
		}
		diitem.params[strlen(diitem.params)-1]='\0';
		if ((ofid=sfschannel(filename,&diitem))<0)
			error("could not open output channel to '%s'",filename);
		di = (struct di_rec *)sfsbuffer(&diitem,1);
	}
	else {
		sfsheader(&coitem,CO_TYPE,-1,4,NPHONE+(sfsstruct[CO_TYPE]/4),
			0.016,0.0,1,0,0);
		sprintf(coitem.history,"%s(file=%s)",PROGNAME,ifilename);
		sprintf(coitem.params,"labels=");
		for (i=0;i<NPHONE;i++) {
			for (j=0;j<NPHONE;j++) {
				if (phonetab[j].pos==i) {
					sprintf(tbuf,"%s|",phonetab[j].label);
					strcat(coitem.params,tbuf);
				}
			}
		}
		coitem.params[strlen(coitem.params)-1]='\0';
		if ((ofid=sfschannel(filename,&coitem))<0)
			error("could not open output channel to '%s'",filename);
		co = (struct co_rec *)sfsbuffer(&coitem,1);
		co->flag=0;
		co->mix=0;
		co->gain=0;
	}

	/* copy the data across */
	i=0;
	t=0;
	while ((c=getc(ip))!=EOF) {
		if (i==0) {
			/* skip first byte */
		}
		else {
			pos = phonetab[i-1].pos;
			if (dodisp)
				di->pixel[pos] = diconvert(c);
			else
				co->data[pos] = coconvert(c);
		}
		i++;
		if (i==NPHONE+1) {
			if (dodisp) {
				di->posn = t;
				di->size = 1;
				sfswrite(ofid,1,di);
			}
			else {
				co->posn = t;
				co->size = 1;
				sfswrite(ofid,1,co);
			}
			t++;
			i=0;
		}
	}

	/* update */
	if (!sfsupdate(filename))
		error("update error on '%s'",filename);

	exit(0);
}
