/* HQanal -- High Quality formant analysis based on Melvyn Hunt's procedure */

/* M.A.Huckvale - January 1987 */

/* version 1.1 - february 1987
	- energy re-estimation restricted to voiced regions
*/
/* version 1.2 - april 1987
	- improved energy estimation in quiet regions
	- optional 6 pole analysis of voiceless regions
	- increased frame size in voiceless regions
*/
/* version 1.2s - November 1987
	- SFS version
*/

/* version 2.0 - October 2002
	- HQanal reborn
*/

#define PROGNAME "HQanal"
#define PROGVERS "2.0"
char	*progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH HQANAL SFS1 UCL
.SH NAME
HQanal - High quality formant analysis
.SH SYNOPSIS
.B HQanal
(-i item) (-o offset) (-t) (-c) (-r) (-6) file
.SH DESCRIPTION
.I HQanal
performs excitation-synchronous formant analysis of a speech waveform using LPC and
root-solving following procedure specified by Melvyn Hunt.
Input should be a 10kHz speech waveform (preferably generated from a
20kHz acquired waveform using
.I down210)
and a set of excitation markers.  The output is a set of raw formant estimates, synchronous
to the excitation markers in the voiced regions, and at 20ms intervals during unvoiced intervals.
The LPC covariance method is used, and pre-emphasis is applied to the voiced regions only.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version number.
.TP 11
.BI -i item
Select input item number.
.TP 11
.BI -o offset
Set offset in micro-seconds between tx markers and start point of analysis window.  Negative is
earlier, positive later, default is 0.
.TP 11
.B -t
Write tx item of analysis windows used in analysis.
.TP 11
.B -c
Write LPC coefficients used in analysis
.I instead of formant estimates.
.TP 11
.B -r
Write LPC analysis residual as lx waveform.
.TP 11
.B -6
Use 6-pole LPC analysis in voiceless regions.
.SH INPUT ITEMS
.IP 1.xx 11
10kHz sampled speech waveform.
.IP 3.xx 11
Excitation markers.
.SH OUTPUT ITEMS
.IP 12 11
High Quality formant estimates.
.IP 14 11
Optional LPC coefficients.
.IP 2 11
Optional residual waveform.
.SH VERSION/AUTHOR
1.2 - Mark Huckvale (from
.I lpc
and
.I rootlpc
by Lynn Whitaker)
.SH SEE ALSO
"Generation of controlled speech stimuli by pitch-synchronous LPC analysis of
natural utterances", M.J. Hunt & C.E. Harvenberg, 12th Intl. Congress on Acoustics,
Toronto Canada, 1986.
.SH BUGS
Good formant analysis relies on good data preparation as much as on
a good program.
.br
The formant item produced by
.I HQanal
is not compatible with
.I fmconv,
use
.I HQsynth
for re-synthesis.
*/
/*--------------------------------------------------------------------------*/

/* standard definitions */
#include "HQ.h"

/* global data */
struct item_header	spitem;	/* speech item header */
short			*sp;	/* speech buffer */
struct item_header	rsitem;	/* residual lx item header */
short			*rsp;	/* residual lx buffer */
struct item_header	txitem;	/* tx item header */
int			*tx;	/* tx buffer */
double			txoffset = 0.0;	/* analysis window offset from tx data */
struct item_header	pcitem;	/* LPC item header */
struct item_header	fmitem;	/* output fm item header */
struct wm_rec {
	int	posn;	/* window location in speech samples */
	int	size;	/* window size in speech samples */
	int	flag;	/* voicing flag */
};				/* window marker structure */
struct wm_rec		*wmb;	/* window marker buffer */

/* calculate number of analysis windows required */
int calcwmsize(struct item_header *txitem,int *tx,double totime)
{
	int	size=0;		/* number of windows found */
	int	wmin;		/* minimum window size */
	int	vmax;		/* number of samples in max voiced frame */
	int	uvsize;		/* number of samples in unvoiced frame */
	int	i,txval;

	/* check tx data covers whole speech */
	txval=0;
	for (i=0;i<txitem->numframes-1;i++) txval += tx[i];
	tx[txitem->numframes-1] = (int)(0.5+totime/txitem->frameduration) - txval;
	if (tx[txitem->numframes-1] <= 0)
		error("tx data does not match speech",NULL);

	/* calculate frame parameters in samples */
	wmin = (int)(0.5 + 0.002/txitem->frameduration);
	vmax = (int)(0.5 + VUVTIME/txitem->frameduration);
	uvsize = (int)(0.5 + UVFRAME/txitem->frameduration);

	/* investigate # windows required for each unvoiced period */
	for (i=0;i<txitem->numframes;i++) {
		txval = tx[i];
		if (txval >= vmax) {
			/* voiceless */
			while (txval >= 2*uvsize) {
				txval -= uvsize;
				size++;
			}
			if (txval > uvsize) size++;
		}
		size++;
	}

	return(size);
}

/* initialise window marker buffer */
int initwm(struct item_header *txitem,int *tx,struct wm_rec *wm,double spdur,double offset,int nframes)
{
	int	wmin;		/* minimum window size */
	int	vmax;		/* number of samples in max voiced frame */
	int	uvsize;		/* number of samples in unvoiced frame */
	int	i,txval;
	double	totime;		/* total time */
	double	txtime;		/* tx sample end time */
	int		wcnt=0;

	/* calculate frame parameters in samples */
	wmin = (int)(0.5 + 0.002/txitem->frameduration);
	vmax = (int)(0.5 + VUVTIME/txitem->frameduration);
	uvsize = (int)(0.5 + UVFRAME/txitem->frameduration);

	/* investigate # windows required for each unvoiced period */
	totime = offset;	/* location of zeroth tx w.r.t speech */
	txval=0;
	for (i=0;i<txitem->numframes;i++) {
		txval += tx[i];
		if (txval >= vmax) {	/* unvoiced section */
			while (txval >= 2*uvsize) {
				/* initialise approx 20ms window marker */
				txval -= uvsize;
				txtime = totime + uvsize*txitem->frameduration;
				wm->posn = (int)((totime > 0.0) ? 0.5 + (totime/spdur) : 0.0);
				wm->size = (int)(0.5 + txtime/spdur - wm->posn);
				wm->flag = 0;
				wm++;
				wcnt++;
				totime = txtime;
			}
			if (txval > uvsize) {
				/* initialise approx 10ms window marker */
				txtime = totime + (txval/2)*txitem->frameduration;
				txval -= txval/2;
				wm->posn = (int)((totime > 0.0) ? 0.5 + (totime/spdur) : 0.0);
				wm->size = (int)(0.5 + txtime/spdur - wm->posn);
				wm->flag = 0;
				wm++;
				wcnt++;
				totime = txtime;
			}
			wm->flag = 0;
		}
		else {
			/* voiced section */
			if (txval > wmin) {
				wm->flag = 1;
				/* initialise window marker data from tx */
				txtime = totime + txval*txitem->frameduration;
				wm->posn = (int)((totime > 0.0) ? 0.5 + (totime/spdur) : 0.0);
				wm->size = (int)(0.5 + txtime/spdur - wm->posn);
				wm++;
				wcnt++;
				totime = txtime;
				txval=0;
			}
		}
	}

	if (txval > 0) {
		/* initialise window marker data from tx */
		txtime = totime + txval*txitem->frameduration;
		wm->posn = (int)((totime > 0.0) ? 0.5 + (totime/spdur) : 0.0);
		wm->size = (int)(0.5 + txtime/spdur - wm->posn);
		wm++;
		wcnt++;
		totime = txtime;
		txval=0;
	}

	/* fix-up last tx sample to match waveform length */
	wm--;
	wm->size = nframes - wm->posn;
	return(wcnt);
}

/* lpc analysis subroutine */
void lpc(short *sp,struct wm_rec *wm,struct co_rec *lpcrec,int ncoeff,int lastvoice)
{
	float		x[NCOEFF+MAXWINDOW],*xp;
					/* speech data for analysis */
	double		gain;		/* lpc frame gain */
	int			i,scnt;
	short		*s;
	LTIState	ltis;

	/* put speech data into floating point buffer */
	xp = x;
	s = &sp[wm->posn - ncoeff];
	scnt = wm->size + ncoeff;
	for (i=0;i<ncoeff;i++,s++) {	/* first 10 samples */
		/* start with low-level random noise - to help stabilise analysis */
		*xp = (float)((rand() % 65536) / 32768.0 - 0.5);
		/* add in signal - if voicing boundary not crossed */
		if ((s >= sp) && (wm->flag == lastvoice)) *xp += *s;
		xp++;
	}
	for (i=ncoeff;i<scnt;i++,s++) {	/* rest of window */
		/* start with low-level random noise - to help stabilise analysis */
		*xp = (float)((rand() % 65536) / 32768 - 0.5);
		/* add in signal */
		*xp++ += *s;
	}

	/* pre-emphasise if voiced */
	if (wm->flag) {
		xp = x+scnt-1;
		for (i=1;i<scnt;i++,xp--) *xp -= (float)(PREEMP * *(xp-1));
	}

	/* do covariance analysis */
	if (LPCCovariance(x,scnt,ncoeff,&ltis,&gain)!=0)
		fprintf(stderr,"Warning: LPC covariance() fails\n");

	/* set up lpc record */
	lpcrec->posn = wm->posn;
	lpcrec->size = wm->size;
	lpcrec->flag = wm->flag;
	lpcrec->mix  = (float)wm->flag;
	lpcrec->gain = (float)((gain > 0.0) ? sqrt(gain) : 0.0);
	for (i=0;i<ncoeff;i++) lpcrec->data[i] = (float)ltis.b[i+1];
	for (i=ncoeff;i<NCOEFF;i++) lpcrec->data[i] = 0.0;

}

/* calculate LPC residual waveform */
void resid(struct co_rec *lpcrec,int ncoeff,short *sp,short *rsp,int lastvoice)
{
	float	x[NCOEFF+MAXWINDOW],*xp;
				/* speech data for analysis */
	short	*s;		/* pointer to input speech */
	float	pred;		/* predicted waveform value */
	int	i,j,scnt;

	/* put speech data into floating point buffer */
	xp = x;
	s = &sp[lpcrec->posn - ncoeff];
	scnt = lpcrec->size + ncoeff;
	for (i=0;i<ncoeff;i++,s++) {	/* first 10 samples */
		/* load with old signal - if voicing boundary not crossed */
		if ((s >= sp) && (lpcrec->flag == lastvoice))
			*xp++ = *s;
		else
			*xp++ = 0.0;
	}
	for (i=ncoeff;i<scnt;i++,s++) {	/* rest of window */
		/* load with signal */
		*xp++ = *s;
	}

	/* pre-emphasise if voiced */
	if (lpcrec->flag) {
		xp = x+scnt-1;
		for (i=1;i<scnt;i++,xp--) *xp -= (float)(PREEMP * *(xp-1));
	}

	/* calculate residual sample by sample */
	xp = x + ncoeff;
	for (i=0;i<lpcrec->size;i++) {
		pred = 0.0;
		for (j=0;j<ncoeff;j++) pred -= lpcrec->data[j] * xp[i-j-1];
		rsp[lpcrec->posn+i] = (short)(xp[i] - pred);
	}

}

/* calculate energy in speech waveform */
double logenergy(struct co_rec *lpcrec,short *sp)
{
	int		i,len;
	short		*s;
	double		mean=0,sum=0.0001,val;

	s = &sp[lpcrec->posn];
	len = (int)(0.9 * lpcrec->size);	/* over first 90% of window only */
	/* find mean value */
	for (i=0;i<len;i++,s++) mean += (double)*s;
	mean /= len;
	/* find sum squared values */
	s = &sp[lpcrec->posn];
	for (i=0;i<len;i++,s++) {
		val = (double)*s - mean;
		sum += val*val;
	}
	return(10.0*log10(sum));
}

/* calculate amplitude at given frequency in LPC frame */
double calcamplitude(struct co_rec *lpcrec,int ncoeff,double tau,double freq)
{
	double		sumsin;		/* sum of sine products */
	double		sumcos;		/* sum of cosine products */
	int		i;
	double		omega;		/* 2.pi.f.t */
	double		amp;		/* amplitude */

	/* initialise omega */
	omega = 8.0 * atan(1.0) * freq * tau;

	/* initialise sine and cosine products */
	sumsin = 0.0;
	sumcos = 1.0;

	/* sum sine and cosine products */
	for (i=0;i<ncoeff;i++) {
		sumsin += lpcrec->data[i] * sin(omega * (i+1));
		sumcos += lpcrec->data[i] * cos(omega * (i+1));
	}

	/* return amplitude */
	amp = sqrt(sumsin*sumsin + sumcos*sumcos);
	return(amp);
}

/* calculate formant amplitudes */
void fmamp(struct fm_rec *formrec,struct co_rec *lpcrec,int ncoeff,double spdur)
{
	double		amp;
	int		i;

	for (i=0;i<formrec->npeaks;i++) {
		amp = calcamplitude(lpcrec,ncoeff,formrec->formant[i].freq,spdur);
		formrec->formant[i].amp = (float)(20.0*log10(lpcrec->gain/amp));
	}
}


/* write analysis window positions as tx item */
void writetx(struct wm_rec *wm,int cnt,double spdur,double offset,int sptype,int txtype,char *filename)
{
	struct item_header 	item;
	int			i,fid;

	sfsheader(&item,TX_TYPE,0,4,1,spdur,0.0,0,0,0);
	sprintf(item.history,"%s/TX(%d.%02d,%d.%02d;offset=%g)",
		PROGNAME,SP_TYPE,sptype,TX_TYPE,txtype,offset);
	strcpy(item.params,"");

	/* open output channel */
	if ((fid=sfschannel(filename,&item))<0)
		error("could not create temporary file",NULL);

	/* write data */
	for (i=0;i<item.numframes;i++,wm++)
		sfswrite(fid,1,&wm->size);

	/* file will get updated with main data */
	return;
}

/* LPC coefficient root-solving */
void roots(struct co_rec *lpcrec,int degree,struct fm_rec *formrec,double sdur)
{
	COMPLEX		poly[MAXWINDOW];	/* polynomial to be solved */
	COMPLEX		root[MAXWINDOW];	/* roots */
	double		twopi;		/* real constant */
	double		omega,rsq;
	struct fm_rec_array fspec[MAXWINDOW],ftemp;
					/* formant freq, band, amp specification */
	int		i,j,npeak=0;

	/* copy LPC coefficients into polynomial */
	poly[0].re = 1.0; poly[0].im = 0.0;
	for (i=0;i<degree;i++) {
		poly[i+1].re = lpcrec->data[i];
		poly[i+1].im = 0.0;
	}

	/* do root-solving using DSP library */
	memset(root,0,sizeof(root));
	AllRoots(poly,degree,root);

	/* convert roots to pole positions */
	twopi = 8.0*atan(1.0);
	for (i=0;i<=degree;i++) if (root[i].im > 1.0E-10) {
		omega = atan2(root[i].im,root[i].re);
 		fspec[npeak].freq = (float)fabs(omega/(twopi*sdur));
 		rsq = sqrt(root[i].re*root[i].re+root[i].im*root[i].im);
 		if (rsq > 1.0) rsq = 1.0/rsq;
		fspec[npeak].band = (float)fabs(log(rsq)/(twopi*sdur));
		if ((fspec[npeak].freq > FREQLIMIT) && (fspec[npeak].band < BANDLIMIT)) npeak++;
	}

	/* sort pole positions in terms of bandwidth */
	for (i=1;i<npeak;i++) {
		j = i;
		ftemp = fspec[i];
		while ((j>0) && (ftemp.band < fspec[j-1].band)) {
			fspec[j]=fspec[j-1];
			j--;
		}
		fspec[j]=ftemp;
	}

	/* sort first NFORMANT pole positions in terms of frequency */
	for (i=1;(i<NFORMANT) && (i<npeak);i++) {
		j = i;
		ftemp = fspec[i];
		while ((j>0) && (ftemp.freq < fspec[j-1].freq)) {
			fspec[j]=fspec[j-1];
			j--;
		}
		fspec[j]=ftemp;
	}

	/* initialise formant record */
	formrec->posn = lpcrec->posn;
	formrec->size = lpcrec->size;
	formrec->flag = lpcrec->flag;
	formrec->gain = lpcrec->gain;
	formrec->npeaks = 0;
	for (i=0;i<NFORMANT;i++) {
		formrec->formant[i].freq=0.0;
		formrec->formant[i].band=0.0;
		formrec->formant[i].amp=0.0;
	}

	/* put upto NFORMANT peaks into formant structure */
	for (i=0;(i<NFORMANT) && (i<npeak);i++) {
		formrec->formant[formrec->npeaks].freq = fspec[i].freq;
		formrec->formant[formrec->npeaks].band = fspec[i].band;
		formrec->npeaks++;
	}

}


void fmtopc(struct fm_rec *formrec,struct co_rec *lpcrec,int ncoeff)
{
	COMPLEX		c[NFORMANT];	/* complex pole positions */
	double		aa[40],a[40];	/* autoregressive working data */
	double		twopi;
	double 		x,w,zr;
	int		i,ii,j;

	twopi = 8.0 * atan(1.0);

	/* calculate complex pole positions */
	j=0;
	for (i=0; i<formrec->npeaks; i++){
		while ((j<ncoeff) && (formrec->formant[j].freq==0)) j++;
		x = exp(-twopi * formrec->formant[j].band / SAMPFREQ);
		w = formrec->formant[j].freq * twopi / SAMPFREQ;
		zr = x * cos(w);
		c[i].re = -2.0*zr;
		c[i].im = x*x;
		j++;
	}

	/* calculate AR coefficients from complex poles */
	/* - code from /usr/dbase/rel2/gec/fmconv.src/subproc.c */
	a[0] = 1.0;
	a[1] = c[0].re;
	a[2] = c[0].im;
	ii = 2;
	for (i=1; i<formrec->npeaks; i++) {
		aa[0] = 1.0;
		aa[1] = a[1] + c[i].re;
		for (j=2; j<=ii; j++)
			aa[j] = a[j] + a[j-1] * c[i].re + a[j-2] * c[i].im;
		aa[ii+1] = a[ii] * c[i].re + a[ii-1] * c[i].im;
		aa[ii+2] = a[ii] * c[i].im;
		ii += 2;
		for (j=1; j<=ii; j++) a[j] = aa[j];
	}
	for (ii++; ii<=ncoeff; ii++) a[ii]=0.0;	/* clear unused coeffs */

	/* create LPC record */
	lpcrec->posn = formrec->posn;
	lpcrec->size = formrec->size;
	lpcrec->flag = formrec->flag;
	lpcrec->mix  = (float)formrec->flag;
	lpcrec->gain = formrec->gain;
	for (i=0;i<ncoeff;i++) lpcrec->data[i] = (float)a[i+1];

}


/* main program */
void main(int argc,char *argv[])
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it,ity;		/* item selections */
	char		*ty;		/* item specification */
	char		*sptype = "0";	/* default sub-type */
	char		*txtype = "0";	/* default sub-type */
	int		txout = 0;	/* tx output required */
	int		pcout = 0;	/* lpc coeff output required */
	int		rsout = 0;	/* residual output required */
	int		sixpole=0;	/* 6 pole-analysis in voiceless regions */
	/* file variables */
	char		filename[SFSMAXFILENAME];	/* dbase file name */
	int		fid,pcfid,fmfid;
	/* data variables */
	int		wmsize;		/* no. of analysis windows */
	struct co_rec	*lpcrec;		/* standard coefficient record */
	struct fm_rec	*formrec;	/* standard formant record */
	int		i,lastvoice,npole,j;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:o:tcr6")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: High Quality formant analysis V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					sptype = ty;
				else if (it == TX_TYPE)
					txtype = ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'o' :	/* tx offset */
			txoffset = ((double)atoi(optarg))/1E6;
			break;
		case 't' :	/* tx output required */
			txout++;
			break;
		case 'c' :	/* LPC coefficients required */
			pcout++;
			break;
		case 'r' :	/* residual waveform required */
			rsout++;
			break;
		case '6' :	/* 6 pole analysis */
			sixpole++;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-o offset) (-t) (-c) (-r) (-6) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open file */
	if ((fid=sfsopen(filename,"w",NULL))<0)
		error("access error on %s",filename);

	/* load sp data */
	if (!sfsitem(fid,SP_TYPE,sptype,&spitem))
		error("could not find input SP item in %s",filename);
	if ((sp=(short *)sfsbuffer(&spitem,spitem.numframes+NCOEFF))==NULL)
		error("could not get memory buffer for SP item",NULL);
	if (sfsread(fid,0,spitem.numframes,sp) != spitem.numframes)
		error("read error on input file",NULL);
	if (100*(int)(0.01/spitem.frameduration) != SAMPFREQ)
		error("speech not sampled at 10kHz",NULL);

	/* load original tx */
	if (!sfsitem(fid,TX_TYPE,txtype,&txitem))
		error("could not find input TX item in %s",filename);
	if ((tx=(int *)sfsbuffer(&txitem,txitem.numframes))==NULL)
		error("could not get memory buffer for TX item",NULL);
	if (sfsread(fid,0,txitem.numframes,tx) != txitem.numframes)
		error("read error on input file",NULL);
	sfsclose(fid);

	/* calculate size of window marker buffer */
	wmsize = calcwmsize(&txitem,tx,spitem.numframes*spitem.frameduration);

	/* allocate window marker buffer */
	wmb = (struct wm_rec *) calloc(wmsize,sizeof(struct wm_rec));

	/* initialise window marker buffer */
	wmsize = initwm(&txitem,tx,wmb,spitem.frameduration,txoffset,spitem.numframes);

	/* write tx item if required */
	if (txout) writetx(wmb,wmsize,spitem.frameduration,txoffset,spitem.subtype,txitem.subtype,filename);

	/* allocate speech residual buffer if required */
	if (rsout) {
		rsp = (short *) sfsbuffer(&spitem,spitem.numframes);
		if (rsp == NULL) error("cannot allocate memory for residual",NULL);
		for (i=0;i<spitem.numframes;i++) rsp[i]=0;
	}

	/* create new item header for PC if required, or FM otherwise */
	if (pcout) {
		/* initialise PC item header */
		sfsheader(&pcitem,PC_TYPE,-1,4,NCOEFF+sfsstruct[PC_TYPE]/4,
			spitem.frameduration,spitem.offset,0,0,1);
		sprintf(pcitem.history,"%s/PC(%d.%02d,%d.%02d;offset=%g,np=%d%s)",
			PROGNAME,spitem.datatype,spitem.subtype,
			txitem.datatype,txitem.subtype,txoffset,NCOEFF,
			(sixpole) ? ",v-=6" : "");
		strcpy(pcitem.params,"");

		/* open channel for PC coefficients */
		if ((pcfid=sfschannel(filename,&pcitem))<0)
			error("could not create temporary file",NULL);

		/* get output record buffer */
		if ((lpcrec=(struct co_rec *)sfsbuffer(&pcitem,1))==NULL)
			error("could not create memory buffer",NULL);
	}
	else {
		/* initialise FM item header */
		sfsheader(&fmitem,FM_TYPE,-1,4,NFORMANT*3+sfsstruct[PC_TYPE]/4,
			spitem.frameduration,spitem.offset,0,0,1);
		sprintf(fmitem.history,"%s/FM(%d.%02d,%d.%02d;offset=%g%s)",
			PROGNAME,spitem.datatype,spitem.subtype,
			txitem.datatype,txitem.subtype,txoffset,
			(sixpole) ? ",v-=6" : "");
		strcpy(fmitem.params,"");

		/* open channel for fm item */
		if ((fmfid=sfschannel(filename,&fmitem))<0)
			error("could not create temporary file",NULL);

		/* get output record buffers */
		pcitem.datatype=PC_TYPE;
		pcitem.floating= -1;
		pcitem.datasize=4;
		pcitem.framesize=NCOEFF+sfsstruct[PC_TYPE]/4;
		if ((lpcrec=(struct co_rec *)sfsbuffer(&pcitem,1))==NULL)
			error("could not create memory buffer",NULL);
		if ((formrec=(struct fm_rec *)sfsbuffer(&fmitem,1))==NULL)
			error("could not create memory buffer",NULL);
	}

	/* process speech frame by frame */
	lastvoice = 0;	/* record of last frame voicing */
	for (i=0;i<wmsize;i++) {

		/* get number of poles for window */
		npole = (sixpole && !wmb[i].flag) ? 6 : NCOEFF;

		/* perform LPC analysis on this window */
		lpc(sp,&wmb[i],lpcrec,npole,lastvoice);

		/* write lpc data to file if required */
		if (pcout) {
			if (sfswrite(pcfid,1,lpcrec) != 1)
				error("write error on output file",NULL);

			/* calculate residual if required */
			if (rsout) resid(lpcrec,npole,sp,rsp,lastvoice);
		}

		/* write formant data to file if required */
		if (!pcout) {
/* printf("In:  ");
for (j=0;j<10;j++) printf("%.3f,",lpcrec->data[j]);
printf("\n"); */
			/* perform root solving */
			roots(lpcrec,npole,formrec,spitem.frameduration);

			/* convert back to LPC coefficients */
			fmtopc(formrec,lpcrec,npole);
/* printf("Out: ");
for (j=0;j<10;j++) printf("%.3f,",lpcrec->data[j]);
printf("\n"); */

			/* calculate frame energy from original speech */
			formrec->gain = (float)logenergy(lpcrec,sp);

			/* calculate formant amplitudes */
			fmamp(formrec,lpcrec,npole,fmitem.frameduration);

			/* write formant record */
			if (sfswrite(fmfid,1,formrec) != 1)
				error("write error on output file",NULL);

			/* calculate residual if required */
			if (rsout) resid(lpcrec,npole,sp,rsp,lastvoice);
		}

		/* print progress */
		if (ttytest()) {
			printf("\rFrame %d/%d",i+1,wmsize);
			fflush(stdout);
		}

		/* keep record of last voicing decision */
		lastvoice = lpcrec->flag;
	}

	/* update */
	if (!sfsupdate(filename))
		error("update error on %s",filename);

	/* also write residual if required */
	if (rsout) {
		/* re-open input file */
		fid = sfsopen(filename,"r",NULL);

		/* find PC or FM item referenced */
		if (pcout) {
			sfsitem(fid,PC_TYPE,"0",&pcitem);
			it = PC_TYPE;
			ity = pcitem.subtype;
		}
		else {
			sfsitem(fid,FM_TYPE,"0",&fmitem);
			it = FM_TYPE;
			ity = fmitem.subtype;
		}

		/* initialise header from speech header */
		sfsheader(&rsitem,
			LX_TYPE,
			spitem.floating,
			spitem.datasize,
			spitem.framesize,
			spitem.frameduration,
			spitem.offset,
			spitem.windowsize,
			spitem.overlap,
			spitem.lxsync);
		sprintf(rsitem.history,"%s/LX(%d.%02d,%d.%02d;offset=%g)",
			PROGNAME,spitem.datatype,spitem.subtype,
			it,ity,txoffset);
		strcpy(rsitem.params,"");

#if 0
		/* remove low freq */
		for (i=1;i<spitem.numframes;i++)
			rsp[i] = (short)(rsp[i]-0.05*rsp[i-1]);
#endif

		/* write header and waveform to output file */
		putitem(filename,&rsitem,spitem.numframes,rsp);
	}

	/* that's all folks ... */
	printf("\r                     \r");fflush(stdout);
	exit(0);
}

