/* analign -- wrapper for phonetic alignment with HTK HVite */

/* M.A.Huckvale - May 2004 */

/* version 1.0 */

/* version 1.1 - fix bug in WAV save for esamp > length of speech */

#define PROGNAME "analign"
#define PROGVERS "1.1"
char *progname=PROGNAME;

/*-------------------------------------------------------------------*/
/**MAN
.TH ANALIGN 1 UCL SFS
.SH NAME
analign - automatic alignment of phonetic transcription to speech signal
.SH SYNOPSIS
.B analign
(-i item) (-f|-p) (-l pauselabel) (-A|-J) file
.SH DESCRIPTION
.I analign
perfoms phonetic alignment of transcription to a speech signal.
It takes as input an annotation item and a speech item and
generates an aligned annotation item.
It has two modes of operation.  In its "fixed label" mode, the boundaries
of the input annotations are not changed, and alignment only takes place
within each labelled region.  In this case the labels are expected to
consist of a sequence of transcription symbols.  In its "fixed pause"
mode, labels identified as pauses are not moved by the alignment, so that
alignment only takes places from one pause to the next.  In this case the
labels are expected to consist of individual segments.  You can use the
first mode to get a rough alignment, and the second mode to refine it.
.I analign
is implemented as a wrapper function for an hidden markov model
alignment performed by a version of the HVite tool that is
part of the Cambridge HTK toolkit.  The HMMs used are stored
in $(SFSBASE)/data/analign.hmm, and the HTK configuration file in
$(SFSBASE)/data/analign.cfg. These can be changed if required.
.PP
By default, input transcription is expected to be in SAMPA
format. Symbols should be separated by spaces.
Stress markers are stripped out.  Options allow input
of ARPABET and JSRU symbols - translations take place both before
and after alignment.
.PP
.I Options:
.TP 11
.B -I
Identify the program name and version.
.TP 11
.BI -i item
Select input item number.
.TP 11
.B -f
Set fixed label mode.  Alignment is only performed within the
boundaries of the input labels.
.TP 11
.B -p
Set fixed pause mode.  Alignment is only performed within the
boundaries of labels identified as pauses.
.TP 11
.BI -l label
Specify the label used to identify pauses in fxied pause mode.
Default is the SAMPA pause symbol "...".
.TP 11
.B -A
Input symbols are in ARPABET format, as used in the BEEP dictionary.
.TP 11
.B -J
Input symbols are in JSRU format.
.SH FILES
.TP 11
.B analign.hmm
HMM definitions in HTK format
.TP 11
.B analign.cfg
HMM configuration in HTK format
.TP 11
.B analign.lst
List of HMM models
.TP 11
.B analign.dct
Translation from SAMPA to phone names used on HMMs
.SH VERSION/AUTHOR
1.0 - Mark Huckvale
*/
/*---------------------------------------------------------------*/

/* include files */
#include "SFSCONFG.h"
#ifdef _WIN32
#include <windows.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <malloc.h>
#include <math.h>
#include "sfs.h"

#ifndef _WIN32
#define DeleteFile(pfilename)	unlink(pfilename)
#endif

/* global data */
struct item_header	spitem;		/* input speech item */
struct item_header	anitem;		/* input annotation item */
struct an_rec		*an;
struct item_header	opitem;		/* output annotation item */
struct an_rec		*oan;

char	filename[SFSMAXFILENAME];
int		dofixedlabel=0;
int		dofixedpause=0;
char	pauselabel[256]="...";
int		doarpa=0;
int		dojsru=0;
int		ocnt,opos;

char	tempname[SFSMAXFILENAME];	/* temporary directory */
char	lfilename[SFSMAXFILENAME];	/* file list file */
int		processid;
char	wfilename[SFSMAXFILENAME];	/* WAV file name */
char	pfilename[SFSMAXFILENAME];	/* LAB/REC file name */
char	basename[SFSMAXFILENAME];	/* executable directory name */
char	fbuf[SFSMAXFILENAME];	/* executable directory name */

/* save a string */
char *strsave(char *str)
{
	char *ptr = malloc(strlen(str)+1);
	strcpy(ptr,str);
	return(ptr);
}

/* convert a SAMPA symbol to ARPABET */
char *sampa2arpa(char *sym)
{
	switch (sym[0]) {
	case 'a':	return((sym[1]=='I')?"ay":((sym[1]=='U')?"aw":sym));
	case 'd':	return((sym[1]=='Z')?"jh":sym);
	case 'e':	return((sym[1]=='I')?"ey":((sym[1]=='@')?"ea":"eh"));
	case 'h':	return("hh");
	case 'i':	return("iy");
	case 'j':	return("y");
	case 't':	return((sym[1]=='S')?"ch":sym);
	case 'u':	return("uw");
	case 'A':	return("aa");
	case 'D':	return("dh");
	case 'I':	return((sym[1]=='@')?"ia":"ih");
	case 'N':	return("ng");
	case 'O':	return((sym[1]=='I')?"oy":"ao");
	case 'Q':	return("oh");
	case 'S':	return("sh");
	case 'T':	return("th");
	case 'U':	return((sym[1]=='@')?"ua":"uh");
	case 'V':	return("ah");
	case 'Z':	return("zh");
	case '3':	return("er");
	case '{':	return("ae");
	case '@':	return((sym[1]=='U')?"ow":"ax");
	case '/':	return("sil");
	default:
		return(sym);
	}
	return(sym);
}

/* convert ARPABET symbol to SAMPA */
char *arpa2sampa(char *sym)
{
	if (sym[0]=='a') {
		switch (sym[1]) {
		case 'h':	return("V");
		case 'y':	return("aI");
		case 'w':	return("aU");
		case 'o':	return("O:");
		case 'e':	return("{");
		case 'a':	return("A:");
		case 'x':	return("@");
		default:	return(sym);
		}
	}
	else if (sym[0]=='e') {
		switch (sym[1]) {
		case 'a':	return("e@");
		case 'y':	return("eI");
		case 'h':	return("e");
		case 'r':	return("3:");
		default:	return(sym);
		}
	}
	else if (sym[0]=='i') {
		switch (sym[1]) {
		case 'y':	return("i:");
		case 'a':	return("I@");
		case 'h':	return("I");
		default:	return(sym);
		}
	}
	else if (sym[0]=='o') {
		switch (sym[1]) {
		case 'h':	return("Q");
		case 'y':	return("OI");
		case 'w':	return("@U");
		default:	return(sym);
		}
	}
	else if (sym[0]=='u') {
		switch (sym[1]) {
		case 'a':	return("U@");
		case 'h':	return("U");
		case 'w':	return("u:");
		default:	return(sym);
		}
	}
	else if (strcmp(sym,"sil")==0)
		return("/");
	else switch (sym[0]) {
		case 'd':	return((sym[1]=='h')?"D":"d");
		case 't':	return((sym[1]=='h')?"T":"t");
		case 's':	return((sym[1]=='h')?"S":"s");
		case 'z':	return((sym[1]=='h')?"Z":"z");
		case 'n':	return((sym[1]=='g')?"N":"n");
		case 'h':	return("h");
		case 'y':	return("j");
		case 'j':	return("dZ");
		case 'c':	return("tS");
		default:	return(sym);
	}
}

/* convert a SAMPA symbol to JSRU */
char *sampa2jsru(char *sym)
{
	switch (sym[0]) {
	case 'a':	return((sym[1]=='I')?"ie":((sym[1]=='U')?"ou":sym));
	case 'd':	return((sym[1]=='Z')?"j":sym);
	case 'e':	return((sym[1]=='I')?"ai":((sym[1]=='@')?"ei":"e"));
	case 'h':	return("h");
	case 'i':	return("ee");
	case 'j':	return("y");
	case 't':	return((sym[1]=='S')?"ch":sym);
	case 'u':	return("uu");
	case 'A':	return("ar");
	case 'D':	return("dh");
	case 'I':	return((sym[1]=='@')?"ia":"i");
	case 'N':	return("ng");
	case 'O':	return((sym[1]=='I')?"oi":"aw");
	case 'Q':	return("o");
	case 'S':	return("sh");
	case 'T':	return("th");
	case 'U':	return((sym[1]=='@')?"ur":"oo");
	case 'V':	return("u");
	case 'Z':	return("zh");
	case '3':	return("er");
	case '{':	return("aa");
	case '@':	return((sym[1]=='U')?"oa":"a");
	case '/':	return("q");
	case '?':	return("gx");
	case '%':	return("'");
	default:
		return(sym);
	}
	return(sym);
}

char *strlower(char *str)
{
	static char buf[1024];
	int	i;
	for (i=0;(i<1024)&&(str[i]);i++) buf[i]=tolower(str[i]);
	buf[i]='\0';
	return(buf);
}

/* convert JSRU symbol to SAMPA */
char *jsru2sampa(char *sym)
{
	if (tolower(sym[0])=='a') {
		switch (tolower(sym[1])) {
		case 'i':	return("eI");
		case 'r':	return("A:");
		case 'w':	return("O:");
		case 'a':	return("{");
		case '\0':	return("@");
		default:	return(strlower(sym));
		}
	}
	else if (tolower(sym[0])=='e') {
		switch (tolower(sym[1])) {
		case 'i':	return("e@");
		case '\0':	return("e");
		case 'e':	return("i:");
		case 'y':	return("i:");
		case 'r':	return("3:");
		default:	return(strlower(sym));
		}
	}
	else if (tolower(sym[0])=='i') {
		switch (tolower(sym[1])) {
		case 'e':	return("aI");
		case 'a':	return("I@");
		case '\0':	return("I");
		default:	return(strlower(sym));
		}
	}
	else if (tolower(sym[0])=='o') {
		switch (tolower(sym[1])) {
		case 'u':	return("aU");
		case 'i':	return("OI");
		case '\0':	return("Q");
		case 'o':	return("U");
		case 'a':	return("@U");
		default:	return(strlower(sym));
		}
	}
	else if (tolower(sym[0])=='u') {
		switch (tolower(sym[1])) {
		case 'u':	return("u:");
		case 'r':	return("U@");
		case '\0':	return("V");
		default:	return(strlower(sym));
		}
	}
	else switch (tolower(sym[0])) {
		case 'd':	return((tolower(sym[1])=='h')?"D":"d");
		case 't':	return((tolower(sym[1])=='h')?"T":"t");
		case 's':	return((tolower(sym[1])=='h')?"S":"s");
		case 'z':	return((tolower(sym[1])=='h')?"Z":"z");
		case 'n':	return((tolower(sym[1])=='g')?"N":"n");
		case 'g':	return((tolower(sym[1])=='x')?"?":"g");
		case 'h':	return("h");
		case 'y':	return("j");
		case 'j':	return("dZ");
		case 'c':	return("tS");
		case 'q':	return("/");
		case '\'':	return("%");
		case '"':	return("");
		case '-':	return("");
		default:	return(strlower(sym));
	}
}

#pragma pack(2)
struct riff_format_rec {
	unsigned short	format;
	unsigned short	nchan;
	unsigned int	srate;
	unsigned int	brate;
	unsigned short	balign;
	unsigned short	nbits;
	unsigned short	xsize;
};
#pragma pack()

void wavsave(char *filename,int fid,int ssamp,int esamp)
{
	FILE	*op;
	struct riff_format_rec rifform;
	char	riffstr[8];
	unsigned int	size,numf;
	unsigned short xbuf[4096];
	int		bstart,bcount,len,cnt;
	int		i;

	if ((op=fopen(filename,"wb"))==NULL)
		error("unable to open '%s'",filename);

	bcount = 2*(esamp-ssamp);

	/* write headers */
	memcpy(riffstr,"RIFF",4);
	fwrite(riffstr,1,4,op);
	size=20 + sizeof(struct riff_format_rec) + bcount;
	fwrite(&size,1,4,op);
	memcpy(riffstr,"WAVE",4);
	fwrite(riffstr,1,4,op);
	memcpy(riffstr,"fmt ",4);
	fwrite(riffstr,1,4,op);
	size=sizeof(struct riff_format_rec);
	fwrite(&size,1,4,op);
	rifform.format=1;
	rifform.nchan=1;
	rifform.srate = (int)(0.5+1.0/spitem.frameduration);
	rifform.brate = rifform.srate*2;
	rifform.balign=2;
	rifform.nbits=16;
	rifform.xsize=0;
	fwrite(&rifform,1,sizeof(struct riff_format_rec),op);
	memcpy(riffstr,"data",4);
	fwrite(riffstr,1,4,op);
	fwrite(&bcount,1,4,op);

	/* copy data */
	for (i=ssamp;i<esamp;) {
		cnt=esamp-i;
		cnt=sfsread(fid,i,(cnt>4096)?4096:cnt,xbuf);
		if (cnt<=0) {
			fprintf(stderr,"read error on input file\n");
			break;
		}
		if (fwrite((char *)xbuf,2,cnt,op)!=cnt)
			error("write error on '%s'\n",filename);
		i += cnt;
	}

	fclose(op);
}

/* label save */
void labsave(char *filename,char *label)
{
	FILE	*op;
	char	buff[2048];
	char	*p,*sym;
	int		cnt;

	if ((op=fopen(filename,"w"))==NULL)
		error("unable to open '%s'",filename);

	strcpy(buff,label);
	p=strtok(buff," ");
	cnt=0;
	while (p && *p) {
		if (doarpa)
			sym = arpa2sampa(p);
		else if (dojsru)
			sym = jsru2sampa(p);
		else
			sym = p;
		if (*sym) {
			fprintf(op,"%d %d %s\n",cnt,cnt+1,sym);
			cnt++;
			ocnt++;
		}
		p = strtok(NULL," ");
	}

	fclose(op);
}

/* label save - multiple annotations */
void labsavemult(char *filename,int start,int stop)
{
	FILE	*op;
	char	buff[2048];
	char	*p,*sym;
	int		cnt;
	int		i;

	if ((op=fopen(filename,"w"))==NULL)
		error("unable to open '%s'",filename);

	for (i=start;i<stop;i++) {
		strcpy(buff,an[i].label);
		p=strtok(buff," ");
		cnt=0;
		while (p && *p) {
			if (doarpa)
				sym = arpa2sampa(p);
			else if (dojsru)
				sym = jsru2sampa(p);
			else
				sym = p;
			if (*sym) {
				fprintf(op,"%d %d %s\n",cnt,cnt+1,sym);
				cnt++;
				ocnt++;
			}
			p = strtok(NULL," ");
		}
	}

	fclose(op);
}

/* label load */
void labload(char *filename,int offset,int limit)
{
	FILE	*ip;
	int		s,e;
	char	l[256],*p;
	int		num,cnt;

	if ((ip=fopen(filename,"r"))!=NULL) {
		/* open recognised file OK */
		while ((opos<ocnt) && (fscanf(ip,"%d %d %s\n",&s,&e,l)==3)) {
			oan[opos].posn = s/1000+offset;
			oan[opos].size = (e-s)/1000;
			if (doarpa)
				oan[opos].label = strsave(sampa2arpa(l));
			else if (dojsru)
				oan[opos].label = strsave(sampa2jsru(l));
			else
				oan[opos].label = strsave(l);
			opos++;
		}

		fclose(ip);
	}
	else {
		/* no recognition file - load original label file, evenly spaced */
		strcpy(l,filename);
		p=strrchr(l,'.');
		if (p) strcpy(p,".lab");
		if ((ip=fopen(l,"r"))!=NULL) {
			fprintf(stderr,"Could not open '%s', loading from '%s' instead, evenly spaced\n",filename,l);
			/* count labels */
			num=0;
			while (fscanf(ip,"%d %d %s\n",&s,&e,l)==3) num++;
			rewind(ip);
			cnt=0;
			while ((opos<ocnt) && (fscanf(ip,"%d %d %s\n",&s,&e,l)==3)) {
				oan[opos].posn = offset+cnt*(limit-offset)/num;;
				oan[opos].size = (limit-offset)/num;
				if (doarpa)
					oan[opos].label = strsave(sampa2arpa(l));
				else if (dojsru)
					oan[opos].label = strsave(sampa2jsru(l));
				else
					oan[opos].label = strsave(l);
				cnt++;
				opos++;
			}
			fclose(ip);
		}
		else
			error("could not open '%s'",filename);

	}
}

/* compare annotations for time */
int ancomp(const void *ep1,const void *ep2)
{
	const struct an_rec *a1 = (const struct an_rec *)ep1;
	const struct an_rec *a2 = (const struct an_rec *)ep2;
	if (a1->posn < a2->posn)
		return(-1);
	else if (a1->posn > a2->posn)
		return(1);
	else
		return(0);
}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */

	/* processing variables */
	int		fid,ofid;
	FILE		*op,*lp;
	int32		it;
	char		*ty;
	char		*sptype="0";
	char		*antype="0";
	int			i,j,idx;
	int			mcnt=0;
	char		buff[256];
	char		obuff[2048];
	char		*p;
	int			ssamp,esamp;
	char		cmd[2048];

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:fpl:AJ")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Annotation alignment V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* item number */
			if (itspec(optarg,&it,&ty)==0) {
				if (it==SP_TYPE)
					sptype=ty;
				else if (it==AN_TYPE)
					antype=ty;
				else
					error("bad item specification",NULL);
			}
			else
				error("illegal item specification",NULL);
			break;
		case 'f' :	/* fixed label */
			dofixedlabel=1;
			dofixedpause=0;
			break;
		case 'p' :	/* fixed pause */
			dofixedlabel=0;
			dofixedpause=1;
			break;
		case 'l' :	/* pause label */
			strcpy(pauselabel,optarg);
			dofixedlabel=0;
			dofixedpause=1;
			break;
		case 'A' :	/* ARPABET output */
			doarpa=1;
			dojsru=0;
			break;
		case 'J' :	/* JSRU output */
			doarpa=0;
			dojsru=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	/* check command line */
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-f|-p) (-l pauselabel) (-A|-J) file",PROGNAME);

	/* get data filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open data file */
	if ((fid=sfsopen(filename,"w",NULL)) < 0)
		error("access error on '%s'",filename);

	/* find input annotation item */
	if (!sfsitem(fid,AN_TYPE,antype,&anitem))
		error("cannot find input AN item in '%s'",filename);

	/* get annotation buffers */
	an=(struct an_rec *)sfsbuffer(&anitem,anitem.numframes);

	/* load annotations */
	sfsread(fid,0,anitem.numframes,an);

	/* find input speech item */
	if (!sfsitem(fid,SP_TYPE,sptype,&spitem))
		error("cannot find input SP item in '%s'",filename);

	/* check sampling rate */
	if ((int)(0.5+0.01/spitem.frameduration) < 160)
		error("SP sampling rate must be at least 16000");

	/* get temporary directory name and a processid */
#ifdef _WIN32
	GetTempPath(256,fbuf);
	GetShortPathName(fbuf,tempname,sizeof(tempname));
	for (i=0;tempname[i];i++) if (tempname[i]=='\\') tempname[i]='/';
	processid = GetCurrentProcessId();
#else
	strcpy(tempname,"c:/tmp/");
	processid=0;
#endif

	/* open file for list of files */
	sprintf(lfilename,"%sfilelist%d.lst",tempname,processid);
	if ((lp=fopen(lfilename,"w"))==NULL)
		error("could not create '%s'",lfilename);

	/* save each region as separate WAV file and LAB file */
	if (dofixedlabel) {
		/* chunks same as input annotation boundaries */
		for (i=0;i<anitem.numframes;i++) {
			sprintf(wfilename,"%ssignal%d-%d.wav",tempname,processid,i);
			ssamp = (int)(((an[i].posn*anitem.frameduration+anitem.offset)-spitem.offset)/spitem.frameduration);
			esamp = (int)((((an[i].posn+an[i].size)*anitem.frameduration+anitem.offset)-spitem.offset)/spitem.frameduration);
			if (esamp > spitem.numframes) esamp=spitem.numframes;
			wavsave(wfilename,fid,ssamp,esamp);
			fprintf(lp,"%s\n",wfilename);

			sprintf(pfilename,"%ssignal%d-%d.lab",tempname,processid,i);
			labsave(pfilename,an[i].label);
		}
	}
	else if (dofixedpause) {
		/* chunks defined by pause labels */
		i=0;
		while (i<anitem.numframes) {
			j = i+1;
			if (strcmp(an[i].label,pauselabel)) {
				while ((j<anitem.numframes)&&strcmp(an[j].label,pauselabel)) j++;
			}

			sprintf(wfilename,"%ssignal%d-%d.wav",tempname,processid,i);
			ssamp = (int)(((an[i].posn*anitem.frameduration+anitem.offset)-spitem.offset)/spitem.frameduration);
			if (j==anitem.numframes)
				esamp = (int)((((an[j-1].posn+an[j-1].size)*anitem.frameduration+anitem.offset)-spitem.offset)/spitem.frameduration);
			else
				esamp = (int)(((an[j].posn*anitem.frameduration+anitem.offset)-spitem.offset)/spitem.frameduration);
			if (esamp > spitem.numframes) esamp=spitem.numframes;
			wavsave(wfilename,fid,ssamp,esamp);
			fprintf(lp,"%s\n",wfilename);

			sprintf(pfilename,"%ssignal%d-%d.lab",tempname,processid,i);
			labsavemult(pfilename,i,j);
			i=j;
		}
	}
	else {
		/* single chunk for whole file */
		sprintf(wfilename,"%ssignal%d-%d.wav",tempname,processid,0);
		ssamp = (int)(((an[0].posn*anitem.frameduration+anitem.offset)-spitem.offset)/spitem.frameduration);
		esamp = (int)((((an[anitem.numframes-1].posn+an[anitem.numframes-1].size)*anitem.frameduration+anitem.offset)-spitem.offset)/spitem.frameduration);
		if (esamp > spitem.numframes) esamp=spitem.numframes;
		wavsave(wfilename,fid,ssamp,esamp);
		fprintf(lp,"%s\n",wfilename);

		sprintf(pfilename,"%ssignal%d-%d.lab",tempname,processid,0);
		labsavemult(pfilename,0,anitem.numframes);
	}

	fclose(lp);

	/* launch HVite */
	strcpy(fbuf,argv[0]);
	p=strrchr(fbuf,'/');
	if (!p) p=strrchr(fbuf,'\\');
	if (p) {
		*p='\0';
		strcpy(pfilename,fbuf);
#ifdef WIN32
		GetShortPathName(fbuf,pfilename,sizeof(pfilename));
#endif
		strcat(pfilename,"/HViteAlign");
		for (i=0;pfilename[i];i++) if (pfilename[i]=='\\') pfilename[i]='/';
	}
	else
		strcpy(pfilename,"HViteAlign");

	strcpy(fbuf,sfsbase());
#ifdef WIN32
	GetShortPathName(fbuf,basename,sizeof(basename));
#else
	strcpy(basename,fbuf);
#endif
	for (i=0;basename[i];i++) if (basename[i]=='\\') basename[i]='/';


	sprintf(cmd,"%s -T 1 -C %s/data/analign.cfg -a -o SM -H %s/data/analign.hmm -S %s %s/data/analign.dct %s/data/analign.lst",
		pfilename,
		basename,
		basename,
		lfilename,
		basename,
		basename);


fprintf(stderr,"Executing: '%s'\n",cmd);

	i=system(cmd);
	if (i!=0) fprintf(stderr,"'%s' returns code %d\n",cmd,i);

	/* load in annotations */
	oan = (struct an_rec *)calloc(ocnt,sizeof(struct an_rec));
	opos=0;
	if (dofixedlabel) {
		for (i=0;i<anitem.numframes;i++) {
			sprintf(pfilename,"%ssignal%d-%d.rec",tempname,processid,i);
			ssamp = (int)((an[i].posn*anitem.frameduration+anitem.offset)*10000);
			esamp = (int)(((an[i].posn+an[i].size)*anitem.frameduration+anitem.offset)*10000);
			labload(pfilename,ssamp,esamp);
		}
	}
	else if (dofixedpause) {
		/* chunks defined by pause labels */
		i=0;
		while (i<anitem.numframes) {
			j = i+1;
			if (strcmp(an[i].label,pauselabel)) {
				while ((j<anitem.numframes)&&strcmp(an[j].label,pauselabel)) j++;
			}
			sprintf(pfilename,"%ssignal%d-%d.rec",tempname,processid,i);
			ssamp = (int)((an[i].posn*anitem.frameduration+anitem.offset)*10000);
			if (j==anitem.numframes)
				esamp = (int)(((an[j-1].posn+an[j-1].size)*anitem.frameduration+anitem.offset)*10000);
			else
				esamp = (int)((an[j].posn*anitem.frameduration+anitem.offset)*10000);
			labload(pfilename,ssamp,esamp);
			i=j;
		}
	}
	else {
		/* single chunk for whole file */
		sprintf(pfilename,"%ssignal%d-%d.rec",tempname,processid,0);
		ssamp = (int)((an[0].posn*anitem.frameduration+anitem.offset)*10000);
		esamp = (int)(((an[anitem.numframes-1].posn+an[anitem.numframes-1].size)*anitem.frameduration+anitem.offset)*10000);
		labload(pfilename,ssamp,esamp);
	}
	qsort(oan,ocnt,sizeof(struct an_rec),ancomp);

	/* delete files */
	if (dofixedlabel) {
		for (i=0;i<anitem.numframes;i++) {
			sprintf(wfilename,"%ssignal%d-%d.wav",tempname,processid,i);
			DeleteFile(wfilename);
			sprintf(pfilename,"%ssignal%d-%d.lab",tempname,processid,i);
			DeleteFile(pfilename);
			sprintf(pfilename,"%ssignal%d-%d.rec",tempname,processid,i);
			DeleteFile(pfilename);
		}
	}
	else if (dofixedpause) {
		/* chunks defined by pause labels */
		i=0;
		while (i<anitem.numframes) {
			j = i+1;
			if (strcmp(an[i].label,pauselabel)) {
				while ((j<anitem.numframes)&&strcmp(an[j].label,pauselabel)) j++;
			}
			sprintf(wfilename,"%ssignal%d-%d.wav",tempname,processid,i);
			DeleteFile(wfilename);
			sprintf(pfilename,"%ssignal%d-%d.lab",tempname,processid,i);
			DeleteFile(pfilename);
			sprintf(pfilename,"%ssignal%d-%d.rec",tempname,processid,i);
			DeleteFile(pfilename);
			i=j;
		}
	}
	else {
		sprintf(wfilename,"%ssignal%d-%d.wav",tempname,processid,0);
		DeleteFile(wfilename);
		sprintf(pfilename,"%ssignal%d-%d.lab",tempname,processid,0);
		DeleteFile(pfilename);
		sprintf(pfilename,"%ssignal%d-%d.rec",tempname,processid,0);
		DeleteFile(pfilename);
	}
	DeleteFile(lfilename);

	/* create output header */
	sfsheader(&opitem,AN_TYPE,-1,1,-1,0.0001,spitem.offset,0,0,1);
	if (dofixedlabel) {
		sprintf(opitem.history,"%s(%d.%02d,%d.%02d;type=phone,mode=fixedlabel%s%s)",
			PROGNAME,
			spitem.datatype,spitem.subtype,
			anitem.datatype,anitem.subtype,
			(doarpa)?",format=arpa":"",
			(dojsru)?",format=jsru":"");
	}
	else if (dofixedpause) {
		sprintf(opitem.history,"%s(%d.%02d,%d.%02d;type=phone,mode=fixedpause,pause=%s%s%s)",
			PROGNAME,
			spitem.datatype,spitem.subtype,
			anitem.datatype,anitem.subtype,
			pauselabel,
			(doarpa)?",format=arpa":"",
			(dojsru)?",format=jsru":"");
	}
	else {
		sprintf(opitem.history,"%s(%d.%02d,%d.%02d;type=phone%s%s)",
			PROGNAME,
			spitem.datatype,spitem.subtype,
			anitem.datatype,anitem.subtype,
			(doarpa)?",format=arpa":"",
			(dojsru)?",format=jsru":"");
	}

	/* save output */
	putitem(filename,&opitem,ocnt,oan);

	/* that's all folks */
	if (!sfsupdate(filename))
		error("update error on '%s'",filename);

	exit(0);
}
