/* dtmf - dual-tone multi frequency generate and detect */

/* M.A. Huckvale - University College London */

/* version 1.0 - September 2001 */

#define PROGNAME "dtmf"
#define PROGVERS "1.1"
char *progname=PROGNAME;

/*-------------------------------------------------------------------------*/
/**MAN
.TH DTMF SFS1 UCL
.SH NAME
dtmf - dual tone multi frequency generate and detect
.SH SYNOPSIS
.B dtmf
(-i item) (-g string) file
.SH DESCRIPTION
.I dtmf
is a program to generate or detect DTMF tones as used for
dialling on touch tone telephones.  The default action is
to scan a speech item for tones and to save the decoded
tones as annotations.  However the program can also generate
a tone sequence, given a telephone number as a command line
argument.
.SH OPTIONS
.TP 11
.B -I
Identify program name and version number.
.TP 11
.BI -i item
Select input speech item (detection)
.TP 11
.BI -g string
Specify string of tones to generate.
.SH INPUT ITEMS
.IP SP
Speech signal containing DTMF tones
.SH OUTPUT ITEMS
.IP SP
Generated DTMF tone sequence.
.IP AN
Detected DTMF tones
.SH VERSION/AUTHOR
.IP 1.0
Mark Huckvale
*/
/*--------------------------------------------------------------------------*/

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <ctype.h>
#include "sfs.h"

/* DTMF stuff */
/*
 *
 * goertzel aglorithm, find the power of different
 * frequencies in an N point DFT.
 *
 * ftone/fsample = k/N
 * k and N are integers.  fsample is 8000 (8khz)
 * this means the *maximum* frequency resolution
 * is fsample/N (each step in k corresponds to a
 * step of fsample/N hz in ftone)
 *
 * N was chosen to minimize the sum of the K errors for
 * all the tones detected...  here are the results :
 *
 * Best N is 240, with the sum of all errors = 3.030002
 * freq  freq actual   k     kactual  kerr
 * ---- ------------  ------ ------- -----
 *  350 (366.66667)   10.500 (11)    0.500
 *  440 (433.33333)   13.200 (13)    0.200
 *  480 (466.66667)   14.400 (14)    0.400
 *  620 (633.33333)   18.600 (19)    0.400
 *  697 (700.00000)   20.910 (21)    0.090
 *  700 (700.00000)   21.000 (21)    0.000
 *  770 (766.66667)   23.100 (23)    0.100
 *  852 (866.66667)   25.560 (26)    0.440
 *  900 (900.00000)   27.000 (27)    0.000
 *  941 (933.33333)   28.230 (28)    0.230
 * 1100 (1100.00000)  33.000 (33)    0.000
 * 1209 (1200.00000)  36.270 (36)    0.270
 * 1300 (1300.00000)  39.000 (39)    0.000
 * 1336 (1333.33333)  40.080 (40)    0.080
 **** I took out 1477.. too close to 1500
 * 1477 (1466.66667)  44.310 (44)    0.310
 ****
 * 1500 (1500.00000)  45.000 (45)    0.000
 * 1633 (1633.33333)  48.990 (49)    0.010
 * 1700 (1700.00000)  51.000 (51)    0.000
 * 2400 (2400.00000)  72.000 (72)    0.000
 * 2600 (2600.00000)  78.000 (78)    0.000
 *
 * notice, 697 and 700hz are indestinguishable (same K)
 * all other tones have a seperate k value.
 * these two tones must be treated as identical for our
 * analysis.
 *
 * The worst tones to detect are 350 (error = 0.5,
 * detet 367 hz) and 852 (error = 0.44, detect 867hz).
 * all others are very close.
 *
 */

int knos[] = { 11, 13, 14, 19, 21, 23, 26, 27, 28, 33, 36, 39, 40,
 /*44,*/ 45, 49, 51, 72, 78, };

/* coefficients for above k's as:
 *   2 * cos( 2*pi* k/N )
 */
double kcoeffs[] = {
1.917639, 1.885283, 1.867161, 1.757634,
1.705280, 1.648252, 1.554292, 1.520812, 1.486290,
1.298896, 1.175571, 1.044997, 1.000000, /* 0.813473,*/
0.765367, 0.568031, 0.466891, -0.618034, -0.907981,  };

#define X1    0    /* 350 dialtone */
#define X2    1    /* 440 ring, dialtone */
#define X3    2    /* 480 ring, busy */
#define X4    3    /* 620 busy */

#define R1    4    /* 697, dtmf row 1 */
#define R2    5    /* 770, dtmf row 2 */
#define R3    6    /* 852, dtmf row 3 */
#define R4    8    /* 941, dtmf row 4 */
#define C1   10    /* 1209, dtmf col 1 */
#define C2   12    /* 1336, dtmf col 2 */
#define C3   13    /* 1477, dtmf col 3 */
#define C4   14    /* 1633, dtmf col 4 */

#define B1    4    /* 700, blue box 1 */
#define B2    7    /* 900, bb 2 */
#define B3    9    /* 1100, bb 3 */
#define B4   11    /* 1300, bb4 */
#define B5   13    /* 1500, bb5 */
#define B6   15    /* 1700, bb6 */
#define B7   16    /* 2400, bb7 */
#define B8   17    /* 2600, bb8 */

#define NUMTONES 18

/* values returned by detect
 *  0-9     DTMF 0 through 9 or MF 0-9
 *  10-11   DTMF *, #
 *  12-15   DTMF A,B,C,D
 *  16-20   MF last column: C11, C12, KP1, KP2, ST
 *  21      2400
 *  22      2600
 *  23      2400 + 2600
 *  24      DIALTONE
 *  25      RING
 *  26      BUSY
 *  27      silence
 *  -1      invalid
 */
#define D0    0
#define D1    1
#define D2    2
#define D3    3
#define D4    4
#define D5    5
#define D6    6
#define D7    7
#define D8    8
#define D9    9
#define DSTAR 10
#define DPND  11
#define DA    12
#define DB    13
#define DC    14
#define DD    15
#define DC11  16
#define DC12  17
#define DKP1  18
#define DKP2  19
#define DST   20
#define D24   21
#define D26   22
#define D2426 23
#define DDT   24
#define DRING 25
#define DBUSY 26
#define DSIL  27

/* translation of above codes into text */
char *dtran[] = {
  "0", "1", "2", "3", "4", "5", "6", "7", "8", "9",
  "*", "#", "A", "B", "C", "D",
  "+C11 ", "+C12 ", " KP1+", " KP2+", "+ST ",
  " 2400 ", " 2600 ", " 2400+2600 ",
  " DIALTONE ", " RING ", " BUSY ","" };

#define RANGE  0.1           /* any thing higher than RANGE*peak is "on" */
#define THRESH 100.0         /* minimum level for the loudest tone */

/* manifest constants */
#define SRATE	8000		/* sample rate */
#define SILTIME	0.05		/* silence duration */
#define TONTIME	0.1			/* tone duration */
#define AMP		16384		/* tone amplitude */
#define WINSIZE	240			/* 240 samples = 1 block at 8000Hz */

/* global variables */
struct item_header spitem;
short		*sp;
struct item_header anitem;
struct an_rec 	*an;

/* operation */
int	dogen=0;			/* do generation */
char	tones[256];		/* tone sequence to generate */


/* generate silence */
void gensilence(int fid)
{
	short	samp[SRATE];
	int		i,n=(int)(SILTIME*SRATE);

	for (i=0;i<n;i++) samp[i]=0;
	sfswrite(fid,n,samp);
}

/* generate 2 tones */
void gen2tone(int fid,int tn)
{
  /* Freqs for 0-9, *, # */
	static int row[] = {  941,  697,  697,  697,  770,  770,  770,  852,  852,  852,  941,  941 };
	static int col[] = { 1336, 1209, 1336, 1477, 1209, 1336, 1477, 1209, 1336, 1477, 1209, 1477 };
	short	samp[SRATE];
	int		i,n=(int)(TONTIME*SRATE);
	double	omega1,omega2;

	omega1 = 8.0 * atan(1.0) * row[tn] / SRATE;
	omega2 = 8.0 * atan(1.0) * col[tn] / SRATE;

	for (i=0;i<n;i++)
		samp[i] = (short)(AMP*sin(i*omega1) + AMP*sin(i*omega2));

	sfswrite(fid,n,samp);
}


/* convert a block to new sample rate (by interpolation) */
void convert(short *src,int slen,double stime,double *dst,int dlen,double dtime)
{
	double	t,m1,m2;
	int		i,j1,j2;

	for (i=0;i<dlen;i++) {
		t = (i*dtime)/stime;
		j1 = (int)t;
		if (j1>=slen) j1 = slen-1;
		j2 = j1+1;
		if (j2>=slen) j2 = slen-1;
		m1 = 1.0 - (t-j1);
		m2 = 1.0 - m1;
		dst[i] = m1 * src[j1] + m2 * src[j2];
	}
}

/* calculate the power at the DTMF frequencies, using goertzel filter */
void calc_power(double *data,double *power)
{
	double u0[NUMTONES],u1[NUMTONES],t,in;
	int i,j;

	for(j=0; j<NUMTONES; j++) {
		u0[j] = 0.0;
		u1[j] = 0.0;
	}
 	for(i=0; i<WINSIZE; i++) {
		in = data[i] / 32768;

	    for(j=0; j<NUMTONES; j++) {
			t = u0[j];
			u0[j] = in + kcoeffs[j] * u0[j] - u1[j];
			u1[j] = t;
		}
	}
	for(j=0; j<NUMTONES; j++)
		 power[j] = u0[j] * u0[j] + u1[j] * u1[j] - kcoeffs[j] * u0[j] * u1[j];
}


/*
 * detect which signals are present.
 *
 * return values defined in the include file
 * note: DTMF 3 and MF 7 conflict.  To resolve
 * this the program only reports MF 7 between
 * a KP and an ST, otherwise DTMF 3 is returned
 */
int decode(double *data)
{
	double power[NUMTONES],thresh,maxpower;
	int on[NUMTONES],on_count;
	int bcount, rcount, ccount;
	int row, col, b1, b2, i;
	int r[4],c[4],b[8];
	static int MFmode=0;

	/* calculate the power at tone frequencies */
	calc_power(data,power);

	/* get a threshold value */
	for (i=0, maxpower=0.0; i<NUMTONES;i++)
		if(power[i] > maxpower)
			maxpower = power[i];
	if (maxpower < THRESH)  /* silence? */
		return(DSIL);
	thresh = RANGE * maxpower;    /* allowable range of powers */

	/* analyse tones */
	for (i=0, on_count=0; i<NUMTONES; i++) {
		if (power[i] > thresh) {
			on[i] = 1;
			on_count ++;
		}
		else
			on[i] = 0;
	}

	if (on_count == 1) {
		if (on[B7])
			return(D24);
	    if (on[B8])
			return(D26);
		return(-1);
	}

	if (on_count == 2) {
		if (on[X1] && on[X2])
			return(DDT);
		if (on[X2] && on[X3])
			return(DRING);
		if (on[X3] && on[X4])
			return(DBUSY);

	    b[0]= on[B1]; b[1]= on[B2]; b[2]= on[B3]; b[3]= on[B4];
	    b[4]= on[B5]; b[5]= on[B6]; b[6]= on[B7]; b[7]= on[B8];
	    c[0]= on[C1]; c[1]= on[C2]; c[2]= on[C3]; c[3]= on[C4];
	    r[0]= on[R1]; r[1]= on[R2]; r[2]= on[R3]; r[3]= on[R4];

		for (i=0, bcount=0; i<8; i++) {
			if (b[i]) {
				bcount++;
				b2 = b1;
				b1 = i;
			}
		}
		for (i=0, rcount=0; i<4; i++) {
			if (r[i]) {
				rcount++;
				row = i;
			}
		}
		for (i=0, ccount=0; i<4; i++) {
			if (c[i]) {
				ccount++;
		        col = i;
			}
		}

	    if (rcount==1 && ccount==1) {   /* DTMF */
			if (col == 3)  /* A,B,C,D */
				return(DA + row);
			else {
				if (row == 3 && col == 0)
					return(DSTAR);
		   		if (row == 3 && col == 2 )
					return(DPND);
		        if (row == 3)
					return(D0);
		        if (row == 0 && col == 2) {   /* DTMF 3 conflicts with MF 7 */
					if (!MFmode)
			            return(D3);
			   	}
			   	else
			          return(D1 + col + row*3);
      		}
	    }

	    if(bcount == 2) {       /* MF */
	      /* b1 has upper number, b2 has lower */
			switch(b1) {
				case 7: return( (b2==6)? D2426: -1);
		        case 6: return(-1);
		        case 5: if(b2==2 || b2==3)  /* KP */
			                  MFmode=1;
		                if(b2==4)  /* ST */
			                  MFmode=0;
		                return(DC11 + b2);
				        /* MF 7 conflicts with DTMF 3, but if we made it
				         * here then DTMF 3 was already tested for
         				 */
		        case 4: return( (b2==3)? D0: D7 + b2);
		        case 3: return(D4 + b2);
		        case 2: return(D2 + b2);
		        case 1: return(D1);
			}
		}
		return(-1);
	}

	if (on_count == 0)
		return(DSIL);

	return(-1);
}

/* main program */
void main(int argc,char *argv[])
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item selection */
	char	*ty;		/* item sub type */
	/* file variables */
	char	filename[SFSMAXFILENAME]; /* SFS data file name */
	int		fid;		/* input file descriptor */
	char	*sptype="0";
	int		ofid;
	int		i;
	double	*fp;
	int		t,last;
	int		wsize;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:g:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: DTMF tone generate/detect V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					sptype = ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'g' :	/* generate */
			dogen=1;
			strncpy(tones,optarg,256);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-g string) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* open file for output */
	if ((fid=sfsopen(filename,"w",NULL))<0)
		error("access error on '%s'",filename);

	/* generate or detect */
	if (dogen) {
		if (tones[0]=='\0')
			error("no telephone number given");

		/* create output item */
		sfsheader(&spitem,SP_TYPE,0,2,1,1.0/SRATE,0.0,1,0,0);
		sprintf(spitem.history,"%s(tones=%s)",PROGNAME,tones);

		if ((ofid=sfschannel(filename,&spitem))<0)
			error("could not open output channel on '%s'",filename);

		/* generate the tones */
		for (i=0;tones[i];i++) {
			gensilence(ofid);
			if (isdigit(tones[i]))
				gen2tone(ofid,tones[i]-'0');
			else if (tones[i]=='*')
				gen2tone(ofid,10);
			else if (tones[i]=='#')
				gen2tone(ofid,11);
		}
		gensilence(ofid);

	}
	else {
		/* find input speech item */
		if (!sfsitem(fid,SP_TYPE,sptype,&spitem))
			error("could not find input SP item in '%s'\n",filename);

		/* read signal in chunks */
		wsize = (int)(((double)WINSIZE / SRATE)/spitem.frameduration);
		sp = (short *)sfsbuffer(&spitem,wsize);
		fp = (double *)calloc(WINSIZE,sizeof(double));

		/* create annotation item */
		sfsheader(&anitem,AN_TYPE,-1,1,-1,spitem.frameduration,spitem.offset,0,0,1);
		sprintf(anitem.history,"%s(%d.%02d)",PROGNAME,spitem.datatype,spitem.subtype);
		ofid = sfschannel(filename,&anitem);
		an = (struct an_rec *)sfsbuffer(&anitem,1);

		/* read signal in overlapping frames */
		an->posn=0;
		an->size=0;
		strcpy(an->label,"/");
		last = DSIL;
		for (i=0;i<spitem.numframes-wsize/2;i+=wsize/2) {

			/* read block */
			sfsread(fid,i,wsize,sp);

			/* convert to 8000Hz */
			convert(sp,wsize,spitem.frameduration,fp,WINSIZE,1.0/SRATE);

			/* decode a frame */
			t = decode(fp);

			/* decode sequence */
		    if (t >= 0) {

				if ((t != DSIL) && (t != last) &&
			         (last == DSIL || last==D24 || last == D26 ||
			          last == D2426 || last == DDT || last == DBUSY ||
			          last == DRING) )  {
					if (i!=0) {
						an->size = i-an->posn;
						sfswrite(ofid,1,an);
						an->posn = i;
					}
					strcpy(an->label,dtran[t]);
					printf("%s",dtran[t]);
				}
				else if ((t==DSIL) && (last!=DSIL)) {
					an->size = i-an->posn;
					sfswrite(ofid,1,an);
					an->posn = i;
					strcpy(an->label,"/");
				}
				last = t;
			}
		}
		an->size = i-an->posn;
		sfswrite(ofid,1,an);
		printf("\n");
	}

	/* update */
	sfsclose(fid);
	if (!sfsupdate(filename))
		error("update error on '%s'",filename);

	/* that's all folks */
	exit(0);
}
