/* intx -- direct audio input if Voicescope pulses to tx item */

/* M.A.Huckvale - University College London */

/* version 1.0 - January 1999 */

/* version 2.0 - January 2000
	- support for ordinary ADC cards
*/

#define PROGNAME "intx"
#define PROGVERS "2.0"
char	*progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH INTX 1 UCL
.SH NAME
intx -- direct audio input of Voicescope pulses
.SH SYNOPSIS
.B intx
(-f sample-rate) file
.SH DESCRIPTION
.I intx
is a program to input Voicescope pulses through the analogue audio
input.  Special code is available for SB16, otherwise input is
done in 30 second blocks, with the possibility of a break in
recording every 30sec.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version number.
.TP 11
.BI -f sample-rate
Select input sample rate.  Default 48,000 samples/sec.
Any annotations item.
.SH OUTPUT ITEMS
.IP 3 11
Voicescope Tx.
.SH VERSION/AUTHOR
2.0 - Mark Huckvale
*/
/*--------------------------------------------------------------------------*/

/* standard definitions */
#include "SFSCONFG.h"
#include <stdio.h>		/* standard io library */
#include <stdlib.h>
#include <unistd.h>
#include "sfs.h"		/* database structures */

#define MAXTX 100000
#define MINFX 80
#define MAXFX 800

#ifdef DAC_DOS_BLAST16
/* routines for Protected Mode recording of 16-bit data */
int	SBT16Init(void);
int	SBT16Record(int *buff,int numf,int srate,int flags);
void	SBT16Close(void);
#endif

/* global data */
struct item_header txitem;	/* item header for tx data */
int	txbuff[MAXTX];		/* buffer for tx data */
short	*sbuf;
int	slen;

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */

	/* file variables */
	char		filename[SFSMAXFILENAME];	/* dbase file name */
	int		fid;
	double		srate=48000.0;
	char		*title=NULL;
	int		txcount;
	int		i,sum,cnt;
#ifndef DOS_DAC_BLAST16
	int		mintx,thresh,lastval,numf,count,nsamp;
#endif	
	/* decode switches */
	while ( (c = getopt(argc,argv,"If:t:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: direct audio input of Voicescope Tx V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'f' :	/* sample freq */
			srate = atof(optarg);
			break;
		case 't' :
			title = optarg;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-f sample-rate) (-t title) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* locate input an item */
	if ((fid=sfsopen(filename,"w",NULL)) < 0) {
		if (fid==-1)
			error("cannot find file '%s'",filename);
		else
			error("access error on '%s'",filename);
	}
	sfsclose(fid);

	/* create new item header */
	sfsheader(&txitem,TX_TYPE,0,4,1,1.0/srate,0.0,0,0,1);
	if (title)
		sprintf(txitem.history,"%s(rate=%g,title=%s)",PROGNAME,srate,title);
	else
		sprintf(txitem.history,"%s(rate=%g)",PROGNAME,srate);

#ifdef DAC_DOS_BLAST16
	if (SBT16Init()) {
		printf("Recording ... Press [Esc] to stop\n");

		txcount = SBT16Record(txbuff,MAXTX,(int)srate,1);

		printf("Recorded %d Tx values\n",txcount);

		SBT16Close();
	}
	else {
		fprintf(stderr,"could not open SB16 device\n");
		exit(1);
	}
#else
	/* get buffer to take 30 secs */
	slen = (int)(5 * srate);
	if ((sbuf=calloc(slen,sizeof(short)))==NULL)
		error("could not get memory buffer");

	/* get thresholds */
	mintx = (int)(srate/1000);
	thresh = 1000;

	/* do recording */
	if (adc_open(NULL)<0) {
		free(sbuf);
		error("could not open ADC device");
	}
	lastval = 0;
	txcount = 0;
	count = 0;
	numf = MAXTX;
	printf("Recording ... Press [Esc] to stop\n");
	while ((nsamp = adc_record(sbuf,slen,srate,1,0)) > 0) {
		for (i=0;i<nsamp;i++) {
			if ((sbuf[i]-lastval)>thresh) {
				if (count > mintx) {
					txbuff[txcount++] = count;
					count=0;
					if (--numf == 0) goto full;
				}
			}
			count++;
			lastval = sbuf[i];
		}
		printf("Recorded %d Tx values\r",txcount); fflush(stdout);
		if (nsamp < slen) break;
	}
	if ((count > 0)&&(numf > 0)) {
		txbuff[txcount++] = count;
	}
full:
	printf("\n");
	adc_close(0);
#endif

	/* report mean Fx */
	sum=0;
	cnt=0;
	for (i=0;i<txcount;i++)
		if (((srate/MAXFX)<=txbuff[i])&&(txbuff[i]<=(srate/MINFX))) {
			sum += txbuff[i];
			cnt++;
		}
	printf("Mean Fx=%.1fHz\n",srate*cnt/sum);

	/* update original file */
	putitem(filename,&txitem,txcount,txbuff);

	/* that's all folks ... */
	exit(0);
}

