/* lsprepro -- manipulate prosody of utterance using LSP analysis/synthesis */

/* M.A.Huckvale - University College London */

/* version 1.0 - February 2008 */

#undef IAG

#define PROGNAME "lsprepro"
#define PROGVERS "1.0"
char	*progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH LSPREPROS SFS1 UCL
.SH NAME
lsprepro - manipulate prosody of speech waveform using pitch-synchronous LSP analysis/synthesis
.SH SYNOPSIS
.B lsprepro
(-n ncoeffs) (-i item) -m control.txt file
.SH DESCRIPTION
.I lsprepro
performs a pitch-syncronous LPC analysis on sections of a speech waveform and
converts the predictor polynomial to line spectral pairs (frequencies).
The LSPs and residuals for each pitch period may then be manipulated to change
the duration, pitch or vocal tract length of any region of the input signal.
The specifications for the changes are supplied in a simple format control file.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and version number.
.TP 11
.BI -i item
Select input item number.
.TP 11
.BI -n ncoeff
Specify number of predictor coefficients to use.  Default is (2+2*sample_rate/1000).
.TP 11
.BI -m control.txt
Manipulation control file. Format is as follows: input time (s), timescaling factor, fx scaling factor,
VT length scaling factor.
All values are linearly interpolated across time. Lines starting with '#' are ignored. Here is an example.
.nf
# t  Dt   Dfx  Dvt
0    1    1    1.0
0.1  1.2  1.1  1.0
0.25 0.8  1.1  1.0
0.46 1    1    0.9
1    1    1    0.9
.fi
.SH INPUT ITEMS
.IP 1.xx 11
Any speech waveform
.IP 3.xx 11
Excitation markers.
.SH OUTPUT ITEMS
.IP 1 11
Modified speech signal.
.SH VERSION/AUTHOR
.nf
1.0 - Mark Huckvale
*/
/*--------------------------------------------------------------------------*/

/* standard definitions */

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "complex.h"
#include "filter.h"
#include "lsys.h"
#include "lsp.h"
#include "sfs.h"

#define MAXCOEFF	100	/* max # LPC coefficients */
#define	PREEMP		0.95	/* pre-emphasis coeff */
#define MAXWINDOW	22050	/* maximum analysis window size = 50ms @ 44.1kHz */

/* global data */
char		filename[SFSMAXFILENAME];	/* dbase file name */
struct item_header	spitem;		/* speech item header */
short		*sp;				/* speech buffer */
struct item_header	opitem;		/* output speech item header */
short		*op;				/* output speech buffer */
struct item_header	fxitem;
short		*fx;
struct item_header	txitem;		/* tx item header */
int			*tx;				/* tx data */
struct item_header	rspitem;	/* residual item header */
double		**rsp;				/* residual buffers for each frame */
struct item_header	pcitem;		/* LPC item header */
struct co_rec	*lpcrec;		/* buffer for LPC data */
int			ncoeff;				/* # LPC coefficients */

/* analysis parameters */
double		witime=0.020;
double		sttime=0.010;
int			wisize;
int			stsize;
double		lsp[MAXCOEFF];
double		pc[MAXCOEFF];

/* tx windows */
struct txwin_rec {
	int		spos;			/* start position */
	int		snum;			/* number of samples */
	int		hlen;			/* half length = time to pitch epoch */
	int		flag;			/* voicing */
} *txwin;
int txcnt;

/* control file parameters */
char		mfilename[SFSMAXFILENAME];
struct control_rec {
	double	itime;		/* input time */
	double	otscale;	/* output time scaling factor */
	double	ofscale;	/* output fx scaling factor*/
	double	ovscale;	/* output VT scaling factor */
	double	otime;		/* calculated output time */
} *ctab;
int	ccnt;

/* median smoothing for fx */
void mediansmooth()
{
	short	window[5];
	short	winsrt[5];
	int		i,j,k;
	int		t;

	window[0]=0;
	window[1]=0;
	for (i=0;i<fxitem.numframes;i++) {

		/* copy over 5 width window */
		for (j=0;j<=2;j++) {
			if ((i+j)>=fxitem.numframes)
				window[2+j]=0;
			else
				window[2+j]=fx[i+j];
		}

		/* sort samples */
		winsrt[0]=window[0];
		for (j=1;j<5;j++) {
			k=j;
			t=window[k];
			while ((k > 0)&&(t < winsrt[k-1])) {
				winsrt[k] = winsrt[k-1];
				k--;
			}
			winsrt[k]=t;
		}
		fx[i] = winsrt[2];

		window[0]=window[1];
		window[1]=window[2];
	}

}

/* join voiced regions with an interpolation */
void interpolate()
{
	double	sval,eval,gradient;
	int		start,stop;
	int		i,j;

	start=0;
	for (i=1;i<fxitem.numframes;i++) {
		if ((fx[i-1]==0)&&(fx[i]!=0)) {
			stop=i;
			sval = fx[start];
			if (sval==0) sval=fx[i];
			eval = fx[i];
			gradient = (eval-sval)/(stop-start);
			for (j=start;j<stop;j++)
				fx[j] = (short)(sval+gradient*(j-start));
		}
		else if ((fx[i-1]!=0)&&(fx[i]==0)) {
			start=i-1;
			sval=fx[start];
		}
	}
	i=fxitem.numframes-1;
	if (fx[i]==0) {
		for (j=start;j<=i;j++)
			fx[j] = (short)sval;
	}
}

/* make up an fx track from the tx track */
void makefx()
{
	double	t,t2;
	int		i,idx,numf;
	int	maxtx=(int)(witime/txitem.frameduration);
#ifdef IAG
	int	ofid;
#endif

	sfsheader(&fxitem,FX_TYPE,0,2,1,0.01,0.0,1,0,0);
	numf=fxitem.numframes = (int)(spitem.numframes*spitem.frameduration/fxitem.frameduration);
	fx = (short *)sfsbuffer(&fxitem,numf);
	memset(fx,0,numf*sizeof(short));

	t=0;
	for (i=0;i<txitem.numframes;i++) {
		t2 = t + tx[i]*txitem.frameduration;
		for (;t<t2;t+=fxitem.frameduration) {
			idx=(int)(t/fxitem.frameduration);
			if ((0<=idx)&&(idx<numf)) {
				if (tx[i]<maxtx)
					fx[idx]=(short)(1.0/(tx[i]*txitem.frameduration));
				else
					fx[idx]=0;
			}
		}
		t = t2;
	}

	/* save to file */
#ifdef IAG
	sprintf(fxitem.history,"%s/FX(%d.%02d,1)",PROGNAME,txitem.datatype,txitem.subtype);
	ofid=sfschannel(filename,&fxitem);
	sfswrite(ofid,numf,fx);
	sfsupdate(filename);
#endif

	mediansmooth();

	/* save to file */
#ifdef IAG
	sprintf(fxitem.history,"%s/FX(%d.%02d,2)",PROGNAME,txitem.datatype,txitem.subtype);
	ofid=sfschannel(filename,&fxitem);
	sfswrite(ofid,numf,fx);
	sfsupdate(filename);
#endif

	interpolate();

	/* save to file */
#ifdef IAG
	sprintf(fxitem.history,"%s/FX(%d.%02d,3)",PROGNAME,txitem.datatype,txitem.subtype);
	ofid=sfschannel(filename,&fxitem);
	sfswrite(ofid,numf,fx);
	sfsupdate(filename);
#endif
}

int getwinsize(int pos)
{
	int idx=(int)(pos*txitem.frameduration/fxitem.frameduration);
	if (idx<0)
		return(int)((1.0/fx[0])/txitem.frameduration);
	else if (idx<fxitem.numframes)
		return(int)((1.0/fx[idx])/txitem.frameduration);
	else
		return(int)((1.0/fx[fxitem.numframes-1])/txitem.frameduration);
}

int findwin(int spos,int epos,int *win)
{
	int	i,cnt=0;
	int	w;
	int pos;
	double	factor;

	pos=spos;
	while (pos < epos) {
		w = getwinsize(pos);
		pos += w;
		cnt++;
	}
	if ((epos-pos) > w/2) {
		/* lengthen everything */
		pos -= w;
		cnt--;
		factor = (double)(epos-spos)/(pos-spos);
	}
	else {
		/* shorten everything */
		factor = (double)(epos-spos)/(pos-spos);
	}

	if (win) {
		pos=spos;
		for (i=0;i<cnt;i++) {
			win[i]=(int)(getwinsize(pos)*factor);
			pos += win[i];
		}
		win[cnt-1] += epos-pos;

#ifdef IAG
printf("win=");
for (i=0;i<cnt;i++) printf("%d,",win[i]);
printf("\n");
#endif

	}

	return(cnt);

}

/* calculate analysis window positions */
void calctxwin()
{
	int	i,j;
	int	num,txpos;
	int	mintx=(int)(2*ncoeff*spitem.frameduration/txitem.frameduration);
	int	maxtx=(int)(witime/txitem.frameduration);
	double	spend=spitem.numframes*spitem.frameduration;
	double	txlast,txthis,txnext;
	int		win[1000];

#ifdef IAG
	struct item_header txitem2;
	int	ofid;
#endif

	/* count how many windows we've got */
	txcnt=0;
	txpos=0;
	for (i=0;i<txitem.numframes;i++) {
		if (tx[i] <= maxtx)
			txcnt++;
		else {
			txcnt+=findwin(txpos,txpos+tx[i],NULL);
		}
		txpos += tx[i];
	}
	txcnt += findwin(txpos,(int)(spend/txitem.frameduration),NULL);

printf("Predict txcnt=%d\n",txcnt);

	/* allocate some memory */
	if ((txwin=(struct txwin_rec *)calloc(2*txcnt,sizeof(struct txwin_rec)))==NULL)
		error("could not get memory for buffer");

	/* calculate the window positions */
	txcnt=0;
	txlast=0;
	for (i=0;(i<txitem.numframes)&&(txlast<spend-witime);i++) {
		txthis = txlast + tx[i]*txitem.frameduration;
		if (i < txitem.numframes-1)
			txnext = txthis+tx[i+1]*txitem.frameduration;
		else
			txnext = spend;
		if (txnext > txthis+witime/2) txnext=txthis+witime/2;
		if (txthis > spend) txthis=spend;
		if (txnext > spend) txnext=spend;
		if (txlast > txthis) break;

		if (tx[i] <= maxtx) {
			txwin[txcnt].spos = (int)(txlast/spitem.frameduration);
			txwin[txcnt].snum = (int)((txnext-txlast)/spitem.frameduration);
			txwin[txcnt].hlen = (int)((txthis-txlast)/spitem.frameduration);
			txwin[txcnt].flag = 1;
#ifdef IAG
printf("1 txlast=%8.4f txthis=%8.4f txnext=%8.4f spos=%d snum=%d hlen=%d stot=%d\n",
	txlast,txthis,txnext,txwin[txcnt].spos,txwin[txcnt].snum,txwin[txcnt].hlen,spitem.numframes);
#endif
			if ((txwin[txcnt].spos+txwin[txcnt].snum)>spitem.numframes)
				txwin[txcnt].snum = spitem.numframes-txwin[txcnt].spos;
			if (txwin[txcnt].snum > ncoeff) txcnt++;
		}
		else {
			num=findwin((int)(txlast/txitem.frameduration),(int)(txthis/txitem.frameduration),win);
			for (j=0;j<num-1;j++) {
				txwin[txcnt].spos = (int)(txlast/spitem.frameduration);
				txwin[txcnt].snum = (int)((win[j]+win[j+1])*txitem.frameduration/spitem.frameduration);
				txwin[txcnt].hlen = (int)(win[j]*txitem.frameduration/spitem.frameduration);
				txwin[txcnt].flag = 0;
#ifdef IAG
printf("2 txlast=%8.4f txthis=%8.4f txnext=%8.4f spos=%d snum=%d hlen=%d stot=%d\n",
	txlast,txthis,txnext,txwin[txcnt].spos,txwin[txcnt].snum,txwin[txcnt].hlen,spitem.numframes);
#endif
				if ((txwin[txcnt].spos+txwin[txcnt].snum)>spitem.numframes)
					txwin[txcnt].snum = spitem.numframes-txwin[txcnt].spos;
				if (txwin[txcnt].snum > ncoeff) txcnt++;
				txlast += win[j]*txitem.frameduration;
			}
			txwin[txcnt].spos = (int)(txlast/spitem.frameduration);
			txwin[txcnt].snum = (int)((txnext-txlast)/spitem.frameduration);
			txwin[txcnt].hlen = (int)((txthis-txlast)/spitem.frameduration);
			txwin[txcnt].flag = 0;
#ifdef IAG
printf("3 txlast=%8.4f txthis=%8.4f txnext=%8.4f spos=%d snum=%d hlen=%d stot=%d\n",
	txlast,txthis,txnext,txwin[txcnt].spos,txwin[txcnt].snum,txwin[txcnt].hlen,spitem.numframes);
#endif
			if ((txwin[txcnt].spos+txwin[txcnt].snum)>spitem.numframes)
				txwin[txcnt].snum = spitem.numframes-txwin[txcnt].spos;
			if (txwin[txcnt].snum > ncoeff) txcnt++;
		}
		txlast=txthis;
	}

	if (txlast < (spend-witime)) {
		txthis=txnext=spend;
		num=findwin((int)(txlast/txitem.frameduration),(int)(txthis/txitem.frameduration),win);
		for (j=0;j<num-1;j++) {
			txwin[txcnt].spos = (int)(txlast/spitem.frameduration);
			txwin[txcnt].snum = (int)((win[j]+win[j+1])*txitem.frameduration/spitem.frameduration);
			txwin[txcnt].hlen = (int)(win[j]*txitem.frameduration/spitem.frameduration);
			txwin[txcnt].flag = 0;
#ifdef IAG
printf("4 txlast=%g txthis=%g txnext=%g spos=%d snum=%d stot=%d\n",
	txlast,txthis,txnext,txwin[txcnt].spos,txwin[txcnt].snum,spitem.numframes);
#endif
			if ((txwin[txcnt].spos+txwin[txcnt].snum)>spitem.numframes)
				txwin[txcnt].snum = spitem.numframes-txwin[txcnt].spos;
			if (txwin[txcnt].snum > ncoeff) txcnt++;
			txlast += win[j]*txitem.frameduration;
		}
		txwin[txcnt].spos = (int)(txlast/spitem.frameduration);
		txwin[txcnt].snum = (int)((txnext-txlast)/spitem.frameduration);
		txwin[txcnt].hlen = (int)((txthis-txlast)/spitem.frameduration);
		txwin[txcnt].flag = 0;
#ifdef IAG
printf("5 txlast=%g txthis=%g txnext=%g spos=%d snum=%d stot=%d\n",
	txlast,txthis,txnext,txwin[txcnt].spos,txwin[txcnt].snum,spitem.numframes);
#endif
		if ((txwin[txcnt].spos+txwin[txcnt].snum)>spitem.numframes)
			txwin[txcnt].snum = spitem.numframes-txwin[txcnt].spos;
		if (txwin[txcnt].snum > ncoeff) txcnt++;
	}

	for (i=0;i<txcnt-1;i++) txwin[i].hlen=txwin[i+1].spos-txwin[i].spos;

printf("Actual txcnt=%d\n",txcnt);

	/* save to file */
#ifdef IAG
	sfsheader(&txitem2,TX_TYPE,0,4,1,spitem.frameduration,0.0,1,0,0);
	sprintf(txitem2.history,"%s/TX(%d.%02d)",PROGNAME,txitem.datatype,txitem.subtype);
	ofid=sfschannel(filename,&txitem2);
	for (i=1;i<txcnt;i++) {
		num=txwin[i].spos - txwin[i-1].spos;
		sfswrite(ofid,1,&num);
	}
#endif

}

/* create random noise at quantisation level */
double quantnoise()
{
	int v=rand()%10000;
	return ((v-5000)/10000.0);
}

/* load control file */
void loadcontrol(char *fname,double etime)
{
	FILE	*ip;
	char	line[256];
	char	*p;
	int		i;

	if ((ip=fopen(fname,"r"))==NULL)
		error("could not open '%s'",fname);

	/* count number of lines */
	ccnt=0;
	while (fgets(line,sizeof(line),ip)) if (line[0]!='#') ccnt++;

	/* allocate table */
	ctab = (struct control_rec *)calloc(ccnt+2,sizeof(struct control_rec));

	/* add initial guard value */
	ctab[0].itime=0;
	ctab[0].otscale=1.0;
	ctab[0].ofscale=1.0;
	ctab[0].ovscale=1.0;
	ctab[0].otime=0;

	/* load in values */
	rewind(ip);
	ccnt=1;
	while (fgets(line,sizeof(line),ip)) if (line[0]!='#') {

		/* reset to safe values */
		ctab[ccnt].itime=ctab[ccnt-1].itime;
		ctab[ccnt].otscale=1.0;
		ctab[ccnt].ofscale=1.0;
		ctab[ccnt].ovscale=1.0;

		p=strtok(line," \t\r\n");
		if (p) ctab[ccnt].itime = atof(p);
		p=strtok(NULL," \t\r\n");
		if (p) ctab[ccnt].otscale = atof(p);
		p=strtok(NULL," \t\r\n");
		if (p) ctab[ccnt].ofscale = atof(p);
		p=strtok(NULL," \t\r\n");
		if (p) ctab[ccnt].ovscale = atof(p);

		ctab[ccnt].otime = ctab[ccnt-1].otime + ctab[ccnt-1].otscale*(ctab[ccnt].itime-ctab[ccnt-1].itime);
		if (ctab[ccnt].itime > etime) break;
		ccnt++;
	}

	/* add guard value */
	ctab[ccnt].itime=etime;
	if (ctab[ccnt].otscale==0.0) ctab[ccnt].otscale=1.0;
	if (ctab[ccnt].ofscale==0.0) ctab[ccnt].ofscale=1.0;
	if (ctab[ccnt].ovscale==0.0) ctab[ccnt].ovscale=1.0;
	ctab[ccnt].otime = ctab[ccnt-1].otime + ctab[ccnt-1].otscale*(ctab[ccnt].itime-ctab[ccnt-1].itime);

printf("Control table from %s:\n",fname);
printf("   Input    TScale   FScale   VScale  Output\n");
for (i=0;i<=ccnt;i++)
	printf("%8.3f %8.3f %8.3f %8.3f %8.3f\n",
		ctab[i].itime,
		ctab[i].otscale,
		ctab[i].ofscale,
		ctab[i].ovscale,
		ctab[i].otime);

	fclose(ip);
}

/* get values from control table for given output time */
int getvalues(double otime,double *itime,double *ofscale,double *ovscale)
{
	int	i;
	double	m1,m2;

	for (i=0;i<ccnt;i++) {
		if ((ctab[i].otime <= otime) && (otime < ctab[i+1].otime)) {
			m2=(otime-ctab[i].otime)/(ctab[i+1].otime-ctab[i].otime);
			m1=1-m2;
			*itime = m1*ctab[i].itime + m2*ctab[i+1].itime;
			*ofscale = m1*ctab[i].ofscale + m2*ctab[i+1].ofscale;
			*ovscale = m1*ctab[i].ovscale + m2*ctab[i+1].ovscale;
			return(1);
		}
	}
	return(0);
}

/* scale the LSP frequencies */
void scalefreq(double *lsp,double fs2,double factor)
{
	int	i;
	double	omega,m1,m2;

	omega=(M_PI/2)/(ncoeff+1);
	for (i=0;i<ncoeff;i++) {
		m2=cos(i*omega);
		m1=1-m2;
		lsp[i] = m1*lsp[i] + m2*lsp[i]*factor;
	}

}

/* shift buffer by fraction of a sample */
void shiftbuff(double *buf1,double *buf2,int len, double frac)
{
	int	i;
	double	m1,m2;

	m1=frac;
	m2=1-m1;
	buf2[0]=m2*buf1[0];
	for (i=1;i<len;i++)
		buf2[i]=m1*buf1[i-1]+m2*buf1[i];

}

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument ptr */
	int		errflg = 0;	/* option error flag */
	int		c;		/* option switch */
	int		it;		/* item selections */
	char		*ty;
	char		*sptype="0";	/* default sub-type = last */
	char		*txtype="0";	/* default sub-type = last */
	int			dotx=0;
	/* file variables */
	int		fid,ofid;
	/* data variables */
	int		i,j,spos,scnt;
	double	omega,sum;
	double	*fsp,*frsp,*fwsp,*fsp2;
	double	itime,ofscale,ovscale,factor;
	int		ipos,oplen,opposfix;
	double	oppos;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:n:m:")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Manipulate prosody using LSP analysis/synthesis V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE)
					sptype = ty;
				else if (it == TX_TYPE) {
					txtype = ty;
					dotx=1;
				}
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'n' :	/* specify # coeffs */
			ncoeff = atoi(optarg);
			if (ncoeff < 2)
				error("too few poles: '%s'",optarg);
			if (ncoeff > MAXCOEFF) ncoeff=MAXCOEFF;
			break;
		case 'm' :	/* MBROLA control file */
			strcpy(mfilename,optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-n numcoeff) -m control.txt file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no database file specified",NULL);

	/* check file ok for writing */
	if ((fid=sfsopen(filename,"w",NULL)) < 0)
		error("access error on %s",filename);

	/* locate TX data */
	if (!sfsitem(fid,TX_TYPE,txtype,&txitem))
		error("unable to find input TX item in %s",filename);
	tx=sfsbuffer(&txitem,txitem.numframes);
	sfsread(fid,0,txitem.numframes,tx);

	/* locate speech data */
	if (!sfsitem(fid,SP_TYPE,sptype,&spitem))
		error("unable to find input speech item in %s",filename);

	/* load in the control file */
	loadcontrol(mfilename,spitem.numframes*spitem.frameduration);

	/* make the Fx contour */
	makefx();

malloc(5);

	/* pitch synchronous analysis */
	calctxwin();

malloc(10);

	/* create buffer for LPC coefficients */
	if (ncoeff==0) ncoeff = (int)(2 + 0.001/spitem.frameduration);
	if (ncoeff > MAXCOEFF) ncoeff=MAXCOEFF;
	wisize=(int)(0.5+witime/spitem.frameduration);
	stsize=(int)(0.5+sttime/spitem.frameduration);

	sfsheader(&pcitem,PC_TYPE,-1,4,sfsstruct[PC_TYPE]/4 + ncoeff,
			spitem.frameduration,spitem.offset,0,0,1);
	sprintf(pcitem.history,"%s/PC(%d.%02d,%d.%02d;win=%g;step=%g,ncoeff=%d)",
		PROGNAME,
		spitem.datatype,spitem.subtype,
		txitem.datatype,txitem.subtype,
		witime,sttime,ncoeff);
	sprintf(pcitem.params,"format=LSP");
	if ((lpcrec=(struct co_rec *)sfsbuffer(&pcitem,txcnt))==NULL)
		error("cannot get LPC buffer",NULL);

	/* get input signal */
	sp = (short *)sfsbuffer(&spitem,spitem.numframes);
	sfsread(fid,0,spitem.numframes,sp);

	/* pre-emp */
	for (j=spitem.numframes-1;j>0;j--) sp[j]=(short)(sp[j]-PREEMP*sp[j-1]);
	sp[0]=0;

	/* get data buffers */
	rsp = (double **)calloc(txcnt,sizeof(double *));
	fsp = (double *)calloc(2*wisize,sizeof(double));
	fsp2 = (double *)calloc(2*wisize,sizeof(double));
	frsp = (double *)calloc(2*wisize,sizeof(double));
	fwsp = (double *)calloc(2*wisize,sizeof(double));


	for (i=0;i<txcnt;i++) {
#ifdef IAG
printf("%d pos=%d num=%d\n",i,txwin[i].spos,txwin[i].snum);
#endif
		/* get speech */
		for (j=0;j<txwin[i].snum;j++) fsp[j]=(double)sp[txwin[i].spos+j];
		if (txwin[i].snum < wisize) {
			/* signal segment too small for good filter estimation */
			spos = txwin[i].spos + txwin[i].snum/2 - wisize/2;
			if (spos < 0) spos=0;
			scnt = wisize;
			if ((spos+scnt)>=spitem.numframes) scnt=spitem.numframes-spos;
			for (j=0;j<scnt;j++) fwsp[j]=(double)sp[spos+j] + quantnoise();
		}
		else {
			spos=txwin[i].spos;
			scnt=txwin[i].snum;
			for (j=0;j<txwin[i].snum;j++) fwsp[j]=(double)sp[txwin[i].spos+j] + quantnoise();
		}

		/* remove mean & window */
		sum=0;
		for (j=0;j<txwin[i].snum;j++) sum += fsp[j];
		sum /= txwin[i].snum;
		omega=(2*M_PI)/(txwin[i].snum-1);
		for (j=0;j<txwin[i].snum;j++)
			fsp[j] = (fsp[j]-sum)*(0.50-0.50*cos(omega*j));

		/* remove mean & window */
		sum=0;
		for (j=0;j<scnt;j++) sum += fsp[j];
		sum /= scnt;
		omega=(2*M_PI)/(scnt-1);
		for (j=0;j<scnt;j++)
			fwsp[j] = (fwsp[j]-sum)*(0.50-0.50*cos(omega*j));

		/* calculate RMS energy per sample */
		sum=0;
		for (j=0;j<txwin[i].snum;j++) sum += fsp[j]*fsp[j];
		sum = sqrt(sum/txwin[i].snum);

		/* perform LSP analysis on this window */
		lsp_auto_lpc(fwsp,scnt,pc,ncoeff);
		lsp_from_lpc(pc,lsp,ncoeff);
		lsp_residual_lpc(fsp,txwin[i].snum,pc,ncoeff,frsp);

		/* save */
		lpcrec[i].posn=txwin[i].spos;
		lpcrec[i].size=txwin[i].snum;
		lpcrec[i].flag=txwin[i].flag;
		lpcrec[i].mix=0;
		lpcrec[i].gain=(float)sum;
		for (j=0;j<ncoeff;j++)
			lpcrec[i].data[j]=(float)lsp[j];
		rsp[i] = (double *)calloc(txwin[i].snum,sizeof(double));
		for (j=0;j<txwin[i].snum;j++)
			rsp[i][j] = frsp[j];

		/* print progress */
		if (ttytest()) {
			printf("\rAnalysis %d/%d",i+1,txcnt);
			fflush(stdout);
		}
	}
	if (ttytest()) printf("\r                     \r");

	/* make output item */
	sfsheader(&opitem,SP_TYPE,0,2,1,spitem.frameduration,spitem.offset,1,0,0);
	sprintf(opitem.history,"%s/SP(%d.%02d,%d.%02d;control=%s,ncoeff=%d)",
		PROGNAME,
		spitem.datatype,spitem.subtype,
		txitem.datatype,txitem.subtype,
		mfilename,ncoeff);
	ofid = sfschannel(filename,&opitem);

	/* get output buffer */
	oplen = (int)(ctab[ccnt].otime/spitem.frameduration);
	op=(short *)sfsbuffer(&opitem,oplen);
	memset(op,0,oplen*sizeof(short));

//printf("spos=%d hlen=%d snum=%d\n",txwin[0].spos,txwin[0].hlen,txwin[0].snum);
//printf("spos=%d hlen=%d snum=%d\n",txwin[1].spos,txwin[1].hlen,txwin[1].snum);

	/* synthesis loop */
	oppos=0;
	while (getvalues(oppos*spitem.frameduration,&itime,&ofscale,&ovscale)) {

		/* get input frame */
		ipos = (int)(itime/spitem.frameduration);
		i=0;
		while ((i < txcnt-1) && (ipos >= txwin[i+1].spos)) i++;
//printf("ipos=%d i=%d spos=%d\n",ipos,i,txwin[i].spos);

		/* synthesis into buffer */
		for (j=0;j<ncoeff;j++) lsp[j]=lpcrec[i].data[j];
		scalefreq(lsp,0.5/spitem.frameduration,1.0/ovscale);
		lsp_synthesis(rsp[i],lpcrec[i].size,lsp,ncoeff,fsp);

		/* get scaling factor */
		sum=0;
		for (j=0;j<lpcrec[i].size;j++) sum += fsp[j]*fsp[j];
		sum = sqrt(sum/lpcrec[i].size);
		factor = lpcrec[i].gain/sum;

		/* overlap add */
		opposfix=(int)oppos;
		shiftbuff(fsp,fsp2,lpcrec[i].size,oppos-opposfix);
		for (j=0;(j<lpcrec[i].size)&&(opposfix+j<oplen);j++)
			op[opposfix+j] = (short)(op[opposfix+j]+fsp2[j]*factor);

		if (txwin[i].flag)
			oppos += txwin[i].hlen/ofscale;
		else
			oppos += txwin[i].hlen;

		/* print progress */
		if (ttytest()) {
			printf("\rSynthesis %d/%d",i+1,txcnt);
			fflush(stdout);
		}
	}
	if (ttytest()) printf("\r                     \r");

	/* de-emphasis */
	op[0]=0;
	for (j=1;j<oplen;j++) op[j]=(short)(op[j]+PREEMP*op[j-1]);

	/* write speech */
	sfswrite(ofid,oplen,op);

	/* update */
	if (!sfsupdate(filename) != 0)
		error("update error on %s",filename);

	/* that's all folks ... */
	exit(0);
}

