// graph.cpp -- implementation for graphics manager class
//
// (c) 1996 Mark Huckvale University College London

#include <stdio.h>
#include "SFSCONFG.h"
#include "sfs.h"
#include "dig.h"
#include "graph.h"

// static default xdata
static double graph_default_xdata[2]={0.0,1.0};

// constructor
Graph::Graph(int nrow,int ncol,const char *title,int dev)
{
	device = dev;
	state = noinit;
	nvert = nrow;
	nhoriz = ncol;
	xmode = DGdouble|DGinc;
	ymode = DGdouble;
	gtype = DGline;
	dobox = 0;
	plotchar = 0;
	if (title) {
		stitle = new char[strlen(title)+1];
		strcpy(stitle,title);
	}
	else {
		stitle = new char[1]; stitle[0]='\0';
	}
	gtitle = new char[1]; gtitle[0]='\0';
	xtitle = new char[1]; xtitle[0]='\0';
	ytitle = new char[1]; ytitle[0]='\0';
	xdata = graph_default_xdata;
	gused = new int[1+nhoriz*nvert];
	memset((char *)gused,0,(1+nhoriz*nvert)*sizeof(int));
}
Graph::~Graph()
{
	if (state!=noinit) close();
	delete plotchar;
	delete stitle;
	delete gtitle;
	delete xtitle;
	delete ytitle;
	delete gused;
}
// set values
void 	Graph::setsize(int nrow,int ncol)
{
	if ((state==dirty) && ((nhoriz!=ncol) || (nvert!=nrow))) clear();
	nhoriz = ncol;
	nvert = nrow;
	delete gused;
	gused = new int[1+nhoriz*nvert];
	memset((char *)gused,0,(1+nhoriz*nvert)*sizeof(int));
}
void	Graph::settitle(const char *title)
{
	delete stitle;
	stitle = new char[strlen(title)+1];
	strcpy(stitle,title);
}
void	Graph::settype(int type)
{
	gtype = type;
}
void	Graph::setchar(char ch)
{
	delete plotchar;
	plotchar = 0;
	if (ch) {
		plotchar = new char[2];
		plotchar[0] = ch;
		plotchar[1] = '\0';
	}
}
void	Graph::setbox(int flag)
{
	dobox = flag;
}
void	Graph::setxzero(int flag)
{
	xmode &= ~DGzero;
	if (flag) xmode |= DGzero;
}
void	Graph::setyzero(int flag)
{
	ymode &= ~DGzero;
	if (flag) ymode |= DGzero;
}
void	Graph::setxaxis(int pos)
{
	xmode &= ~(DGlowax|DGhighax);
	if (pos==GRAPH_BOTTOM)
		xmode |= DGlowax;
	else
		xmode |= DGhighax;
}
void	Graph::setyaxis(int pos)
{
	ymode &= ~(DGlowax|DGhighax);
	if (pos==GRAPH_LEFT)
		ymode |= DGlowax;
	else
		ymode |= DGhighax;
}
void	Graph::setxlabel(int pos)
{
	xmode &= ~(DGlowlab|DGhighlab);
	if (pos==GRAPH_BOTTOM)
		xmode |= DGlowlab;
	else
		xmode |= DGhighlab;
}
void	Graph::setylabel(int pos)
{
	ymode &= ~(DGlowlab|DGhighlab);
	if (pos==GRAPH_BOTTOM)
		ymode |= DGlowlab;
	else
		ymode |= DGhighlab;
}
void	Graph::setxscale(int flag)
{
	xmode |= DGnoscale;
	if (flag) xmode &= ~DGnoscale;
}
void	Graph::setyscale(int flag)
{
	ymode |= DGnoscale;
	if (flag) ymode &= ~DGnoscale;
}
void	Graph::setxlog(int flag)
{
	xmode &= ~DGlogax;
	if (flag) xmode |= DGlogax;
}
void	Graph::setylog(int flag)
{
	ymode &= ~DGlogax;
	if (flag) ymode |= DGlogax;
}
void	Graph::setequal(int flag)
{
	xmode &= ~DGequal;
	if (flag) xmode |= DGequal;
}
void	Graph::setgtitle(const char *title)
{
	delete gtitle;
	gtitle = new char[strlen(title)+1];
	strcpy(gtitle,title);
}
void	Graph::setxtitle(const char *title)
{
	delete xtitle;
	xtitle = new char[strlen(title)+1];
	strcpy(xtitle,title);
}
void	Graph::setytitle(const char *title)
{
	delete ytitle;
	ytitle = new char[strlen(title)+1];
	strcpy(ytitle,title);
}
void	Graph::setxdata(int flag,double *xarray)
{
	xmode &= ~(DGfirst|DGnext|DGlast);
	xmode |= flag;
	if (!xarray)
		xdata = graph_default_xdata;
	else
		xdata = xarray;
}

// force open/clear/close/wait-for-key
void	Graph::open(int dev)
{
	if (dev>=0) device=dev;
	digstart(device,NULL,1);
	digscale(1.0,1.2,0);
	digfont("");
	state = clean;
}
void	Graph::openraw(int dev)
{
	if (dev>=0) device=dev;
	digscale(1.0,1.22,0);
	digfont("");
	state = clean;
}
void	Graph::clear()
{
	if (state==noinit)
		open();
	else
		digclearscreen();
	state = clean;
	memset((char *)gused,0,(1+nhoriz*nvert)*sizeof(int));
}
void	Graph::close()
{
	if (state!=noinit) digquit(1);
	state = noinit;
}
void	Graph::wait()
{
	if (state!=noinit) digwait();
}

// basic graph display
void	Graph::plot(double *ydata,int ylen,int gno)
{
	int	gx,gy;

	if ((gno < 1) || (gno > (nhoriz*nvert))) return;

	if (state==noinit) open();

	if ((gno==1) && (state==clean)) {
		digscale(1.0,1.2,0);
		digorigin(0.0,0.0);
		digfont("triplro");
		digboxtext(3,0.5,1.13,0.7,0.06,0.0,stitle,1);
		digfont("");
	}

	digscale((float)nhoriz,1.2*(float)nvert,0);

	gx = (gno-1) % nhoriz;
	gy = (gno-1) / nhoriz;
	digorigin((float)gx,1.1*((float)nvert-gy-1));

	if (gused[gno])
		dgraph((short *)xdata,(short *)ydata,ylen,
			xmode|DGprev,ymode|DGprev,gtype,23242423+gused[gno],
			plotchar,xtitle,ytitle);
	else
		dgraph((short *)xdata,(short *)ydata,ylen,
			xmode,ymode,gtype,23242423,
			plotchar,xtitle,ytitle);
	if (gtitle[0])
		digboxtext(9,0.5,1.02,0.9,0.06,0.0,gtitle,1);
	if (dobox)
		digbox(24,0.0,0.0,1.0,1.1);
	digflush();

	gused[gno]++;

	state=dirty;
}

void	Graph::axes(double xmin,double xmax,double ymin,double ymax,int gno)
{
	int	gx,gy;
	double	xval[2];
	double	yval[2];

	if ((gno < 1) || (gno > (nhoriz*nvert))) return;

	if (state==noinit) open();

	if ((gno==1) && (state==clean)) {
		digscale(1.0,1.2,0);
		digorigin(0.0,0.0);
		digfont("triplro");
		digboxtext(3,0.5,1.13,0.7,0.06,0.0,stitle,1);
		digfont("");
	}

	digscale((float)nhoriz,1.2*(float)nvert,0);

	gx = (gno-1) % nhoriz;
	gy = (gno-1) / nhoriz;
	digorigin((float)gx,1.1*((float)nvert-gy-1));

	xval[0] = xmin;
	xval[1] = xmax;
	yval[0] = ymin;
	yval[1] = ymax;
	dgraph((short *)xval,(short *)yval,2,
		xmode,ymode,DGaxonly,23242423,
		plotchar,xtitle,ytitle);

	if (dobox)
		digbox(24,0.0,0.0,1.0,1.1);
	digflush();

	gused[gno]++;

	state = dirty;
}

#ifdef EMO
void main()
{
	double	y[10];

	y[0] = 10;	y[1] = 1;
	y[2] = 9;	y[3] = 2;
	y[4] = 8;	y[5] = 3;
	y[6] = 7;	y[7] = 4;
	y[8] = 6;	y[9] = 5;

	double x[10];

	x[0] = 1;	x[1] = 2;
	x[2] = 3;	x[3] = 4;
	x[4] = 5;	x[5] = 6;
	x[6] = 7;	x[7] = 8;
	x[8] = 9;	x[9] = 10;

	Graph g(2,3,"6 Example Graphs");

	g.setxtitle("X-axis");
	g.setytitle("Y-axis");
	g.setxdata(GRAPH_X_ALL,x);
	g.setyzero(GRAPH_ON);

	for (int i=1;i<=6;i++) {
		switch (i) {
		case 1:
			g.setgtitle("         Line           ");
			g.settype(GRAPH_LINE);
			break;
		case 2:
			g.setgtitle("         Scatter        ");
			g.settype(GRAPH_POINT);
			g.setchar('*');
			break;
		case 3:
			g.setgtitle("        Histogram       ");
			g.settype(GRAPH_HIST);
			g.setchar(0);
			break;
		case 4:
			g.setgtitle("   Histogram with end   ");
			g.settype(GRAPH_HISTEND);
			break;
		case 5:
			g.setgtitle("Histogram with divisions");
			g.settype(GRAPH_HISTDIV);
			break;
		case 6:
			g.setgtitle("           Bar          ");
			g.settype(GRAPH_BAR);
			break;
		}
		g.plot(y,4+i,i);
	}
	g.close();
}
#endif
