// lpcsolve.cpp - elementary solution of LPC equations
//
// C++ (c) 1996 Mark Huckvale University College London

#include "tools.h"
#include "ltisys.h"
#include "lpcsolve.h"

// smallest allowable value
const double EPSILON=1.0E-10;

// Solve set of simultaneous linear equations
//   by Gauss-Jordan elimination
double MatrixSolve(
	Equation *mat,		// matrix of equations
				//   n rows x (n+1) cols
	int n			// matrix size
	)			// returns determinant
{
	int	pos;		// pivot row
	double	max;		// maximum value in column
	double	det=1.0;	// determinant

	// Gauss-Jordan elimination
	for (int i=1;i<=n;i++) {	// for each row

		// find pivot row from max column entry
		max = 0;
		pos = 1;
		int j;
		for (j=i;j<=n;j++)
			if (fabs(mat[j][i]) > max) {
				max = fabs(mat[j][i]);
				pos = j;
			}

		// check for column of zeros
		if (max < EPSILON) return(0.0);

		// transpose current row with pivot row
		//   and normalise diagonal element to unity
		max = mat[pos][i];
		for (j=1;j<=n+1;j++) {
			double temp = mat[pos][j];
			mat[pos][j] = mat[i][j];
			mat[i][j] = temp/max;
		}

		// keep record of determinant
		if (i!=pos)
			det = det * -max;
		else
			det = det * max;

		// reduce matrix by dividing through by row
		for (int k=1;k<=n;k++) if (k!=i) {
			double val = mat[k][i];
			for (int j=i;j<=n+1;j++)
				mat[k][j] = mat[k][j] -
						val * mat[i][j];
		}
	}

	// return determinant
	return det;
}

// set up and solve LPC equations
int	LPCSolve(
	Waveform x,		// windowed input signal
	int order,		// predictor order required
	LTISystem& ltis		// returned predictor coefficients
	)			// returns 0=OK, 1=zero power, 2=fail
{
	// compute autocorrelations
	double	*r = new double[order+1];
	int i;
	for (i=0;i<=order;i++) {
		double sum = 0;
		for (int k=1;k<=x.count()-i;k++)
			sum += x[k] * x[k+i];
		r[i] = sum;
	}

	// build set of linear equations
	Equation *mat = new Equation[order+1];	// builds matrix of N rows x N+1 cols
	for (i=1;i<=order;i++)
		for (int j=1;j<=order;j++)
			mat[i][j] = r[abs(i-j)];
	for (i=1;i<=order;i++)
		mat[i][order+1] = -r[i];

	// free autocorrelation array
	delete [] r;

	// solve them
	if (MatrixSolve(mat,order)==0) {
		delete [] mat;
		return -1;		// fail
	}

	// copy coefficients into LTI System
	ltis = LTISystem(0,order);
	ltis.a[0] = 1.0;
	for (i=1;i<=order;i++)
		ltis.b[i] = mat[i][order+1];

	// OK
	delete [] mat;
	return 0;
}
