// sfsfunc.cpp -- SFS interface routines
//
// (c) 1996 Mark Huckvale University College London

// This module allows the loading and saving of waveforms
// from/to SFS files.  It also supports analogue recording
// and replay using SFS library routines.

#include "sfs.h"
#include <iostream>
#include <string.h>
#include "waveshor.h"
#include "sfsfunc.h"

using namespace std;

// load a waveform from an SFS file
int	SFSLoad(const char *fname,const char *spitem,WaveShort& wv,double stime,double sdur)
{
	int	fid;
	int	it;
	char	*ty="0";
	struct item_header item;
	char	*hold = new char[strlen(spitem)+1];
	strcpy(hold,spitem);

	if (itspec(hold,&it,&ty)!=0) {
		delete [] hold;
		return(-1);
	}
	if ((it!=SP_TYPE) && (it!=LX_TYPE)) {
		delete [] hold;
		return(-2);
	}
	if ((fid=sfsopen((char *)fname,"r",NULL)) < 0) {
		delete [] hold;
		return(-3);
	}
	if (!sfsitem(fid,it,ty,&item)) {
		sfsclose(fid);
		delete [] hold;
		return(-4);
	}
	int	numsamp;
	if (sdur < 0)
		numsamp = item.numframes;
	else
		numsamp = (int)(sdur / item.frameduration);
	int firstsamp = (int)(stime / item.frameduration);
	if (firstsamp < 0) firstsamp = 0;
	if (firstsamp > item.numframes) firstsamp = item.numframes;
	if ((firstsamp+numsamp) > item.numframes)
		numsamp = item.numframes - firstsamp;
	wv = WaveShort(numsamp,1.0/item.frameduration);
	if (sfsread(fid,firstsamp,numsamp,wv.buf)!=numsamp) {
		sfsclose(fid);
		delete [] hold;
		return(-5);
	}
	sfsclose(fid);
	delete [] hold;
	return(0);
}

// save a waveform back into an SFS file as speech item
int	SFSSave(const char *fname,const char *hist,const WaveShort& wv)
{
	int	ofid;
	struct item_header item;

	sfsheader(&item,SP_TYPE,0,2,1,wv.tau,0.0,1,0,0);
	strncpy(item.history,hist,sizeof(item.history)-1);

	if ((ofid=sfschannel((char *)fname,&item)) < 0)
		return(-6);
	if (sfswrite(ofid,wv.siz,wv.buf)!=wv.siz) {
		sfsclose(ofid);
		return(-7);
	}
	if (!sfsupdate((char *)fname))
		return(-8);
	return(0);
}

int SFSReplay(const WaveShort &wv)
{
	if (dac_open(NULL) >= 0) {
		dac_playback(wv.buf,wv.siz,wv.rate(),16,1,1);
		dac_close(0);
		return(0);
	}
	return(-1);
}
extern "C" {
double adc_selected_rate;
}
int SFSRecord(WaveShort &wv)
{
cout << "In SFSRecord\n";
	if (adc_open(NULL)>=0) {
cout << "Recording ... "; cout.flush();
		wv.siz = adc_record(wv.buf,wv.siz,wv.rate(),1,0);
		wv.tau = 1.0/adc_selected_rate;
		adc_close();
cout << "Done.\n";
		return(wv.siz);
	}
	return(-1);
}

#ifdef EMO
void main()
{
	WaveShort wv;

	cout << "SFSLoad(\"test.sfs\",\"sp.\",wv) returns " <<
		SFSLoad("test.sfs","sp.",wv) << "\n";

	cout << "Loaded " << wv.count() << " samples.\n";
	cout << "Sampling rate " << wv.rate() << " Hz.\n";
	for (int i=0;i<10;i++)
		cout << wv[i] << ',';
	cout << "\n";

	// reverse
	for (int i=0;i<wv.count()/2;i++) {
		int t = wv[i];
		wv[i] = wv[wv.count()-i-1];
		wv[wv.count()-i-1] = t;
	}
	cout << "SFSSave(\"test.sfs\",\"waveio()\",wv) returns " <<
		SFSSave("test.sfs","waveio()",wv) << "\n";

	// record
	cout << "Press RETURN to start recording 5s @ 10000Hz...";
	cout.flush();
	char ans[80];
	cin.getline(ans,80);

	WaveShort iwv(50000,10000);
	SFSRecord(iwv);

	cout << "Press RETURN to hear recording of " << iwv.count()/iwv.rate() << "s @" << iwv.rate() << "Hz\n";;
	cout.flush();
	cin.getline(ans,80);

	SFSReplay(iwv);
}
#endif
