// WaveDouble.cpp -- implementation for dynamic floating-point WaveShort class
//
// (c) 1996 Mark Huckvale University College London

#include <iostream>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include "wavedoub.h"
#include "waveshor.h"
#include "wavecomp.h"

// constructors
WaveDouble::WaveDouble(int length,double rate,int first)
{
	alloc = WAVE_ALLOC_CHUNK;
	while (alloc < length) alloc += WAVE_ALLOC_CHUNK;
	siz = length;
	lobound = first;
	tau = 1.0/rate;
	buf = new double[alloc];
	memset(buf,0,alloc*sizeof(double));
}
WaveDouble::WaveDouble(const WaveDouble& wv)
{
	alloc = wv.alloc;
	siz = wv.siz;
	lobound = wv.lobound;
	tau = wv.tau;
	buf = new double[alloc];
	memcpy(buf,wv.buf,alloc*sizeof(double));
}
WaveDouble::~WaveDouble()
{
	delete [] buf;
}

// virtual dump function
void WaveDouble::dump(int len,int first) const
{
	first -= lobound;
	if (first < 0) first=0;
	double	*ptr = buf+first;
	if ((len < 0) || (len > (siz-first))) len=(siz-first);
	while (len-- > 0)
		cout << *ptr++ << ' ';
	cout.flush();
}

// assignment operator
WaveDouble& WaveDouble::operator= (const WaveDouble& wv)
{
	if (&wv == this)
		return *this;		// self-assignment
	delete [] buf;
	alloc = wv.alloc;
	siz = wv.siz;
	lobound = wv.lobound;
	tau = wv.tau;
	buf = new double[alloc];
	memcpy(buf,wv.buf,alloc*sizeof(double));
	return *this;
}

// sample access
double& WaveDouble::operator[] (const int idx)
{
	static double junk=0;
	int nidx = idx - lobound;
	if ((nidx < 0) || (nidx >= siz)) {
		junk = 0;
		return junk;		// silently ignore range errors
	}
	else
		return buf[nidx];
}
const double& WaveDouble::operator[] (const int idx) const
{
	static double junk=0;
	int nidx = idx - lobound;
	if ((nidx < 0) || (nidx >= siz)) {
		junk = 0;
		return junk;		// silently ignore range errors
	}
	else
		return buf[nidx];
}

// concatenation
WaveDouble& WaveDouble::operator+= (const double val)
{
	if (siz >= alloc) {
		double * ptr = new double[alloc+WAVE_ALLOC_CHUNK];
		memcpy(ptr,buf,alloc*sizeof(double));
		alloc += WAVE_ALLOC_CHUNK;
		delete [] buf;
		buf = ptr;
	}
	buf[siz++] = val;
	return *this;
}
WaveDouble& WaveDouble::operator+= (const WaveDouble& wv)
{
	if (fabs(tau-wv.tau)/(tau+wv.tau) > 1.0E-6) {
		cerr << "WaveDouble: sample rate mismatch.\n";
		exit(1);
	}
	if ((siz+wv.siz) >= alloc) {
		while (alloc < (siz+wv.siz)) alloc += WAVE_ALLOC_CHUNK;
		double *ptr = new double[alloc];
		memcpy(ptr,buf,siz*sizeof(double));
		delete [] buf;
		buf = ptr;
	}
	memcpy(buf+siz,wv.buf,wv.siz*sizeof(double));
	siz += wv.siz;
	return *this;
}
#if defined(__GNUC__)
WaveDouble WaveDouble::operator+ (const WaveDouble& wv) return owv(siz+wv.siz,wv.rate(),lobound)
{
	if (fabs(tau-wv.tau)/(tau+wv.tau) > 1.0E-6) {
		cerr << "WaveDouble: sample rate mismatch.\n";
		exit(1);
	}
	memcpy(owv.buf,buf,siz*sizeof(double));
	memcpy(owv.buf+siz,wv.buf,wv.siz*sizeof(double));
	return owv;
}
#else
WaveDouble WaveDouble::operator+ (const WaveDouble& wv)
{
	if (fabs(tau-wv.tau)/(tau+wv.tau) > 1.0E-6) {
		cerr << "WaveDouble: sample rate mismatch.\n";
		exit(1);
	}
	WaveDouble owv(siz+wv.siz,wv.rate(),lobound);
	memcpy(owv.buf,buf,siz*sizeof(double));
	memcpy(owv.buf+siz,wv.buf,wv.siz*sizeof(double));
	return owv;	// unfortunately uses copy constructor
}
#endif

// cut function
WaveDouble WaveDouble::cut(int first,int len) const
{
	first -= lobound;
	if ((first < 0) || (first >= siz)) {
		first=0;
	}
	if (len < 0) len=0;
	if ((first+len) > siz) len = siz-first;
	WaveDouble owv(len,rate(),lobound);
	memcpy(owv.buf,buf+first,len*sizeof(double));
	return owv;
}

// conversion to other wave types
WaveShort WaveDouble::toWaveShort() const
{
	WaveShort owv(siz,rate(),lobound);
	for (int i=0;i<siz;i++) owv[i+lobound] = (short)buf[i];
	return owv;
}
WaveComplex WaveDouble::toWaveComplex() const
{
	WaveComplex owv(siz,rate(),lobound);
	for (int i=0;i<siz;i++) owv[i+lobound] = Complex(buf[i],0.0);
	return owv;
}


#ifdef EMO

void wvdump(Wave *wv)
{
	cout << "wv->alloc=" << wv->allocsize() << " ";
	cout << "wv->count=" << wv->count() << " ";
	cout << "wv->first=" << wv->first() << " ";
	cout << "wv->rate=" << wv->rate() << "\n";
	cout << "wv->buf="; wv->dump(); cout << "\n";
}

void main()
{
	WaveDouble wv(10,20000,1);
	int	i;

	for (i=1;i<=wv.count();i++) wv[i] = i/10.0;
	cout << "\nlist 0 to 9\n";
	wvdump(&wv);

	WaveDouble wvnew = wv;
	cout << "\nlist 0 to 9\n";
	wvdump(&wvnew);

	wvnew += 1.0;
	cout << "\nlist 0 to 10\n";
	wvdump(&wvnew);

	wvnew += wv;
	cout << "\nlist 0 to 10, then 0 to 9\n";
	wvdump(&wvnew);

	wvnew = wv;
	cout << "\nlist 0 to 9\n";
	wvdump(&wvnew);

	wvnew = wv + wv;
	cout << "\nlist 0 to 9, then 0 to 9\n";
	wvdump(&wvnew);

	wvnew = wvnew + wv;
	cout << "\nlist 0 to 9, then 0 to 9, then 0 to 9\n";
	wvdump(&wvnew);

	wvnew = wv.cut(3,4);
	cout << "\nlist 3 to 6\n";
	wvdump(&wvnew);

	wv += 1.0;
	WaveShort wvf = wv.toWaveShort();
	cout << "\nShort version\n";
	wvdump(&wvf);

	WaveComplex wvc = wv.toWaveComplex();
	cout << "\nComplex version\n";
	wvdump(&wvc);

	cout << "setting wv[0] to 10\n";
	wv[0] = 10;
	cout << "wv[0] contains " << wv[0] << "\n";
}
#endif

