// WaveShort.cpp -- implementation for dynamic WaveShort class
//
// (c) 1996 Mark Huckvale University College London

#include <iostream>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include "waveshor.h"
#include "wavedoub.h"
#include "wavecomp.h"
#include "graph.h"

// constructors
WaveShort::WaveShort(int length,double rate,int first)
{
	alloc = WAVE_ALLOC_CHUNK;
	while (alloc < length) alloc += WAVE_ALLOC_CHUNK;
	siz = length;
	lobound = first;
	tau = 1.0/rate;
	buf = new short[alloc];
	memset(buf,0,alloc*sizeof(short));
#ifdef IAG
cout << "WaveShort:constructor\n";
#endif
}
WaveShort::WaveShort(const WaveShort& wv)
{
	alloc = wv.alloc;
	siz = wv.siz;
	lobound = wv.lobound;
	tau = wv.tau;
	buf = new short[alloc];
	memcpy(buf,wv.buf,alloc*sizeof(short));
#ifdef IAG
cout << "WaveShort:copy\n";
#endif
}
WaveShort::~WaveShort()
{
	delete [] buf;
#ifdef IAG
cout << "WaveShort:destructor\n";
#endif
}

// virtual dump function
void WaveShort::dump(int len,int first) const
{
	first -= lobound;
	if (first < 0) first=0;
	short	*ptr = buf+first;
	if ((len < 0) || (len > (siz-first))) len=(siz-first);
	while (len-- > 0)
		cout << *ptr++ << ' ';
	cout.flush();
}

// assignment operator
WaveShort& WaveShort::operator= (const WaveShort& wv)
{
	if (&wv == this)
		return *this;		// self-assignment
	delete [] buf;
	alloc = wv.alloc;
	siz = wv.siz;
	lobound = wv.lobound;
	tau = wv.tau;
	buf = new short[alloc];
	memcpy(buf,wv.buf,alloc*sizeof(short));
	return *this;
}

// sample access
short& WaveShort::operator[] (const int idx)
{
	static short junk=0;
	int nidx = idx - lobound;
	if ((nidx < 0) || (nidx >= siz)) {
		junk = 0;
		return junk;		// silently ignore range errors
	}
	else
		return buf[nidx];
}
const short& WaveShort::operator[] (const int idx) const
{
	static short junk=0;
	int nidx = idx - lobound;
	if ((nidx < 0) || (nidx >= siz)) {
		junk = 0;
		return junk;		// silently ignore range errors
	}
	else
		return buf[nidx];
}

// concatenation
WaveShort& WaveShort::operator+= (const short val)
{
	if (siz >= alloc) {
		short * ptr = new short[alloc+WAVE_ALLOC_CHUNK];
		memcpy(ptr,buf,alloc*sizeof(short));
		alloc += WAVE_ALLOC_CHUNK;
		delete [] buf;
		buf = ptr;
	}
	buf[siz++] = val;
	return *this;
}
WaveShort& WaveShort::operator+= (const WaveShort& wv)
{
	if (fabs(tau-wv.tau)/(tau+wv.tau) > 1.0E-6) {
		cerr << "WaveShort: sample rate mismatch.\n";
		exit(1);
	}
	if ((siz+wv.siz) >= alloc) {
		while (alloc < (siz+wv.siz)) alloc += WAVE_ALLOC_CHUNK;
		short *ptr = new short[alloc];
		memcpy(ptr,buf,siz*sizeof(short));
		delete [] buf;
		buf = ptr;
	}
	memcpy(buf+siz,wv.buf,wv.siz*sizeof(short));
	siz += wv.siz;
	return *this;
}
#ifdef __GNUC__
WaveShort WaveShort::operator+ (const WaveShort& wv) return owv(siz+wv.siz,wv.rate(),lobound)
{
	if (fabs(tau-wv.tau)/(tau+wv.tau) > 1.0E-6) {
		cerr << "WaveShort: sample rate mismatch.\n";
		exit(1);
	}
	memcpy(owv.buf,buf,siz*sizeof(short));
	memcpy(owv.buf+siz,wv.buf,wv.siz*sizeof(short));
	return owv;
}
#else
WaveShort WaveShort::operator+ (const WaveShort& wv)
{
	if (fabs(tau-wv.tau)/(tau+wv.tau) > 1.0E-6) {
		cerr << "WaveShort: sample rate mismatch.\n";
		exit(1);
	}
	WaveShort owv(siz+wv.siz,wv.rate(),lobound);
	memcpy(owv.buf,buf,siz*sizeof(short));
	memcpy(owv.buf+siz,wv.buf,wv.siz*sizeof(short));
	return owv;	// unfortunately uses copy constructor
}
#endif

// cut function
WaveShort WaveShort::cut(int first,int len) const
{
	first -= lobound;
	if ((first < 0) || (first >= siz)) {
		first=0;
	}
	if (len < 0) len=0;
	if ((first+len) > siz) len = siz-first;
	WaveShort owv(len,rate(),lobound);
	memcpy(owv.buf,buf+first,len*sizeof(short));
	return owv;
}

// conversion to other wave types
WaveDouble WaveShort::toWaveDouble() const
{
	WaveDouble owv(siz,rate(),lobound);
	for (int i=0;i<siz;i++) owv[i+lobound] = (double)buf[i];
	return owv;
}
WaveComplex WaveShort::toWaveComplex() const
{
	WaveComplex owv(siz,rate(),lobound);
	for (int i=0;i<siz;i++) owv[i+lobound] = Complex((double)buf[i],0.0);
	return owv;
}

#ifdef EMO

void wvdump(Wave *wv)
{
	cout << "wv->alloc=" << wv->allocsize() << " ";
	cout << "wv->count=" << wv->count() << " ";
	cout << "wv->rate=" << wv->rate() << "\n";
	cout << "wv->buf="; wv->dump(); cout << "\n";
}

void main()
{
	WaveShort wv(10);
	int	i;

	for (i=0;i<wv.count();i++) wv[i] = i;
	cout << "\nlist 0 to 9\n";
	wvdump(&wv);

	WaveShort wvnew = wv;
	cout << "\nlist 0 to 9\n";
	wvdump(&wvnew);

	wvnew += 10;
	cout << "\nlist 0 to 10\n";
	wvdump(&wvnew);

	wvnew += wv;
	cout << "\nlist 0 to 10, then 0 to 9\n";
	wvdump(&wvnew);

	wvnew = wv;
	cout << "\nlist 0 to 9\n";
	wvdump(&wvnew);

	wvnew = wv + wv;
	cout << "\nlist 0 to 9, then 0 to 9\n";
	wvdump(&wvnew);

	wvnew = wvnew + wv;
	cout << "\nlist 0 to 9, then 0 to 9, then 0 to 9\n";
	wvdump(&wvnew);

	wvnew = wv.cut(3,4);
	cout << "\nlist 3 to 6\n";
	wvdump(&wvnew);

	WaveDouble wvr = wv.toWaveDouble();
	cout << "\nReal version\n";
	wvdump(&wvr);

	WaveComplex wvc = wv.toWaveComplex();
	cout << "\nComplex version\n";
	wvdump(&wvc);

}
#endif

