/* spclex -- lexical analyser and symbol table manipulation for SPC */

/* Mark Huckvale - University College London - April 1991 */

/* external definitions */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <math.h>
#include <malloc.h>
#include "spc.h"

/* symbol table */
Symbol 	*symtab;

/* reserved words */
static struct {
	char	*name;
	short	tok;
} reserved[] = {			/* KEEP SORTED */
{	"abs",		ABS_TOK },
{	"and",		AND_TOK },
{	"array",	ARRAY_TOK },
{	"begin",	BEGIN_TOK },
{	"case",		CASE_TOK },
{	"chr",		CHR_TOK },
{	"close",	CLOSE_TOK },
{	"const",	CONST_TOK },
{	"div",		DIV_TOK },
{	"do",		DO_TOK },
{	"downto",	DOWNTO_TOK },
{	"else",		ELSE_TOK },
{	"end",		END_TOK },
{	"external",	EXTERNAL_TOK },
{	"file",		FILE_TOK },
{	"for",		FOR_TOK },
{	"function",	FUNCTION_TOK },
{	"goto",		GOTO_TOK },
{	"if",		IF_TOK },
{	"in",		IN_TOK },
{	"label",	LABEL_TOK },
{	"mod",		MOD_TOK },
{	"new",		NEW_TOK },
{	"not",		NOT_TOK },
{	"nil",		NIL_TOK },
{	"of",		OF_TOK },
{	"or",		OR_TOK },
{	"ord",		ORD_TOK },
{	"procedure",	PROCEDURE_TOK },
{	"program",	PROGRAM_TOK },
{	"read",		READ_TOK },
{	"readln",	READLN_TOK },
{	"record",	RECORD_TOK },
{	"repeat",	REPEAT_TOK },
{	"reset",	RESET_TOK },
{	"rewrite",	REWRITE_TOK },
{	"set",		SET_TOK },
{	"sqr",		SQR_TOK },
{	"str",		STR_TOK },
{	"then",		THEN_TOK },
{	"to",		TO_TOK },
{	"type",		TYPE_TOK },
{	"until",	UNTIL_TOK },
{	"var",		VAR_TOK },
{	"while",	WHILE_TOK },
{	"with",		WITH_TOK },
{	"write",	WRITE_TOK },
{	"writeln",	WRITELN_TOK },
{	0,		0 },		/* KEEP SORTED */
};

/* predeclared types */
static struct {
	char	*name;
	char	*cname;
	short	etype;
	short	btype;
} types[] = {
{	"integer", 	"int32",			(short)EXPINT,	0 },
{	"short", 	"short",		(short)EXPINT,	0 },
{	"byte", 	"unsigned char",	(short)EXPINT,	0 },
{	"real", 	"double",		(short)EXPREAL,	0 },
{	"char",		"char",			(short)EXPCHAR,	0 },
{	"boolean",	"unsigned char",	(short)EXPBOOL,	0 },
{	"string",	"string",		(short)EXPSTR,	(short)EXPCHAR },
{	"text",		"filedesc",		(short)EXPFILE,	0 },
{	"waveform",	"waveform",		(short)EXPWAVE,	(short)EXPINT },
{	0,		0 },
};

/* predeclared field names */
static struct {
	char	*name;
	char	*cname;
	short	etype;
	short	btype;
} fields[] = {
{	"_size",	"len",		(short)EXPINT,		(short)0 },
{	"_rate",	"smr",		(short)EXPREAL,		(short)0 },
{	"_char",	"buf",		(short)EXPADDR,		(short)EXPCHAR },
{	"_data",	"buf",		(short)EXPADDR,		(short)EXPINT },
{	0,		0,		0,			0 },
};

/* predeclared files */
static struct {
	char	*name;
	char	*cname;
} files[] = {
{	"output", 	"stdout" },
{	"error", 	"stderr" },
{	"input",	"stdin" },
{	0,		0 },
};

/* built-in functions */
static struct {
	char	*name;
	char	*cname;
	short	etype;
} functions[] = {
{	"arctan",	"_ATAN",	(short)EXPREAL },
{	"cos",		"_COS",		(short)EXPREAL },
{	"eof",		"_EOF",		(short)EXPBOOL },
{	"eoln",		"_EOLN",	(short)EXPBOOL },
{	"exp",		"_EXP",		(short)EXPREAL },
{	"ioerror",	"_IOERROR",	(short)EXPINT },
{	"ln",		"_LOG",		(short)EXPREAL },
{	"odd",		"_ODD",		(short)EXPBOOL },
{	"pred",		"_PRED",	(short)EXPINT },
{	"round",	"_ROUND",	(short)EXPINT },
{	"sin",		"_SIN",		(short)EXPREAL },
{	"sqrt",		"_SQRT",	(short)EXPREAL },
{	"succ",		"_SUCC",	(short)EXPINT },
{	"trunc",	"_TRUNC",	(short)EXPINT },
{	0,		0 },
};

/* built-in procedures */
static struct {
	char	*name;
	char	*cname;
} procedures[] = {
{	"ioerrortrap",	"_ioerrortrap" },
{	"dispose",	"_dispose" },
{	0,		0 },
};

/* built-in constants */
static struct {
	char	*name;
	char	*cname;
	short	etype;
} constants[] = {
{	"true", 	"!0",		(short)EXPBOOL },
{	"false", 	"0",		(short)EXPBOOL },
{	"maxint",	"0x7FFFFFFFL",	(short)EXPINT },
{	0,		0 },
};

/* save a string in dynamic memory */
char *strsave(str)
char	*str;
{
	char *p;
	if (!str) return(NULL);
	if ((p=malloc(strlen(str)+1))==NULL)
		spcerror("out of memory",NULL);
	strcpy(p,str);
	return(p);
}

/* convert to lower case */
void strlowercase(s)
char	*s;
{
	while (*s) {
		if (isupper(*s)) *s = tolower(*s);
		s++;
	}
}

/* install a symbol into symbol table */
Symbol	*installsymbol(name,type,status)
char	*name;
short	type;
short	status;
{
	Symbol	*sym;

	if ((sym=(Symbol *)malloc(sizeof(Symbol)))==NULL)
		spcerror("out of memory",NULL);
	memset((char *)sym,0,sizeof(Symbol));

	sym->name = strsave(name);
	sym->cname = NULL;
	sym->type = type;
	sym->status = status;
	sym->etype = 0;
	sym->btype = 0;
	sym->ptype = NULL;
	sym->next = symtab;

	symtab = sym;
	return(sym);
}

/* free symbol */
void freesymbol(sym)
Symbol *sym;
{
	if (sym->cname != sym->name) free(sym->cname);
	if (sym->name) free(sym->name);
	if (sym->ptype) free(sym->ptype);
	free(sym);
}

/* duplicate symbol */
Symbol *dupsymbol(osym)
Symbol *osym;
{
	Symbol	*sym;

	if ((sym=(Symbol *)malloc(sizeof(Symbol)))==NULL)
		spcerror("out of memory",NULL);

	sym->name = strsave(osym->name);
	sym->cname = strsave(osym->cname);
	sym->type = LOCAL;
	sym->status = UNDEF_VAR;
	sym->etype = 0;
	sym->btype = 0;
	sym->ptype = NULL;
	sym->next = symtab;

	symtab = sym;
	return(sym);
}

/* start symbol table block */
void symblockstart()
{
	Symbol	*sym;

	if ((sym=(Symbol *)malloc(sizeof(Symbol)))==NULL)
		spcerror("out of memory",NULL);

	sym->name = NULL;
	sym->cname = NULL;
	sym->type = 0;
	sym->status = 0;
	sym->etype = 0;
	sym->btype = 0;
	sym->ptype = NULL;
	sym->next = symtab;
	symtab = sym;
}

/* initialise symbol block */
void symblockinit()
{
	Symbol *sym;

	outputstr("/* init local symbols */\n");
	sym=symtab;
	while (sym && sym->name) {
		if ((sym->etype==(int)EXPSTR) &&
		    ((sym->status!=VALPAR) && (sym->status!=ADRPAR)
			&& (sym->status != CONSTANT)) ) {
			outputstr(sym->cname);
			outputstr("=stringNULL;\n");
		}
		else if ((sym->etype==(int)EXPWAVE) &&
		    ((sym->status!=VALPAR) && (sym->status!=ADRPAR)
			&& (sym->status != CONSTANT)) ) {
			outputstr(sym->cname);
			outputstr("=waveformNULL;\n");
		}
		sym = sym->next;
	}
}

/* pop symbol table block */
void symblockend()
{
	Symbol *sym,*nsym;

	sym=symtab;
	while (sym && sym->name) {
		if ((sym->etype==(int)EXPSTR) &&
		    ((sym->status!=ADRPAR) &&
			(sym->status != CONSTANT))) {
			outputstr("_sfree(");
			outputstr(sym->cname);
			outputstr(");\n");
		}
		else if ((sym->etype==(int)EXPWAVE) &&
		    ((sym->status!=ADRPAR) &&
			(sym->status != CONSTANT))) {
			outputstr("_wfree(");
			outputstr(sym->cname);
			outputstr(");\n");
		}
		nsym = sym->next;
		freesymbol(sym);
		sym = nsym;
	}
	if (sym) {
		symtab = sym->next;
		freesymbol(sym);
	}
}

/* pop symbol table block */
void symblockcancel()
{
	Symbol *sym,*nsym;

	sym=symtab;
	while (sym && sym->name) {
		nsym = sym->next;
		freesymbol(sym);
		sym = nsym;
	}
	if (sym) {
		symtab = sym->next;
		freesymbol(sym);
	}
}

/* check symbol table block */
int symblockcheck(sym1,sym2)
Symbol *sym1,*sym2;
{

	while (sym1 && (sym1!=sym2)) {
		if (sym1->name)
			sym1 = sym1->next;
		else
			return(1);
	}
	return(0);
}

/* initialise symbol table */
void initsymbol()
{
	int	i;
	Symbol	*s,*ns;

	/* clear any existing contents of table */
	s = symtab;
	while (s) {
		ns = s->next;
		freesymbol(s);
		s = ns;
	}
	symtab = NULL;

	/* install reserved words */
	for (i=0;reserved[i].name;i++)
		s = installsymbol(reserved[i].name,reserved[i].tok,RESERVED);

	/* install predeclared types */
	for (i=0;types[i].name;i++) {
		s = installsymbol(types[i].name,TYPE_VAR,BLTIN);
		s->etype = types[i].etype;
		s->btype = types[i].btype;
		if (strcmp(types[i].name,types[i].cname))
			s->cname = strsave(types[i].cname);
		else
			s->cname = s->name;
	}

	/* install predeclared field names */
	for (i=0;fields[i].name;i++) {
		s = installsymbol(fields[i].name,FIELD_VAR,BLTIN);
		s->etype = fields[i].etype;
		s->btype = fields[i].btype;
		if (strcmp(fields[i].name,fields[i].cname))
			s->cname = strsave(fields[i].cname);
		else
			s->cname = s->name;
	}

	/* install predeclared files */
	for (i=0;files[i].name;i++) {
		s = installsymbol(files[i].name,FILE_VAR,BLTIN);
		if (strcmp(files[i].name,files[i].cname))
			s->cname = strsave(files[i].cname);
		else
			s->cname = s->name;
	}

	/* install built-in functions and procedures */
	for (i=0;functions[i].name;i++) {
		s = installsymbol(functions[i].name,FUNCTION_VAR,BLTIN);
		s->etype = functions[i].etype;
		if (strcmp(functions[i].name,functions[i].cname))
			s->cname = strsave(functions[i].cname);
		else
			s->cname = s->name;
	}
	for (i=0;procedures[i].name;i++) {
		s = installsymbol(procedures[i].name,PROCEDURE_VAR,BLTIN);
		if (strcmp(procedures[i].name,procedures[i].cname))
			s->cname = strsave(procedures[i].cname);
		else
			s->cname = s->name;
	}

	/* install built-in constants */
	for (i=0;constants[i].name;i++) {
		s = installsymbol(constants[i].name,CONST_VAR,BLTIN);
		s->etype = constants[i].etype;
		if (strcmp(constants[i].name,constants[i].cname))
			s->cname = strsave(constants[i].cname);
		else
			s->cname = s->name;
	}
}

int stringcomp(s,t)
char	*s;
char	*t;
{
	int	c=0;

	while ((*t || *s) && ((c=(toupper(*t)-toupper(*s)))==0)) {
		s++;
		t++;
	}
	return(c);
}

/* symbol table lookup */
Symbol	*lookupsymbol(name)
char 	*name;	/* symbol */
{
	Symbol	*sym;

	sym = symtab;

	while (sym) {
		if (sym->name && (stringcomp(name,sym->name)==0)) {
			return(sym);
		}
		sym = sym->next;
	}
	return(NULL);
}

/* input file stack */
#define FSTACKSIZE	10
struct {
	char	fname[80];
	FILE	*fp;
	int	lno;
	int	col;
} fstack[FSTACKSIZE];
int		fsp;
extern FILE	*ip;
extern char	filename[];
int		lno;
int		errcomments;
static int	lastlno=1;
static int 	last2c;
static int 	lastc;
static int 	lastcol;
static int	col;

void resetlex()
{
	fsp = 0;
	lastlno=1;
	lastcol=0;
	last2c = lastc = 0;
	errcomments=0;
}

char *findfile(name)
char	*name;
{
	static char	filename[128];
	char		*sfsbase();

	/* try file straight */
	strncpy(filename,name,128);
	if (access(filename,0)==0)
		return(filename);

	/* try SFS include directory */
	strncpy(filename,sfsbase(),128);
	strcat(filename,"/include/");
	strcat(filename,name);
	if (access(filename,0)==0)
		return(filename);

	/* try /usr/local directory */
	strcpy(filename,"/usr/local/include/");
	strcat(filename,name);
	if (access(filename,0)==0)
		return(filename);
	else
		return(name);
}

void pushfcontext(fname)
char *fname;
{
	strcpy(fstack[fsp].fname,filename);
	fstack[fsp].fp = ip;
	fstack[fsp].lno = lastlno;
	fstack[fsp].col = lastcol;
	fsp++;
	strcpy(filename,fname);
	if ((ip=fopen(findfile(filename),"r"))==NULL)
		spcerror("unable to find '%s'",filename);
	lastlno=1;
	lastcol=1;
}
int popfcontext()
{
	if (fsp==0)
		return(0);
	else {
		fclose(ip);
		fsp--;
		strcpy(filename,fstack[fsp].fname);
		ip = fstack[fsp].fp;
		lastlno = fstack[fsp].lno;
		lastcol = fstack[fsp].col;
	}
	return(1);
}

/* get a source character */
int charget()
{
	int	c;

	lno = lastlno;
	col = lastcol;

	if (lastc) {
		c = lastc;
		lastc = last2c;
		last2c = 0;
		return(c);
	}
	else do {
		if ((c = getc(ip))!=EOF) {
			if (c=='\n') {
				lastlno++;
				lastcol=1;
			}
			else if (c=='\t') do {
				lastcol++;
			} while ((lastcol%8)!=1);
			else
				lastcol++;
			return(c);
		}
	} while (popfcontext());
	return(EOF);
}

/* unget a source character */
void charunget(c)
int	c;
{
	last2c = lastc;
	lastc = c;
}

/* yylex token positioning */
YYLTYPE yylloc;

/* yylex definition */
int yylex()
{
	char	sbuf[100], *p;
	int	c,c2;
	Symbol	*s;
	int	numtype;

	/* store token position */
	yylloc.first_line = lno;
	yylloc.first_column = col;

	/* get a character */
	if ((c = charget())==EOF)
		return(0);

	/* get following character */
	c2 = charget();

	/* skip whitespace and comments */
	while ((c == ' ') || (c == '\t') || (c=='\n') || (c=='\r') ||
		(c=='{') || ((c=='(') && (c2=='*'))) {
		/* strip comments here */
		if (c=='{') {
			/* skip comment until } */
			if ((c=c2)==EOF)
				return(0);
			if (c=='$') {
				/* embedded command */
				if ((c=charget())==EOF)
					return(0);
				if (c=='I') {
					/* include file commmand */
					p = sbuf;
					if ((c=charget())==EOF)
						return(0);
					while (isspace(c)) {
						if ((c=charget())==EOF)
							return(0);
					}
					while ((c!='}') && !isspace(c)) {
						*p++ = c;
						if ((c=charget())==EOF)
							return(0);
					}
					*p='\0';
					while (c!='}') {
						if ((c=charget())==EOF)
							return(0);
					}

					/* push file context */
					pushfcontext(sbuf);
				}
				else if (c=='E')
					errcomments=1;
			}
			while (c!='}') {
				if (c=='{')
					warning(&yylloc,"nested comments?","");
				if ((c2=charget())==EOF)
					return(0);
				if ((c=='(') && (c2=='*'))
					warning(&yylloc,"nested comments?","");
				c = c2;
			}
			if ((c=charget())==EOF)
				return(0);
			c2=charget();
		}
		if ((c=='(') && (c2=='*')) {
			/* skip comment until *) */
			if ((c=charget())==EOF)
				return(0);
			c2=charget();
			while (!((c=='*') && (c2==')'))) {
				if (c=='{')
					warning(&yylloc,"nested comments?","");
				if ((c==EOF) || (c2==EOF))
					return(0);
				if ((c=='(') && (c2=='*'))
					warning(&yylloc,"nested comments?","");
				c = c2;
				c2 = charget();
			}
			if ((c=charget())==EOF)
				return(0);
			c2=charget();
		}
		while ((c==' ') || (c=='\t') || (c=='\n') || (c=='\r')) {
			if ((c = c2)==EOF)
				return(0);

			/* store token position */
			yylloc.first_line = lno;
			yylloc.first_column = col;

			c2 = charget();
		}
		if (c==EOF) {
			fprintf(stderr,"?SPC Compiler error, unexpected EOF\n");
			return(0);
		}

	}

	/* process numbers */
	if (isdigit(c)) {
		numtype=INTEGER_NUMBER;
		p = sbuf;
		*p++ = c;
		if ((c = c2)==EOF)
			return(0);
		c2 = charget();
		while (isdigit(c)) {
			if (p >= (sbuf + sizeof(sbuf) - 1)) {
				*p ='\0';
				warning(&yylloc,"number too long",sbuf);
			}
			*p++ = c;
			if ((c = c2)==EOF)
				return(0);
			c2 = charget();
		}
		if ((c=='.') && (c2=='.')) {
			/* return integer */
			*p='\0';
			s = installsymbol(sbuf,INTEGER_NUMBER,CONSTANT);
 			sprintf(sbuf,"%s",s->name);
			s->cname = strsave(sbuf);
			yylval = s;
			charunget(c2);
			charunget(c);
			return(s->type);
		}
		if (c=='.') {
			/* add in decimals */
			numtype=REAL_NUMBER;
			*p++ = c;
			if ((c = c2)==EOF)
				return(0);
			c2 = charget();
			while (isdigit(c)) {
				if (p >= (sbuf + sizeof(sbuf) - 1)) {
					*p ='\0';
					warning(&yylloc,"number too long",sbuf);
				}
				*p++ = c;
				if ((c = c2)==EOF)
					return(0);
				c2 = charget();
			}
		}
		if ((c=='e') || (c=='E')) {
			/* add in exponent */
			numtype=REAL_NUMBER;
			*p++ = c;
			if ((c = c2)==EOF)
				return(0);
			c2 = charget();
			if (c=='-') {
				*p++ = c;
				if ((c = c2)==EOF)
					return(0);
				c2 = charget();
			}
			while (isdigit(c)) {
				if (p >= (sbuf + sizeof(sbuf) - 1)) {
					*p ='\0';
					warning(&yylloc,"number too long",sbuf);
				}
				*p++ = c;
				if ((c = c2)==EOF)
					return(0);
				c2 = charget();
			}
		}
		*p = '\0';
		s = installsymbol(sbuf,(short)numtype,CONSTANT);
		if (numtype==INTEGER_NUMBER) {
 			sprintf(sbuf,"%s",s->name);
			s->cname = strsave(sbuf);
		}
		else
			s->cname = s->name;
		yylval = s;
		charunget(c2);
		charunget(c);
		return(s->type);
	}

	/* process identifiers */
	if (isalpha(c) || (c=='_')) {
		p = sbuf;
		*p++ = c;
		if ((c = c2)==EOF)
			return(0);
		c2 = charget();
		while (isalnum(c) || (c=='_')) {
			if (p >= (sbuf + sizeof(sbuf) - 1)) {
				*p ='\0';
				warning(&yylloc,"name too long",sbuf);
			}
			*p++ = c;
			if ((c = c2)==EOF)
				return(0);
			c2 = charget();
		}
		*p='\0';
		strlowercase(sbuf);
		if ((s=lookupsymbol(sbuf)) == NULL) {
			/* add new variable */
			s = installsymbol(sbuf,UNDEF_VAR,LOCAL);
			if (sbuf[0]=='_') {
				/* reference to built-in C function */
				*p = '\0';
				strcpy(sbuf,sbuf+1);
			}
			else
				*p++ = '_';
			*p='\0';
			s->cname = strsave(sbuf);
		}
		yylval = s;
		charunget(c2);
		charunget(c);
/*
fprintf(stderr,"[Symbol='%s',cname='%s',line=%ld,col=%ld]\n",
	s->name,s->cname,yylloc.first_line,yylloc.first_column);
*/
		return(s->type);
	}

	/* quoted strings */
	if (c=='\'') {
		if ((c = c2)==EOF)
			return(0);
		c2 = charget();
		if ((c=='\'') && (c2=='\'')) c='\\';
		for (p=sbuf;c != '\''; p++) {
			if ((c=='\n') || (c== EOF)) {
				warning(&yylloc,"missing quote","");
				break;
			}
			if (p>=sbuf + sizeof(sbuf) -1) {
				*p='\0';
				warning(&yylloc,"string too long",sbuf);
				break;
			}
			if (c=='\\') {
				if (c2=='\'') {
					c = c2;
					c2 = charget();
				}
				else
					*p++ = '\\';
			}
			*p = c;
			if ((c = c2)==EOF)
				return(0);
			c2 = charget();
			if ((c=='\'') && (c2=='\'')) c='\\';
		}
		*p = '\0';
		s = installsymbol("",STRING,CONSTANT);
		s->cname = strsave(sbuf);
		yylval = s;
		charunget(c2);
		if ((sbuf[0] && !sbuf[1]) || ((sbuf[0]=='\\') && sbuf[1] && !sbuf[2]))
			return(CHAR);
		else
			return(STRING);
	}
	yylval = NULL;

	/* graphs and digraphs */
	switch (c) {
	case '>':	return follow(c2,'=',GE_TOK,GT_TOK);
	case '<':	if (follow(c2,'=',LE_TOK,0)==LE_TOK)
				return(LE_TOK);
			else {
				c2 = charget();
				return follow(c2,'>',NE_TOK,LT_TOK);
			}
	case ':':	return follow(c2,'=',ASSIGN_TOK,':');
	case '.':	return follow(c2,'.',SUBRANGE_TOK,'.');
	default:
		charunget(c2);
		return(c);
	}
}

/* interpret backslash char */
int backslash(c)
int	c;
{
	char 	*p;
	static char transtab[]="b\bf\fn\nr\rt\t\"\"";
	if ((p=strchr(transtab,c)))
		return(*(p+1));
	return(c);
}

/* look ahead for digraphs */
int follow(c2,expect,ifyes,ifno)
int	c2;
int	expect,ifyes,ifno;
{
	if (c2==expect)
		return(ifyes);
	charunget(c2);
	return(ifno);
}

/* set up symbol expression type */
void setvaretype(itype,etype,btype,bsym)
int	itype,etype,btype;
Symbol	**bsym;
{
	Symbol	*sym;

	sym = symtab;

	while (sym) {
		if (sym->etype==itype) {
			if (etype==(short)EXPFILE)
				sym->type = FILE_VAR;
			sym->etype=etype;
			sym->btype=btype;
			sym->basesym[0]=bsym[0];
			sym->basesym[1]=bsym[1];
			sym->basesym[2]=bsym[2];
		}
		sym = sym->next;
	}
}

/* output NULL types as given type */
void outputnullvar(itype,typename)
int	itype;
char	*typename;
{
	Symbol	*sym;
	int	count=0;

	sym = symtab;

	while (sym) {
		if (sym->etype==itype) {
			if (count) outputstr(",");
			outputstr(typename);
			if (sym->status==ADRPAR)
				outputstr(" *");
			count++;
		}
		sym = sym->next;
	}
}

/* add a parameter and its type */
void addparamtype(sym,pno,pty)
Symbol *sym;
int	pno;
int	pty;
{
	if (sym->ptype)
		sym->ptype=realloc(sym->ptype,pno+2);
	else
		sym->ptype=malloc(pno+2);
	if (!sym->ptype)
		spcerror("failed to allocate memory",NULL);
	sym->ptype[pno]=pty;
}

/* adjust a parameter type */
void setparamtype(sym,p1,p2,pty)
Symbol *sym;
int	p1,p2;
int	pty;
{
	int	i;

	for (i=p1;i<=p2;i++)
		sym->ptype[i]=pty;
}

/* get a parameter type */
int getparamtype(sym,pno)
Symbol *sym;
int	pno;
{
	if (sym->ptype)
		return(sym->ptype[pno]);
	else
		return(0);
}

#ifdef EMO
FILE	*ip;
Symbol	*yylval;
main()
{
	Symbol 	*sym;
	int	ch;

	ip=stdin;

	initsymbol();

	while (ch=yylex()) {
		if (yylval) {
			sym = yylval;
			printf("name='%s' cname='%s' type=%d status=%d\n",
				sym->name,sym->cname,sym->type,sym->status);
		}
		else if (ch < 128)
			printf("name='%c'\n",ch);
		else
			printf("name=%d\n",ch);
	}
}

#endif

