/*
Subroutine to draw a log or lin axis

*/

#include <stdio.h>
#include "SFSCONFG.h"
#include "dgraph.h"
#include "dig.h"
#include "digdata.h"


/* Structure for division labels */

typedef struct { int maxdivs,ndivs;
		 int *mant,*exp,*prec,*order;
		 double *pos;
		 int *label,*lmant,*lexp; } DIGDIV;

#define LOG10 2.302585093
#define LOGe 0.434294482
#define PI 3.14159265

#define MAXDIVS 200
#define ASPECT 0.9	/* width/height for widest digit  */
#define NTICKLEN 4	/* Number of different tick lengths */
#define YTICK 1.0	/* Longest tick length / height	*/
#define TRULE 0.6	/* separation between ticks and labels */
#define LRULE 0.6	/* separation between labels and title */
#define HTITLE 1.1	/* Title lettering size		*/
#define LEXTEND 1.1	/* Label extension to left	*/
#define REXTEND 1.1	/* Label extension to right	*/

static double ticklens[NTICKLEN] = { 0.25,0.5,0.75,1.0};

#define LOGSCALE 1
#define FORCEINT 2
#define NOAXIS 4
#define TICKDOWN 8
#define TICKUP 16
#define LABELDOWN 32
#define LABELUP 64
#define TITLEDOWN 128
#define TITLEUP 256
#define GRIDX 512
#define GRIDY 1024

#ifdef __STDC__
void	divlabel(DIGDIV *,float,DIGSPACE *,char *,char *);
void	divlogprec(DIGDIV *divvs);
int	divlogpos(DIGDIV *,float,float,float,float);
void	logdivs(DIGDIV *,float,float,float,int);
void	divlinprec(DIGDIV *divs);
int	lindivs(DIGDIV *,float,float,float,int);
int	divlinpos(DIGDIV *,float,float,float,float);
int	divadd(DIGDIV *,int,int);
int 	exponent(double x);
int 	sndigit(int,int,char *);
int	ndigit(int,int);
void	divorder(DIGDIV *divvs);
int 	siexp(int,int,char *);
int	fpexp(int,int);
#else
void	divlabel();
void 	divlogprec();
int	divlogpos();
void	logdivs();
void	divlinprec();
int	lindivs();
int	divlinpos();
int	divadd();
int 	exponent();
int 	sndigit();
int	ndigit();
void	divorder();
int 	siexp();
int	fpexp();
#endif

#ifdef __STDC__
void	digaxis(int bundle,float vmin,float vmax,char * title,char *unit,int mode,float x1,float y1,float x2,float y2,float height,float spacing,DIGSPACE * spaces)
#else
void	digaxis(bundle,vmin,vmax,title,unit,mode,x1,y1,x2,y2,height,spacing,spaces)
DIGSPACE *spaces;	/* NULL or linked list of available space (0,axlen) */
int bundle;
float vmin,vmax;	/* axis limits */
char title[];
char unit[];		/* SI unit name (or NULL) */
int mode;
float x1,x2,y1,y2;	/* axis positions	*/
float spacing,height;		/* label character height */
/*
	mode = 0 - normal
	      +1 - log scale
	      +2 - Force integer labels
	      +4 - Don't draw axis
	      +8 - ticks below line
	     +16 - ticks above line
	     +24 - ticks through line
	     +32 - label below line
	     +64 - label above line
	    +128 - title below line
	    +256 - title above line
*/
#endif
{
    DIGSPACE *space,space0;

    int mant[MAXDIVS],exp[MAXDIVS];
    int itick=0;
    int forceint;
    int prec=0;
    int i,j;
    int modelabel,modetick;
    int bund[4];

    char modif[50],slabel[50],label[50];

    double axlen,cosang,sinang,ang;
    double xyscale,realy,xsin,ycos;
    double width;
    double ylabel=0,ytitle=0;
    double minrat,mininc;
    double tickup0,tickup=0,tickdown0,tickdown=0;
    double tlen,mtlen,mlen;
    double deltx,delty;
    double xpos,ypos,posj;
    double yup,ydown;

    DIGDIV divs;

    double pow(),sqrt(),atan2();

/* decode bundles 0:title, 1:labels/unit, 2:ticks, 3:axis */

    bund[0]  = bundle%100;
    for (i=1; i<4; i++) {
		bundle /= 100;
		if (!(bund[i] = bundle%100)) bund[i] = bund[i-1];
	}

/* set up Divs structure */

    divs.maxdivs = MAXDIVS;
    divs.mant = mant;
    divs.exp = exp;
    divs.ndivs = 0;

    if (mode & GRIDX) {
		deltx = x2-x1;
		delty = 0;
    }
    else if (mode & GRIDY) {
    	deltx = 0;
    	delty = y2-y1;
    }
    else {
    	deltx = x2-x1;
    	delty = y2-y1;
    }
    axlen = sqrt(deltx*deltx + delty*delty);
    cosang = deltx/axlen;
    sinang = delty/axlen;
    ang = (180.0/PI)*atan2(sinang,cosang);
    xyscale = digdata.aspect * digdata.yscale / digdata.xscale;
    ycos = cosang/xyscale;
    xsin = sinang*xyscale;
    realy = 1.0 / (xsin*sinang + ycos*cosang);
    ycos *= realy;
    xsin *= realy;
    width = ASPECT*height*realy;
    if (spaces) space = spaces;
    else {
	space = &space0;
	space0.left = (float)(-LEXTEND*width);
	space0.right = (float)(axlen+REXTEND*width);
	space0.next = NULL;
	}
    modelabel = mode & (LABELUP | LABELDOWN);
    modetick = mode & (TICKUP | TICKDOWN);

    tickup0 = (mode & TICKUP) ? height*YTICK : 0.0;
    tickdown0 = (mode & TICKDOWN) ? height*YTICK : 0.0;
    yup = tickup0 + height*TRULE;
    ydown = -tickdown0 - height*TRULE;
    if (mode & LABELUP) {
	ylabel = yup;
	ytitle = yup = ylabel + height*(1.0+LRULE);
	}
    else if (mode & LABELDOWN) {
	ylabel = ydown - height;
	ydown = ylabel - height*LRULE;
	ytitle = ydown - height*HTITLE;
	}
    if (mode & TITLEUP)  ytitle = yup;
    else if (mode & TITLEDOWN) ytitle = ydown - height*HTITLE;

/* Draw axis line */

    if(!(mode & NOAXIS)) {
    	if (mode & GRIDX)
			digline(bund[3],x1,y1,x2,y1);
    	else if (mode & GRIDY)
			digline(bund[3],x1,y1,x1,y2);
    	else
			digline(bund[3],x1,y1,x2,y2);
    }

/* Calculate tick positions */

    if (modetick | modelabel) {
	forceint = ((mode & FORCEINT) != 0);
	if (mode & LOGSCALE) {
	    minrat = pow(vmax/vmin,spacing/axlen);
	    logdivs(&divs,(float)vmin,(float)vmax,(float)minrat,forceint);
	    divlogpos(&divs,(float)vmin,(float)vmax,(float)0.0,(float)axlen);
	    divlogprec(&divs);
	    }
	else {
	    mininc = (vmax-vmin)*spacing/axlen;
	    lindivs(&divs,(float)vmin,(float)vmax,(float)mininc,forceint);
	    divlinpos(&divs,(float)vmin,(float)vmax,(float)0.0,(float)axlen);
	    divlinprec(&divs);
	    }
	divorder(&divs);

    /* Now sort out ticks */

	if (modetick) {
	    itick = NTICKLEN-1;
	    tickup = ticklens[NTICKLEN-1]*tickup0;
	    tickdown = ticklens[NTICKLEN-1]*tickdown0;
	    prec = divs.prec[divs.order[1]];
	    }
	if (modelabel) {
	    divlabel(&divs,(float)width,space,unit,modif);
	    }
	for (i=0; i<divs.ndivs; i++) {
	    j = divs.order[i];
	    posj = divs.pos[j];
	    xpos = x1+posj*cosang;
	    ypos = y1+posj*sinang;
	    if (modetick) {
		if (i && itick && (divs.prec[j] != prec)) {
		    prec = divs.prec[j];
		    itick--;
		    tickup = ticklens[itick]*tickup0;
		    tickdown = ticklens[itick]*tickdown0;
		    }
		digline(bund[2],(float)(xpos-tickup*xsin),(float)(ypos+tickup*ycos),
			       (float)(xpos+tickdown*xsin),(float)(ypos-tickdown*ycos));
		if (mode & GRIDX) {
			digline((divs.label[j]>=0)?38:94,(float)(xpos-tickup*xsin),(float)y1,
			       (float)(xpos+tickdown*xsin),(float)y2);
		}
		if (mode & GRIDY) {
			digline((divs.label[j]>=0)?38:94,(float)x1,(float)(ypos+tickup*ycos),
			       (float)x2,(float)(ypos-tickdown*ycos));
		}

	    }
	    if(modelabel && (divs.label[j] >= 0)) {
		sndigit(divs.lmant[j],divs.lexp[j],slabel);
		if(divs.label[j]) {
		    sprintf(label,"%s%c",slabel,divs.label[j]);
		    }
		else {
		    sprintf(label,"%s",slabel);
		    }
		digboxtext(bund[1],(float)(xpos-ylabel*xsin),(float)(ypos+ylabel*ycos),
			 (float)axlen,(float)height,(float)ang,label,1);
		}
	    }
	}

    if(modelabel)
	mlen = digvtextlen(modif,(float)height,(float)ang,1);
    else mlen = 0.0;
    if ((mode & (TITLEUP | TITLEDOWN)) && (title)) {
	if (mlen > 0.0) {
	    tlen = digvtextlen(title,(float)(height*HTITLE),(float)ang,1);
	    if((mtlen = tlen+mlen) > axlen) {
		mlen *= axlen/mtlen;
		}
	    }
	tlen = axlen-mlen;
	posj = 0.5*tlen;
	digboxtext(bund[0],(float)(x1+posj*cosang-ytitle*xsin),
			  (float)(y1+posj*sinang+ytitle*ycos),
			  (float)tlen,(float)(height*HTITLE),(float)ang,title,1);
	}
    if (mlen > 0.0) {
	if (mlen > axlen) mlen = axlen;
	posj = axlen-0.5*mlen;
	digboxtext(bund[1],(float)(x1+posj*cosang-ytitle*xsin),
			  (float)(y1+posj*sinang+ytitle*ycos),
			  (float)mlen,(float)height,(float)ang,modif,1);
	}
    free(divs.prec);
    free(divs.order);
    free(divs.pos);
    if (modelabel) {
	free(divs.label);
	free(divs.lmant);
	free(divs.lexp);
	}
    }

#ifdef __STDC__
void	divlabel(DIGDIV *divs,float width,DIGSPACE *spaces,char *unit,char *modif)
#else
void	divlabel(divs,width,spaces,unit,modif)
DIGDIV 	*divs;
float 	width;
DIGSPACE *spaces;
char 	unit[],modif[];
#endif
/*
	This routine selects which divisions to label
	and indicates them in the label[] array. The
	lmant[] and lexp[] arrays are set to the
	appropriate mantissa and exponent while modif
	contains the string to be printed on the axis.

	Bugs: 1) May not work if ndivs < 2
	      2) Should allow for a global offset
	      3) If only 1 extra label is produced at a
		 new precedence level and there were >=3
		 at the previous level, then the new one
		 should be ignored.
*/
{
    DIGSPACE	*space,*space0,*space1;
    int ndivs;
    int *order,*mant,*exp,*prec;
    int *temp,*label,*lmant,*label2,*lmant2,*lexp2,*lexp;
    int state,i,j,nspace;
    int oldprec;
    int lind,rind,preclim;
    int expl,expr,lsiexp,rsiexp,expi;
    int bestcount,bestglob=0;
    int nlabels,nspaces;
    int niceness,target,prorder=0,labexp,globexp;

    char lsichar,rsichar;
    char sichar,sichari,str1[100];

    double *pos;
    double posi,hwidth;
    double need,labelsep=0;

    ndivs = divs->ndivs;
    divs->label = label2 = (int*) malloc(ndivs*sizeof(int));
    label = (int*) malloc(ndivs*sizeof(int));
    lmant = (int*) malloc(ndivs*sizeof(int));
    divs->lmant = lmant2 = (int*) malloc(ndivs*sizeof(int));
    divs->lexp = lexp2 = (int*) malloc(ndivs*sizeof(int));
    lexp = (int*) malloc(ndivs*sizeof(int));

/* allocate space array */

    nspaces = 2;
    space = spaces;
    while ((space = space->next))  nspaces++;

    space0 = (DIGSPACE *) malloc((ndivs+nspaces)*sizeof(DIGSPACE));

    mant = divs->mant;
    exp = divs->exp;
    prec = divs->prec;
    order = divs->order;
    pos = divs->pos;
    space0->right = space0->left = (float)1.0E20;	/* dummy space at infinity */

    hwidth = width*0.5;

/* Work out a suitable multiplying factor */

    lind = rind = order[0];
    preclim = prec[1] - (prec[1]&1);
    i = 1;
    while((i<ndivs) && (prec[(j=order[i])] >= preclim)) {
	if (j<lind) lind=j;
	else if (j>rind) rind=j;
	i++;
	}
    expl = fpexp(mant[lind],exp[lind]);	/* leftmost major division */
    expr = mant[rind] ? fpexp(mant[rind],exp[rind]) : expl;
    lsiexp = siexp(mant[lind],exp[lind],&lsichar);
    rsiexp = mant[rind] ? siexp(mant[rind],exp[rind],&rsichar) : lsiexp;
    if (!mant[lind]) {
	lsiexp = rsiexp;
	expl = expr;
	}

/* Loop for decreasing prettiness until we get >= 2 labels */

    bestcount = 0;			/* initialise to awful count */
    for (niceness = 2; niceness >=0; niceness-- ) {
        switch (niceness) {
	    case 2:
		prorder = 1;		/* order by precedence */
		labelsep = 2.0*width;	/* wide separation	*/
		break;
	    case 1:
		prorder = 0;		/* any old order will do now */
		break;
	    case 0:
		labelsep = 0.5*width;	/* close labels */
		break;
	    }
/* This is the main state-driven loop	*/

	target = 3;			/* try for 3 labels	*/
	labexp = 0;			/* no label exponents	*/
	sichari = 0;			/* no si char on label	*/
	globexp = 0;			/* no global exponent   */
	state = 100* !unit;		/* 0 if unit specified else 100 */
	do {				/* state driven loop	*/
	    switch (state) {
		case 0:			/* unit: use multiplier */
		    if (lsiexp == rsiexp) globexp = lsiexp;
		    else {
			labexp = 1;	/* use individual si exponents */
			}
		    break;
		case 1:
		    target = 2;		/* lower our requirements */
		    sichari = 0;	/* no si multiplier on labels */
		    labexp = 0;
		    globexp = expl<expr ? expl : expr;
		    state = -1;
		    break;

    /* Should allow a global offset */

	    /*  case 100:  **** default **** */
		case 101:			/* no unit, si exponent */
		    globexp = lsiexp<rsiexp ? lsiexp : rsiexp;
		    state = 0;
		    break;
		}
	    nlabels = 0;
	    nspace=nspaces;			/* initialize space	*/
	    space1 = space0;
	    space = spaces;
	    while(space) {	/* copy across from input argument */
		space1++;
		space1->left = space->left;
		space1->right = space->right;
		space1->next = space1+1;
		space = space->next;
		}
	    space1->next = space0;
	    space = space0+1;

	    for (i=ndivs-1; i>=0; i--) {	/* zero labels	*/
		label[i] = -1;
		}

	    oldprec = 0;
	    for (j=0; j<ndivs; j++) {
		if (prorder) i = order[j];
		else i = j;
		if (oldprec != prec[i]) {
		    if (nlabels>=5) break; 	/* >= 5 labels found */
		    space = space0+1;
		    oldprec = prec[i];
		    }
		posi = pos[i];
		expi = exp[i]-globexp;
		if (labexp) expi -= siexp(mant[i],expi,&sichari);
		while (space->right < posi) space = space->next;
		if (space->left > posi) continue;	/* no room */
		need = hwidth*(ndigit(mant[i],expi) + 2*(sichari != '\0'));
		if ((need<(space->right-posi)) &&
		    (need<(posi - space->left))) {

    /* Add in a new label	*/

		    nlabels++;
		    label[i] = sichari;
		    lmant[i] = mant[i];
		    lexp[i] = expi;
		    space1 = space0+nspace;
		    space1->left = posi+need+labelsep;
		    space1->right = space->right;
		    space1->next = space->next;
		    space->right = posi-need-labelsep;
		    space->next = space1;
		    space = space1;	/* move on to next one	*/
		    nspace++;
		    }
		}
	    if (nlabels > bestcount) {
		bestcount = nlabels;
		bestglob = globexp;
		temp = label2;
		divs->label = label2 = label;
		label = temp;
		temp = lmant2;
		divs->lmant = lmant2 = lmant;
		lmant = temp;
		temp = lexp2;
		divs->lexp = lexp2 = lexp;
		lexp = temp;
		}
	    if (state < 0) break;	/* try less niceness */
	    state++;
	    } while(bestcount < target);
	if (bestcount >= 2) break;	/* don't reduce niceness if >= 2 */
	}				/* end of niceness loop */

/* Finally sort out the scaling label */

    *modif = '\0';
    if(unit) {
	if (bestglob) {
	    bestglob -= siexp(1,bestglob,&sichar);
	    if (bestglob) {
		sndigit(1,bestglob,str1);
		sprintf(modif,"*%s",str1);
		}
	    if (sichar) {
		sprintf(str1,"%c",sichar);
		strcat(modif,str1);
		}
	    }
	strcat(modif,unit);

	}
    else {
	if (bestglob) {
	    if((bestglob>=-2) && (bestglob<=3)) {
		sndigit(1,bestglob,str1);
		sprintf(modif,"*%s",str1);
		}
	    else sprintf(modif,"*10**%d",bestglob);
	    }
	}
    free(label); /* could all be combined */
    free(lmant);
    free(lexp);
    free(space0);
}

void divlogprec(divvs)
DIGDIV *divvs;
/*
	Calculate the precedence of divisions
*/
{
    int i;
    int absmant,mant,prec;

    divvs->prec = (int*) malloc(divvs->ndivs * sizeof(int));
    for (i=divvs->ndivs-1; i>=0; i--) {
	if ((mant=divvs->mant[i]) == 1) {
	    divvs->prec[i] = 2 + (!(divvs->exp[i] % 3));
	    }
	else {
	    prec = (!(mant % 5));
	    absmant = abs(mant);
	    while (absmant >= 10) {
		absmant /= 10;
		prec -= 2;
		}
	    divvs->prec[i] = prec;
	    }
	}
    }

#ifdef __STDC__
int divlogpos(DIGDIV *divs,float vmin,float vmax,float left,float right)
#else
int divlogpos(divs,vmin,vmax,left,right)
DIGDIV *divs;
float vmin,vmax,left,right;
#endif
{
    double log();
    int ndivs,i,*mant,*iexp;
    double *pos,a,b;

    ndivs = divs->ndivs;
    divs->pos = pos = (double *) malloc(ndivs*sizeof(double));
    mant = divs->mant;
    iexp = divs->exp;
    a = (right-left)/log(vmax/vmin);
    b = left - log(vmin)*a;

    for (i=ndivs-1; i>=0; i--) {
	pos[i] = (log((double) mant[i])+LOG10*iexp[i])*a + b;
	}
    return(0);
    }

#ifdef __STDC__
void logdivs(DIGDIV *divvs,float vmin,float vmax,float minrat,int forceint)
#else
void logdivs(divvs,vmin,vmax,minrat,forceint)
float vmin,vmax,minrat;
int forceint;
DIGDIV *divvs;
#endif
{
    double log(),exp(),pow();

    static double divs125[3] = { 1.0, 2.0, 5.0};
    static int idivs125[3] = { 1, 2, 5};
    static double divs12468[5] = { 1.0, 2.0, 4.0, 6.0, 8.0 };
    static int idivs12468[5] = { 1, 2, 4, 6, 8};
    double *decdivs;
    int *idecdivs;

    int log10min;
    int ndecdivs,decindex;
    int i;
    int iexp,imant;

    double minpow;
    double logmin,logmax;
    double valdivnew,valdivold,valdiv,valpow;
    double mininc;

    if (forceint && (vmin < 1.0)) vmin = (float)1.0;	/* force integers */
    log10min = (int) (log(vmin)*LOGe - 0.001);
    minpow = exp(LOG10* log10min);	/* exact power of 10 */
    if (minrat>2.0) {		/* whole powers of 10	*/
	logmin = log(vmin)*LOGe;
	logmax = log(vmax)*LOGe;
	lindivs(divvs,(float)logmin,(float)logmax,(float)(log(minrat)*LOGe),1);
	for (i=divvs->ndivs-1; i>=0; i--) {	/* replace by 10**n */
	    imant = divvs->mant[i];
	    iexp  = divvs->exp[i];
	    while (iexp--) imant *= 10;
	    divvs->mant[i] = 1;
	    divvs->exp[i] = imant;
	    }
	}
    else if (minrat>1.111) {
	if (minrat>1.25) {
	    ndecdivs = 3;
	    decdivs = divs125;
	    idecdivs = idivs125;
	    }
	else {
	    ndecdivs = 5;
	    decdivs = divs12468;
	    idecdivs = idivs12468;
	    }
	valdiv = valpow = minpow;
	decindex = 0;

	while(valdiv < vmin) {
	    decindex = (decindex+1) % ndecdivs;
	    if (decindex) valdiv = valpow*decdivs[decindex];
	    else {
		valdiv = valpow *= 10.0;
		log10min++;
		}
	    }
	while(valdiv <= vmax) {
	    divadd(divvs,idecdivs[decindex],log10min);
	    decindex = (decindex+1) % ndecdivs;
	    if (decindex) valdiv = valpow*decdivs[decindex];
	    else {
		valdiv = valpow *= 10.0;
		log10min++;
		}
	    }
	}

    else {				/* normal linear	*/
	valdiv = valpow = minpow;
	decindex = 1;

	while (valdiv < vmin) {
	    decindex = (decindex+1) % 10;
	    if (decindex) valdiv = valpow * decindex;
	    else {
		valdiv = valpow *= 10.0;
		decindex = 1;
		}
	    }
	mininc = (1.0 - 1.0/minrat);
	valdivold = vmin;
	while(valdivold <= vmax) {
	    valdivnew = (valdiv > vmax) ? vmax : valdiv;
	    lindivs(divvs,(float)valdivold,(float)valdivnew,(float)(valdivnew*mininc),forceint);
	    valdivold = valdiv;
	    decindex = (decindex+1) % 10;
	    if (decindex) valdiv = valpow * decindex;
	    else {
		valdiv = valpow *= 10.0;
		decindex = 1;
		}
	    }



	}
    }

void divlinprec(divs)
DIGDIV *divs;
/*
	Calculate the precedence of divisions
*/
{
    int i;
    int *prec;

    divs->prec = prec = (int*) malloc(divs->ndivs * sizeof(int));
    for (i=divs->ndivs-1; i>=0; i--) {
	if (!(divs->mant[i])) prec[i] = 10000;
	else prec[i] = 2*divs->exp[i] + (!(divs->mant[i] % 5));
	}
    }

#ifdef __STDC__
int divlinpos(DIGDIV *divs,float vmin,float vmax,float left,float right)
#else
int divlinpos(divs,vmin,vmax,left,right)
DIGDIV *divs;
float vmin,vmax,left,right;
#endif
{
    double exp();
    int ndivs,i,*mant,*iexp;
    double *pos,a,b;

    ndivs = divs->ndivs;
    divs->pos = pos = (double*) malloc(ndivs*sizeof(double));
    mant = divs->mant;
    iexp = divs->exp;
    a = (right-left)/(vmax-vmin);
    b = left - vmin*a;

    for (i=ndivs-1; i>=0; i--) {
	pos[i] = mant[i]*exp(LOG10*iexp[i])*a + b;
	}
    return(0);
    }

#ifdef __STDC__
int lindivs(DIGDIV *divvs,float xmin,float xmax,float mininc,int forceint)
#else
int lindivs(divvs,xmin,xmax,mininc,forceint)
float xmin,xmax,mininc;
int forceint;
DIGDIV *divvs;
#endif
/*
    This routine calculates the divisions for a linear axis
    going from xmin to xmax with a minimum division interval
    of mininc.
*/
{
    double floor(),ceil(),pow();

    double minpow,divval;
    int i;
    int imin,imax,ratio,iexp,imant;

    if (forceint && (mininc < 0.9)) mininc = (float)0.9;	/* force integers */
    xmin -= (float)(mininc*0.01);
    xmax += (float)(mininc*0.01);	/* round up to include ends	*/
    iexp = exponent(mininc);	/* 10**iexp <= mininc	*/
    minpow = pow(10.0,(double) iexp);
    imant = 1;
    ratio = (int)(mininc/minpow);
    if (ratio >= 2) {
	if (ratio >= 5) {
	    iexp++;
	    minpow = minpow * 10.0;
	    }
	else imant = 5;
	}
    else imant = ratio + 1;	/* = 1 or 2	*/
    divval = minpow*imant;

/* now output the division values	*/

    imin = (int)(ceil(xmin/divval));
    imax = (int)(floor(xmax/divval));
    for (i=imin; i<=imax; i++) {
	divadd(divvs,i*imant,iexp);
	}
    return(0);
    }

int divadd(divvs,mant,exp)
int mant,exp;
DIGDIV *divvs;
/*
    This routine adds a division into the divs[] array
    and calculates its precedence.
*/
{
    int ndivvs;

    ndivvs = divvs->ndivs;
    if (ndivvs >= divvs->maxdivs) return(1);

    if (!mant) {		/* zero is a special case */
	if (ndivvs && !(divvs->mant[ndivvs-1])) return(1);	/* duplicate	*/
	divvs->mant[ndivvs] = mant;
	divvs->exp[ndivvs] = 0;
	}

    else {

/* First remove trailing zeros */

	 while (!(mant % 10)) {
	     mant /= 10;
	     exp++;
	     }
	if (ndivvs && (divvs->mant[ndivvs-1] == mant) && (divvs->exp[ndivvs-1] == exp)) return(1);
	divvs->mant[ndivvs] = mant;
	divvs->exp[ndivvs] = exp;
	}

    divvs->ndivs++;
    return(0);
    }

int exponent(x)
double x;
/*
    This routine returns the integer n such that
    10**n <= x

    x must be > 0
*/
{
    double y,valexp;
    int n;

    valexp = 1.0;
    n = 0;
    if (x > valexp) {
	y = x * 0.001;
	while (y > valexp) {
	    valexp = valexp * 1000.0;
	    n += 3;
	    }
	while (x >= valexp*10.0) {
	    valexp = valexp * 10.0;
	    n++;
	    }
	}
    else {
	y = x * 1000.0;
	while (y < valexp) {
	    valexp = valexp * 0.001;
	    n -= 3;
	    }
	while (x < valexp) {
	    valexp = valexp * 0.1;
	    n--;
	    }
	}
    return(n);
    }
#define MAX 50

int sndigit(m,e,s)
int m,e;
char *s;
/********************************************************
*							*
* Routine to return the number of characters needed to	*
* print a number.					*
*							*
*********************************************************
* Inputs:	m,e	input number is m * 10**e	*
*							*
* Outputs:	s[]	output string is put into s[]	*
*							*
* Return Value:		number of characters to print	*
*********************************************************
* External References:					*
*							*
*********************************************************
*    Mike Brookes       Imperial College    June 1986	*
********************************************************/
{
    char *s0,*p;
    static char buf[MAX];

    s0=s;
    p = buf+MAX-1;
    if (m<0) { m = -m; *s++ = '-'; }
    else if(!m) e=0;
    while (e<0 && m%10 == 0) {	/* remove trailing zeros after . */
	e++;
	m /= 10;
	}
    while (e<0) {
	if(p>buf) *--p = '0' + (char) (m%10);
	e++;
	m /=10;
	}
    if(p != buf+MAX-1) *--p = '.';
    while (e-- && p>buf) *--p = '0';
    if (m) {
	while (m) {
	    if(p>buf) *--p = '0' + (char) (m%10);
	    m/=10;
	    }
	}
    else *s++ = '0';
    while ((*s++ = *p++));
    return(s-s0-1);
    }

int ndigit(m,e)
int m,e;
/********************************************************
*							*
* Routine to return the number of characters needed to	*
* print a number.					*
*							*
*********************************************************
* Inputs:	m,e	input number is m * 10**e	*
*							*
* Outputs:						*
*							*
* Return Value:		number of characters to print	*
*********************************************************
* External References:					*
*							*
*********************************************************
*    Mike Brookes       Imperial College    May 1986	*
********************************************************/
{
    int sign,n;

    if (m) {
	if (m<0) { m = -m; sign = 1; }
	else	 { sign = 0; }
	while (e<0 && m%10 == 0) {	/* remove trailing zeros after . */
	    e++;
	    m /= 10;
	    }
	n = 1+e;
	m /= 10;
	while ( m >= 100 ) { m /= 1000; n += 3; }
	while ( m )	   { m /= 10;   n++;    }
	if ( e < 0 ) {
	    if ( n<1 ) n = 1;
	    n += 1-e;
	    }
	return (n+sign);
	}
    else return (1);		/* 0 takes 1 character	*/
    }

void divorder(divvs)
DIGDIV *divvs;
/*
    This routine creates the array order containing division
    indexes arranged in descending order of precedence
*/
{
    int ndivs,preci;
    int *order,*prec;
    int i,j,k;

    ndivs = divvs->ndivs;
    divvs->order = order = (int*) malloc(ndivs*sizeof(int));
    prec = divvs->prec;

    for (i=ndivs-1; i>=0; i--) {
	preci = prec[i];
	for (j=i+1; j<ndivs; j++) {
	    k = order[j];
	    if (prec[k] > preci) order[j-1] = order[j];
	    else {
		order[j-1] = i;
		break;
		}
	    }
	if (j==ndivs) order[ndivs-1] = i;
	}
    }

int siexp(m,e,c)
int m,e;
char *c;
/*
    This routine returns the value of the multiplier needed for
    printing in engineering format together with
    the appropriate SI multiplier character

    The ruturned value is always a multiple of 3
*/
{
    static char sich[] = { 'a','f','p','n','u','m','\0','k','M','G','T' };

    int si;

    e = fpexp(m,e);
    si = (e>=0) ? ((e>=12) ? 4 : e/3) : ((e<=-16) ? -6 : (e-2)/3);
    *c = sich[si+6];
    return (3*si);
    }
int fpexp(m,e)
int m,e;
/*
    This routine returns the exponent needed to print in
    floating point format
*/
{
    m = abs(m);
    while (m >= 1000) {
	m/=1000;
	e += 3;
	}
    while (m >= 10) {
	m/=10;
	e++;
	}
    return(e);
    }


#ifdef EMO
main()
{
    DIGSPACE *space,spaces[20];
    int init=1,mode,uspec;
    int nspaces,i;
    double height,vmin,vmax,x1,y1,x2,y2;
    double ang,spacing;

    double sin(),cos();

    for (;;) {
	if (init) {
	    init = 0;
	    printf("0 for no unit: ");
	    scanf("%d",&uspec);
	    printf("Enter height: ");
	    scanf("%f",&height);
	    printf("Enter spacing: ");
	    scanf("%f",&spacing);
	    printf("Enter angle: ");
	    scanf("%f",&ang);
	    x2 = cos(ang);
	    y2 = sin(ang);
	    x1 = -x2;
	    y1 = -y2;
	    printf("Enter number of label spaces (or 0): ");
	    scanf("%d",&nspaces);
	    if (nspaces) {
		for (i=0; i<nspaces; i++) {
		    printf("Enter space range: ");
		    scanf("%f%f",&(spaces[i].left),&(spaces[i].right));
		    spaces[i].next = &(spaces[i+1]);
		    }
		spaces[nspaces-1].next = NULL;
		space = &(spaces[0]);
		}
	    else space = NULL;

	    printf("Enter 1,0 to reenter parameters\n");
	    }
	printf("Enter mode: ");
	scanf("%d",&mode);
	printf("Enter min,max: ");
	scanf("%f%f",&vmin,&vmax);
	if (vmin > vmax) {
	    init = 1;
	    continue;
	    }
	digstart('\0',NULL,1);
	digclearscreen();
	digscale(4.4,2.2,0);
	digorigin(2.2,1.1);
	if (uspec)
	    digaxis(1040510,vmin,vmax,"title","Hz",mode,x1,y1,x2,y2,height,spacing,space);
	else
	    digaxis(1040510,vmin,vmax,"title",NULL,mode,x1,y1,x2,y2,height,spacing,space);
	digend();
	}
    }
#endif
