/* wordperf -- DIG graphics output to WordPerfect .WPG file */

/* Mark Huckvale - University College London */

/* version 1.0 - May 1993 */

#include "SFSCONFG.h"
#ifdef EVw		/* <- letter chosen for device type */

#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include "dig.h"
#include "digdata.h"
#include "wp.h"

#define PAGEWIDTH	6.5	/* inches = A4 with 1" margins */
#define PAGEHEIGHT	5.0	/* inches = approx 1/2 A4 */
#define DPI		300	/* work to laser resolution */
#define CHARWIDTH	0.050 	/* 1/20" */
#define CHARHEIGHT	0.066	/* 1/16" */
#define MIN(x,y) (((x)<(y))?(x):(y))

/* global data */
struct {
	short mode;		/* 0=?, 1=initialised */
	short currbun;		/* current bundle */
	short currtextbun;	/* current bundle */
	char  myfilename[SFSMAXFILENAME];
	FILE  *tfp;		/* temporary file */
} digDEVw;

static wp_gscale_end();

/* device characteristics */
int wp_params(subtype,digdev)
int	subtype;
struct digdevice *digdev;
{
	digdev->nhoriz = (int)(DPI*PAGEWIDTH);
	digdev->nvert  = (int)(DPI*PAGEHEIGHT);
	digdev->aspect = (float)1.0;
	digdev->chwidth = (int)(DPI * CHARWIDTH);
	digdev->chheight = (int)(DPI * CHARHEIGHT);
	digdev->greylevels = 16;
	digdev->greypixwidth = 4;
	digdev->greypixheight = 4;
	return(0);
}

/* open output file */
int wp_outfile(fp)
FILE	**fp;
{
	char	*p,*getenv();
	if ((p=getenv("WPFILE"))) {
		if ((*fp=fopen(p,"w+b")))
			return(0);
	}
	if ((*fp=fopen("dig.wpg","w+b")))
		return(0);
	else
		error("unable to open WP output file");
	return(0);
}

/* open graphics */
int wp_open()
{
	prefix			wphead;
	wpg_start_record 	wpgstart;
	wpg_start_record2 	wpgstart2;

	if (!(digDEVw.mode)) {
		/* open temporary file */
		digDEVw.tfp = /*tmpfile(); */ fopen("dig.tmp","w+b");

		/* standard WordPerfect Corp header */ 
		wphead.pfmagic = PFMAGIC;
		strncpy(wphead.pfname,PFNAME,3);
		wphead.pfleng = sizeof(wphead);
		wphead.pfprod = PFWP;
		wphead.pftype = PFWPGIF;
		wphead.pfmajv = 1;
		wphead.pfminv = 0;
		wphead.pfcrpt = 0;
		wphead.pfreserved = 0;
		if (fwrite((char *)&wphead,sizeof(prefix),1,digDEVw.tfp) != 1)
			error("write error on temporary file");

		/* WPG start record */
		wpgstart.wpgcode = 0x0F;
		wpgstart.wpgsize = 6;
		wpgstart.wpgversion = 1;
		wpgstart.wpguseps = 0;
		wpgstart.wpgwidth = digdata.nhoriz << 2;
		wpgstart.wpgheight = digdata.nvert << 2;
		if (fwrite((char *)&wpgstart,sizeof(wpg_start_record),1,digDEVw.tfp) != 1)
			error("write error on temporary file");

		/* WPG start record 2 */
		wpgstart2.wpgcode = 0x19;
		wpgstart2.wpgsize = 5;
		wpgstart2.wpgsubtype = 0;
		wpgstart2.wpgslen_lo = 2;
		wpgstart2.wpgslen_hi = 0;
		wpgstart2.wpgbgcol = 0x0F;
		wpgstart2.wpgfgcol = 0x00;
		if (fwrite((char *)&wpgstart2,7,1,digDEVw.tfp) != 1)
			error("write error on temporary file");

		(digDEVw.mode)++;
		digDEVw.currbun= -1;
		digDEVw.currtextbun= -1;
	}
	return(0);
}

/* close graphics - return to text mode */
int wp_close()
{
	wpg_end		wpgend;
	char		xbuf[512];
	int		len;

	wp_gscale_end();

	if (digDEVw.mode) {
		/* write end record */
		wpgend.wpgcode = 0x10;
		wpgend.wpgsize = 0;
		if (fwrite((char *)&wpgend,sizeof(wpg_end),1,digDEVw.tfp) != 1)
			error("write error on temporary file");

		/* xfer from temporary file */
		fflush(digDEVw.tfp); 
		rewind(digDEVw.tfp);
		while ((len=fread(xbuf,1,512,digDEVw.tfp)) > 0)
			fwrite(xbuf,1,len,digdata.outfile);

		/* close file */
		fclose(digDEVw.tfp);
		remove("dig.tmp");
		fclose(digdata.outfile);

		/* return to text mode */
		digDEVw.mode = 0;
	}
	return(0);
}

/* interrupt graphics */
int wp_inter()
{
	/* deal with interrupted drawing */
	wp_close();
	fprintf(stderr,"Interrupt\n");
	fflush(stderr);
	return(0);
}

/* clear screen */
int wp_clear()
{
	/* clear screen to background -> NULL */
	wp_gscale_end();
	return(0);
}

/* set up colour from bundle */
static int wp_colour(bundle)
int	bundle;
{
	wpg_lineattr	wpl;

	wp_gscale_end();

	/* bundles are 0=background,1-100 various colours/line styles */
	if (bundle != digDEVw.currbun) {
		wpl.wpgcode = WPG_LINEATTR;
		wpl.wpgsize = 4;
		wpl.wpgstyle = 1;	/* solid */
		wpl.wpgcolour = 0;	/* black */
		wpl.wpgwidth = 1;	/* 1 pixel */
		if (fwrite((char *)&wpl,sizeof(wpg_lineattr),1,digDEVw.tfp) != 1)
			error("write error on temporary file");
		digDEVw.currbun = bundle;
	}
	return(0);
}

static int wp_textcolour(bundle)
int	bundle;
{
	wpg_gtextattr		wpgtext;

	/* line attributes */
	wp_colour(bundle);

	/* WPG Graphics Text Attributes */
	if (bundle != digDEVw.currtextbun) {
		memset((char *)&wpgtext,0,sizeof(wpg_gtextattr));
		wpgtext.wpgcode = WPG_TEXTATTR;
		wpgtext.wpgsize = 22;
		wpgtext.wpgcellwidth = (int)(1200 * CHARWIDTH);
		wpgtext.wpgcellheight = (int)(1200 * CHARHEIGHT);
		wpgtext.wpgfonttype = 0x0D50;	/* courier */
		wpgtext.wpghalign = 0;		/* left */
		wpgtext.wpgvalign = 3;		/* bottom */
		wpgtext.wpgcolour = 0x00;	/* black */
		wpgtext.wpgangle = 0;
		if (fwrite((char *)&wpgtext,sizeof(wpg_gtextattr),1,digDEVw.tfp) != 1)
			error("write error on temporary file");
		digDEVw.currtextbun = bundle;
	}
	return(0);
}

/* draw a poly-line */
int wp_polyline(bundle,buff,len)
int	bundle;
short	*buff;
int	len;
{
	wpg_polyline	wpl;
	unsigned short	xy[2];

	/* sort out colours */
	wp_colour(bundle);

	/* write line header */
	wpl.wpgcode = WPG_POLYLINE;
	wpl.wpgsizeflag = 0xFF;
	wpl.wpgsizehi = 0x8000 | ((2*len+2) >> 16);
	wpl.wpgsizelo = (2*len+2) & 0xFFFF;
	wpl.wpgnumcoord = len/2;
	fwrite((char *)&wpl,sizeof(wpg_polyline),1,digDEVw.tfp);

	/* plot points */
	while (len > 0) {
		xy[0] = (*buff++) << 2;
		xy[1] = (*buff++) << 2;
		fwrite((char *)xy,2,2,digDEVw.tfp);

		/* plot line to x,y */
		len -= 2;
	}
	return(0);

}

/* draw 'hardware' text */
int wp_text(bundle,x,y,str)
int	bundle;
int	x,y;
char	*str;
{
	wpg_gtext	wpt;
	int		len;

	/* sort out colours */
	wp_textcolour(bundle);

	/* write text header */
	len = strlen(str);
	wpt.wpgcode = WPG_TEXT;
	wpt.wpgsizeflag = 0xFF;
	wpt.wpgsizehi = 0x8000 | ((len+6) >> 16);
	wpt.wpgsizelo = (len+6) & 0xFFFF;
	wpt.wpgtextlen = len;
	wpt.wpgx = x << 2;
	wpt.wpgy = y << 2;
	fwrite((char *)&wpt,sizeof(wpg_gtext),1,digDEVw.tfp);

	/* write string */
	fwrite(str,1,len,digDEVw.tfp);	
	return(0);
}

/* draw a filled rectangle */
int wp_fillrect(bundle,x1,y1,x2,y2)
int	bundle;
int	x1,y1,x2,y2;
{
	/* sort out colours */
	wp_colour(bundle);

	/* draw filled box */
	return(0);
}

/* draw a circle */
int wp_circle(bundle,xc,yc,r)
int	bundle;
int	xc,yc,r;
{
	/* sort out colours */
	wp_colour(bundle);

	/* draw a circle */
	return(0);
}

/* draw a filled circle */
int wp_fcircle(bundle,xc,yc,r)
int	bundle;
int	xc,yc,r;
{
	/* sort out colours */
	wp_colour(bundle);

	/* draw a filled circle */
	return(0);
}

/* grey level bitmap control */
static	wpg_bitmap2	wpgbitmap;
static	long		wpgbitmapstart;
static 	int		wpgbitmapwidth;

static int wp_gscale_end()
{
	long	wpgbitmapend;
	long	len;

	if (wpgbitmapwidth > 0) {
		wpgbitmapend = ftell(digDEVw.tfp);
		fseek(digDEVw.tfp,wpgbitmapstart,0);
		len = wpgbitmapend - wpgbitmapstart - 6;

		wpgbitmap.wpgsizehi = (unsigned short)(0x8000 | (len >> 16));
		wpgbitmap.wpgsizelo = len & 0xFFFF;
		wpgbitmap.wpgiheight = wpgbitmapwidth;

		wpgbitmap.wpgxl += (wpgbitmapwidth/2 - wpgbitmap.wpgiwidth/2) << 4;
		wpgbitmap.wpgyb += (wpgbitmap.wpgiwidth/2 - wpgbitmapwidth/2) << 4;
		wpgbitmap.wpgxr = wpgbitmap.wpgxl + (wpgbitmap.wpgiwidth << 4) + 8;
		wpgbitmap.wpgyt = wpgbitmap.wpgyb + (wpgbitmapwidth << 4) + 8;

		if (fwrite((char *)&wpgbitmap,sizeof(wpg_bitmap2),1,digDEVw.tfp) != 1)
			error("write error on temporary file");

		fseek(digDEVw.tfp,wpgbitmapend,0);
		wpgbitmapwidth=0;
	}
	return(0);
}

/* grey-level display */
int wp_gscale(x,y,height,buff)
int	x,y;
int	height;
unsigned char	*buff;
{
	int	i,len;
	static int	oldy = -1;

	/* check if new grey scale */
	if (y != oldy)
		wp_gscale_end();
	oldy = y;

	/* start of new bitmap */
	if (wpgbitmapwidth==0) {
		/* WPG bitmap record */
		wpgbitmapstart = ftell(digDEVw.tfp);
		wpgbitmap.wpgcode = WPG_BITMAP2;
		wpgbitmap.wpgsizeflag = 0xFF;
		wpgbitmap.wpgsizehi = 0;		/* filled in later */
		wpgbitmap.wpgsizelo = 0;		/* filled in later */
		wpgbitmap.wpgangle = 90;		/* turn so left to right */
		wpgbitmap.wpgxl = (x << 2);
		wpgbitmap.wpgyb = (y << 2);
		wpgbitmap.wpgxr = 0;			/* filled in later */
		wpgbitmap.wpgyt = 0;			/* filled in later */
		wpgbitmap.wpgiwidth = height;
		wpgbitmap.wpgiheight = 0;		/* filled in later */
		wpgbitmap.wpgbpp = 8;
		wpgbitmap.wpgxdpi = 75;			/* assume 4x4 maxel */
		wpgbitmap.wpgydpi = 75;
		if (fwrite((char *)&wpgbitmap,sizeof(wpg_bitmap2),1,digDEVw.tfp) != 1)
			error("write error on temporary file");
	}

	/* write bitmap data */
	buff += height - 1;
	while (height > 0) {
		len = MIN(height,127);
		putc(len,digDEVw.tfp);
		for (i=0;i<len;i++,buff--)
			putc(31 - *buff,digDEVw.tfp);
		height -= len;
	}
	wpgbitmapwidth++;
	return(0);
}

int wp_pause()
{
	return(0);
}
#endif
