/* filter -- General Purpose IIR Digital Filter Routines */

/* M.A.Huckvale -- November 1991 */

/* Code borrowed largely and loosely from:
		Cappellini, Constantinides, Emillani,
		"Digital Filters and their Applications"
		Academic Press, 1976
*/

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>
#include <malloc.h>
#include <memory.h>
#include "sfsfilter.h"

#ifndef PI
#define PI	3.141592653589
#endif
#define MAX(x,y) (((x)>(y))?(x):(y))

static	complex	pol[FILTER_MAX_ORDER];
static	double a[FILTER_MAX_ORDER][5],b[FILTER_MAX_ORDER][5];

/* do filter transformation */
static int	filter_transform(nsection,fp,f1,f2,ftype)
int	nsection;
double	fp;
double	f1;
double	f2;
int	ftype;
{
	double	al,a0,a1,a2,b0,b1,b2;
	int	ncoeff;
	double	cappa;
	double	an[3][5],fina[3],finb[3];
	int	i,j,k;

	switch (ftype) {
	case FILTER_LOW_PASS:
		al = sin(PI*(fp-f1))/sin(PI*(fp+f1));
		a0 = -al;
		a1 = 1.0;
		a2 = 0.0;
		b0 = a1;
		b1 = a0;
		b2 = 0.0;
		ncoeff=3;
		break;
	case FILTER_HIGH_PASS:
		al = -cos(PI*(fp+f1))/cos(PI*(fp-f1));
		a0 = -al;
		a1 = -1.0;
		a2 = 0.0;
		b0 = -a1;
		b1 = -a0;
		b2 = 0.0;
		ncoeff = 3;
		break;
	case FILTER_BAND_PASS:
		al = cos(PI*(f2+f1))/cos(PI*(f2-f1));
		cappa = tan(PI*fp)/tan(PI*(f2-f1));
		a0 = -(cappa-1.0)/(cappa+1.0);
		a1 = 2.0*al*cappa/(cappa+1.0);
		a2 = -1.0;
		b0 = -a2;
		b1 = -a1;
		b2 = -a0;
		ncoeff = 5;
		break;
	case FILTER_BAND_STOP:
		al = cos(PI*(f2+f1))/cos(PI*(f2-f1));
		cappa = tan(PI*fp)*tan(PI*(f2-f1));
		a0 = (1.0-cappa)/(cappa+1.0);
		a1 = -2.0*al/(cappa+1.0);
		a2 = 1.0;
		b0 = a2;
		b1 = a1;
		b2 = a0;
		ncoeff = 5;
		break;
	default:
		fprintf(stderr,"filter: unknown filter type: %d\n",ftype);
		return(3);
	}

	an[0][0] = b0*b0;
	an[0][1] = 2.0*b0*b1;
	an[0][2] = b1*b1 + 2.0*b0*b2;
	an[0][3] = 2.0*b1*b2;
	an[0][4] = b2*b2;

	an[1][0] = a0*b0;
	an[1][1] = a0*b1+a1*b0;
	an[1][2] = a0*b2 + a1*b1 + a2*b0;
	an[1][3] = a1*b2 + a2*b1;
	an[1][4] = a2*b2;

	an[2][0] = a0*a0;
	an[2][1] = 2.0 * a0 * a1;
	an[2][2] = a1*a1 + 2.0*a0*a2;
	an[2][3] = 2.0*a1*a2;
	an[2][4] = a2*a2;

	for (i=0;i<nsection;i++) {
		for (j=0;j<3;j++) {
			fina[j] = a[i][j];
			finb[j] = b[i][j];
		}
		for (j=0;j<5;j++) {
			a[i][j] = 0.0;
			b[i][j] = 0.0;
			for (k=0;k<3;k++) {
				a[i][j] += an[k][j] * fina[k];
				b[i][j] += an[k][j] * finb[k];
			}
		}
	}
	for (i=0;i<nsection;i++) {
		for (j=1;j<ncoeff;j++) {
			a[i][j] /= b[i][0];
			b[i][j] /= b[i][0];
		}
		a[i][0] /= b[i][0];
		b[i][0] = 1.0;
	}


	return(ncoeff);
}

/* filter design program */
FILTER * filter_design(ftype,forder,flow,fhigh,fsamp)
int	ftype;
int	forder;
double	flow;
double	fhigh;
double	fsamp;
{
	double		ftrans;		/* freq, prior to transform */
	double		fca;
	int		i,j;
	FILTER		*filter;
	int		nsection;
	int		ncoeff;

	switch (ftype) {
	case FILTER_LOW_PASS:
		ftrans = flow/fsamp;
		break;
	case FILTER_HIGH_PASS:
		ftrans = (fsamp-flow)/fsamp;
		break;
	case FILTER_BAND_PASS:
		ftrans = (fhigh-flow)/fsamp;
		break;
	case FILTER_BAND_STOP:
		ftrans = (fhigh-flow)/fsamp;
		break;
	default:
		fprintf(stderr,"filter: unknown filter type: %d\n",ftype);
		return(NULL);
	}

	if (forder > FILTER_MAX_ORDER) {
		fprintf(stderr,"filter: order of filter too large\n");
		return(NULL);
	}
	if (forder & 1) {
		fprintf(stderr,"filter: filter order must be even\n");
		return(NULL);
	}

	/* get filter corner frequency */
	fca = tan(PI*ftrans);

	/* determine pole and zero locations of prototype */
	nsection = forder/2;
	for (i=0;i<nsection;i++) {
		double	tetap;
		tetap = PI * (2.0 * i + 1) / (2.0 * forder);
		pol[i].re = -fca * cos(tetap);
		pol[i].im = fca * sin(tetap);
	}
	for (i=0;i<nsection;i++) {
		double	p1,p2,p3;
		p1 = pol[i].im * pol[i].im;
		p2 = (1.0 - pol[i].re) * (1.0 - pol[i].re);
		p3 = pol[i].re * pol[i].re;
		pol[i].re = (1.0 - p3 - p1)/(p2+p1);
		pol[i].im = (2.0 * pol[i].im)/(p2+p1);
#ifdef EMO
		printf("Pole at %g +/- %gi\n",pol[i].re,pol[i].im);
#endif
	}

	/* calculate coefficients from pole/zero spec */
	for (i=0;i<nsection;i++) {
		b[i][0] = 1.0;
		b[i][1] = -2.0 * pol[i].re;
		b[i][2] = pol[i].re * pol[i].re + pol[i].im * pol[i].im;
	}
	for (i=0;i<nsection;i++) {
		double	tot;
		tot = 4.0/(b[i][0] + b[i][1] + b[i][2]);
		a[i][0] = 1.0/tot;
		a[i][1] = 2.0/tot;
		a[i][2] = 1.0/tot;
	}

	if (ftype != FILTER_LOW_PASS)
		/* transform low-pass, using a[],b[] global coeff */
		ncoeff = filter_transform(nsection,ftrans,flow/fsamp,fhigh/fsamp,ftype);
	else
		ncoeff = 3;

	/* build FILTER structure */
	if ((filter=(FILTER *)calloc(1,sizeof(FILTER)))==NULL) {
		fprintf(stderr,"filter: out of memory\n");
		return(NULL);
	}
	filter->nsection = nsection;
	if ((filter->section=(FILTER_SECTION *)calloc(nsection,sizeof(FILTER_SECTION)))==NULL) {
		fprintf(stderr,"filter: out of memory\n");
		return(NULL);
	}
	for (i=0;i<nsection;i++) {
		filter->section[i].ncoeff = ncoeff;
		if ((filter->section[i].acoeff=(double *)calloc(ncoeff,sizeof(double)))==NULL) {
			fprintf(stderr,"filter: out of memory\n");
			return(NULL);
		}
		if ((filter->section[i].bcoeff=(double *)calloc(ncoeff,sizeof(double)))==NULL) {
			fprintf(stderr,"filter: out of memory\n");
			return(NULL);
		}
		if ((filter->section[i].memory=(double *)calloc(ncoeff+1,sizeof(double)))==NULL) {
			fprintf(stderr,"filter: out of memory\n");
			return(NULL);
		}
	}
	for (i=0;i<nsection;i++) {
		for (j=0;j<ncoeff;j++) {
			filter->section[i].acoeff[j] = (double)a[i][j];
			filter->section[i].bcoeff[j] = (double)b[i][j];
		}
	}

	/* return filter */
	return(filter);
}

/* build a simple resonator */
FILTER *filter_resonator(double cf,double bw,double fsamp)
{
	FILTER		*filter;
	int			i;
	double		wf,wb,r,gain;

	/* build FILTER structure */
	if ((filter=(FILTER *)calloc(1,sizeof(FILTER)))==NULL) {
		fprintf(stderr,"filter: out of memory\n");
		return(NULL);
	}
	filter->nsection = 1;
	if ((filter->section=(FILTER_SECTION *)calloc(filter->nsection,sizeof(FILTER_SECTION)))==NULL) {
		fprintf(stderr,"filter: out of memory\n");
		return(NULL);
	}
	for (i=0;i<filter->nsection;i++) {
		filter->section[i].ncoeff = 3;
		if ((filter->section[i].acoeff=(double *)calloc(3,sizeof(double)))==NULL) {
			fprintf(stderr,"filter: out of memory\n");
			return(NULL);
		}
		if ((filter->section[i].bcoeff=(double *)calloc(3,sizeof(double)))==NULL) {
			fprintf(stderr,"filter: out of memory\n");
			return(NULL);
		}
		if ((filter->section[i].memory=(double *)calloc(3+1,sizeof(double)))==NULL) {
			fprintf(stderr,"filter: out of memory\n");
			return(NULL);
		}
	}


	wf = 2*PI*cf/fsamp;
	wb = 2*PI*bw/fsamp;
	r = 1.0 - wb/2;

	filter->section[0].acoeff[0] = 1.0;
	filter->section[0].acoeff[1] = 0.0;
	filter->section[0].acoeff[2] = -1.0;
	filter->section[0].bcoeff[0] = 1.0;
	filter->section[0].bcoeff[1] = -2.0 * r * cos(wf);
	filter->section[0].bcoeff[2] = r * r;

	/* set gain at CF to unity */
	gain = filter_response(filter,cf,fsamp);
	filter->section[0].acoeff[0] /= gain;
	filter->section[0].acoeff[2] /= gain;

	return(filter);
}

/* free up a filter structure */
void filter_free(flt)
FILTER	*flt;
{
	int	i;

	for (i=0;i<flt->nsection;i++) {
		free(flt->section[i].acoeff);
		free(flt->section[i].bcoeff);
		free(flt->section[i].memory);
	}
	free(flt->section);
	free(flt);
}

/* zero filter memory */
void filter_clear(flt)
FILTER *flt;
{
	int	i;

	for (i=0;i<flt->nsection;i++)
		memset(flt->section[i].memory,0,
			sizeof(double)*(flt->section[i].ncoeff+1));
}

/* filter a sample */
float filter_sample(flt,y)
FILTER	*flt;
double	y;
{
	register int 	i,j;
	FILTER_SECTION	*sct;
	int		mm,mmm;

	for (i=0;i<flt->nsection;i++) {
		sct = &(flt->section[i]);
		mm = sct->ncoeff-1;
		sct->memory[mm] = (double)y;
		y = (double)0.0;
		for (j=0;j<sct->ncoeff-1;j++) {
			mmm = mm - j;
			sct->memory[mm] -= sct->bcoeff[mmm]*
						sct->memory[j];
			y += sct->acoeff[mmm] * sct->memory[j];
			sct->memory[j] = sct->memory[j+1];
		}
		y += sct->acoeff[0] * sct->memory[mm];
	}
	return((float)y);
}

/* filter a signal */
double filter_signal(flt,isp,osp,len)
FILTER	*flt;
short	*isp;
short	*osp;
int	len;
{
	register int 	i,j;
	FILTER_SECTION	*sct;
	double		y;
	int		mm,mmm;
	double		max=0;

	while (len-- > 0) {
		y = *isp++;
		for (i=0;i<flt->nsection;i++) {
			sct = &(flt->section[i]);
			mm = sct->ncoeff-1;
			sct->memory[mm] = y;
			y = (double)0.0;
			for (j=0;j<sct->ncoeff-1;j++) {
				mmm = mm - j;
				sct->memory[mm] -= sct->bcoeff[mmm]*
							sct->memory[j];
				y += sct->acoeff[mmm] * sct->memory[j];
				sct->memory[j] = sct->memory[j+1];
			}
			y += sct->acoeff[0] * sct->memory[mm];
		}
		if (y > max) max = y;
		if (y < -max) max = -y;
		*osp++ = (short)y;
	}
	return(max);
}

/* complex multiply */
static void CMLT(d,a,b)
complex	*d,*a,*b;
{
	complex t;
	t.re = a->re*b->re - a->im*b->im;
	t.im = a->re*b->im + a->im*b->re;
	*d = t;
}

/* calculate response of filter at given frequency */
double	filter_response(flt,f,fsamp)
FILTER	*flt;
double	f;
double	fsamp;
{
	int		i,j;
	double		omega;
	double		gain;
	complex		wj[5];
	FILTER_SECTION	*sct;
	complex		h;
	complex		an,ad;
	complex		t;
	double		r1,r2;

	omega = 2.0 * PI * f/fsamp;
	wj[0].re = 1.0;
	wj[0].im = 0.0;
	wj[1].re = cos(omega);
	wj[1].im = sin(omega);
	CMLT(&wj[2],&wj[1],&wj[1]);
	CMLT(&wj[3],&wj[2],&wj[1]);
	CMLT(&wj[4],&wj[2],&wj[2]);

	h.re = 1.0;
	h.im = 0.0;
	for (i=0;i<flt->nsection;i++) {
		sct = &(flt->section[i]);
		an.re = 0;
		an.im = 0;
		ad.re = 0;
		ad.im = 0;
		for (j=0;j<sct->ncoeff;j++) {
			an.re += wj[j].re * sct->acoeff[j];
			an.im += wj[j].im * sct->acoeff[j];
			ad.re += wj[j].re * sct->bcoeff[j];
			ad.im += wj[j].im * sct->bcoeff[j];
		}

		r1 = sqrt(an.re*an.re+an.im*an.im);
		r2 = sqrt(ad.re*ad.re+ad.im*ad.im);

		if ((r1==0.0) || (r2==0.0)) {
			t.re = 0.0;
			t.im = 0.0;
		}
		else {
			t.re = (r1/r2) * ((an.im/r1)*(ad.im/r2) + (an.re/r1)*(ad.re/r2));
			t.im = (r1/r2) * ((an.im/r1)*(ad.re/r2) - (an.re/r1)*(ad.im/r2));
		}

		gain = h.re*t.re - h.im*t.im;
		h.im = h.re*t.im + h.im*t.re;
		h.re = gain;
	}
	gain = sqrt(h.re*h.re + h.im*h.im);
	return(gain);
}


#ifdef EMO

#define NPOINT	200
static short sig[NPOINT]={10000};

main()
{
	int	ftype,forder;
	float	fsamp,f1,f2;
	FILTER	*flt;
	FILTER_SECTION *sct;
	int	i,j;
	FILE	*op;
	float	f,g;

	printf("Enter type (1-lowpass,2-highpass,3-bandpass,4-bandstop) : ");
	scanf("%d",&ftype);

	printf("Enter sampling frequency :");
	scanf("%f",&fsamp);

	printf("Enter lower band edge :");
	scanf("%f",&f1);

	if (ftype > 2) {
		printf("Enter upper band edge :");
		scanf("%f",&f2);
	}
	else
		f2 = f1;

	printf("Enter order of filter: ");
	scanf("%d",&forder);

	if ((flt = filter_design(ftype,forder,f1,f2,fsamp))==NULL)
		exit(1);

	printf("Numerators:\n");
	for (i=0;i<flt->nsection;i++) {
		sct = &(flt->section[i]);
		for (j=0;j<sct->ncoeff;j++)
			printf("%10g  ",sct->acoeff[j]);
		printf("\n");
	}
	printf("Denominators:\n");
	for (i=0;i<flt->nsection;i++) {
		sct = &(flt->section[i]);
		for (j=0;j<sct->ncoeff;j++)
			printf("%10g  ",sct->bcoeff[j]);
		printf("\n");
	}

	printf("Filtered pulse=");
	filter_signal(flt,sig,sig,NPOINT);
	for (i=0;i<NPOINT;i++) printf("%d ",sig[i]);
	printf("\n");

	/* calculate frequency response */
	op=fopen("filter.tf","w");
	for (f=5.0;f<fsamp/2;f+=5.0) {
		g=filter_response(flt,f,fsamp);
		fprintf(op,"%g\n",20.0*log10(MAX(g,1e-10)));
		printf("%d\r",(int)f);
	}
	fclose(op);

	exit(0);
}
#endif
