/* lpca.c -- LPC Autocorrelation routine */

/*
 * (c) 1994 Mark Huckvale
 * Search Systems Ltd
 * 52-54 High Holborn
 * London WC1E 6BT
 * Tel. 071-430 1685
 */

#include <stdio.h>
#include <malloc.h>
#include <math.h>
#include "complex.h"
#include "lsys.h"
#include "lpca.h"

int LPCAutocorrelation(x,n,order,ltis,pe)
float *x;		/* windowed input signal */
int n;			/* # samples in x */
int order;		/* predictor order required */
LTIState *ltis;		/* returned predictor coefficients */
double *pe;		/* returned predictor error */
			/* returns 0=OK, 1=zero power, 2=fail */
{
	double	r[MAXORDER+1];		/* autocorrelations */
	double	pc[MAXORDER+1];		/* predictor coefficients */
	double	ai,aj,akk;		/* temporary coefficient values */
	double	sum;			/* accumulator */
	int	i,k;			/* loop counters */
	int	retcode=0;

	/* check */
	if (order > MAXORDER) {
		order = MAXORDER;
		retcode=2;
	}

	/* compute autocorrelations */
	for (i=0;i<=order;i++) {
		sum = 0;
		for (k=0;k<(n-i);k++)
			sum += x[k] * x[k+i];
		r[i] = sum;
	}

	/* compute predictor coefficients */
	if (r[0] == 0)
		/* no signal ! */
		retcode = 1;
	else {
		*pe = r[0];
		pc[0] = 1.0;
		for (k=1;k<=order;k++) {
			sum = 0;
			for (i=1;i<=k;i++)
				sum -= pc[k-i] * r[i];
			akk = sum/(*pe);
			/* new predictor coefficients */
			pc[k] = akk;
			for (i=1;i<=(k/2);i++) {
				ai = pc[i];
				aj = pc[k-i];
				pc[i] = ai + akk * aj;
				pc[k-i] = aj + akk * ai;
			}
			/* new prediction error */
			*pe = *pe * (1.0 - akk*akk);
			if (*pe <= 0)
				/* negative/zero error ! */
				retcode = 2;
		}
	}

	/* copy into LTI System */
	LTISystemClear(ltis);
	ltis->na = 0;
	ltis->a[0] = 1.0;
	ltis->nb = order;
	for (i=1;i<=order;i++)
		ltis->b[i] = pc[i];

	return(retcode);
}

int LPCBurg(float *x,int len,int order,LTIState *ltis,double *pe)
{
	int	i,m;
	float	*f,*b,*fold;
	double	*a,*aold;
	double	sum;
	double	err,crit,critold;
	double	k,num,den;

	/* initialise error */
	for (sum=0,i=0;i<len;i++) sum+=x[i]*x[i];
	err = sum;
	critold=2*err/len;

	/* no signal !*/
	if (err==0) return(0);

	/* get buffers */
	f = (float *)calloc(len,sizeof(float));
	fold = (float *)calloc(len,sizeof(float));
	b = (float *)calloc(len,sizeof(float));
	a = (double *)calloc(order+1,sizeof(double));
	aold = (double *)calloc(order+1,sizeof(double));

	/* copy signal into forward and backward buffers */
	for (i=0;i<len-1;i++) {
		f[i] = x[i+1];
		b[i] = x[i];
	}

	/* calculate each coefficient in turn */
	for (m=1;m<=order;m++) {

		/* calculate the mth reflection coefficient */
		for (sum=0,i=0;i<len-m;i++) sum+=f[i]*b[i];
		num = -2*sum;
		for (sum=0,i=0;i<len-m;i++) sum+=f[i]*f[i]+b[i]*b[i];
		den = sum;
		k = num/den;

		/* update the forward and backward prediction errors */
		for (i=0;i<len-m;i++) fold[i]=f[i];
		for (i=0;i<len-m-1;i++) f[i]=(float)(f[i+1]+k*b[i+1]);
		for (i=0;i<len-m-1;i++) b[i]=(float)(b[i]+k*fold[i]);

		/* update the total error */
		err = (1-k*k)*err;

		/* calculate approximate corrected Kullback information criterion */
		crit = log(err/len) + ((double)m/len)/(len-m) + (3-(m+2.0)/len)*(m+1)/(len-m-2);
		if (crit > critold) {
			/* no worthwhile decrease in error */
			order=m-1;
			break;
		}
		critold=crit;

   		/* generate next coefficient */
		for (i=1;i<m;i++) aold[i]=a[i];
		for (i=1;i<m;i++) a[i]=aold[i] + k*aold[m-i];
		a[m] = k;

	}

	/* copy into LTI System */
	LTISystemClear(ltis);
	ltis->na = 0;
	ltis->a[0] = 1.0;
	ltis->nb = order;
	ltis->b[0] = 1.0;
	for (i=1;i<=order;i++) ltis->b[i] = a[i];
	*pe = err;

	/* free buffers */
	free(f);
	free(fold);
	free(b);
	free(a);
	free(aold);

	return(order);
}



#ifdef EMO
#define WINDOWSIZE 32
#define NCOEFF 4
#define C1	0.16
#define C2	-0.12
#define C3	0.08
#define C4	-0.04
main()
{
	float	iwin[WINDOWSIZE+1];
	LTIState osys;
	double	res;
	int	i;

	/* make a test signal from recursive filter */
	iwin[1] = 0.75;		/* put in pulse of power sqr(0.75) */
	iwin[2] = -C1 * iwin[1];
	iwin[3] = -C1 * iwin[2] - C2 * iwin[1];
	iwin[4] = -C1 * iwin[3] - C2 * iwin[2] - C3 * iwin[1];
	for (i=5;i<=WINDOWSIZE;i++)
		iwin[i] = -C1 * iwin[i-1] - C2 * iwin[i-2] -
				C3 * iwin[i-3] - C4 * iwin[i-4];

	/* do LPC analysis */
	if (LPCAutocorrelation(iwin+1,WINDOWSIZE,NCOEFF,&osys,&res)==0) {
		printf("Predictor coefficients:\n");
		for (i=1;i<=NCOEFF;i++)
			printf("%d = %g\n",i,osys.b[i]);
		printf("Residual Power=%g\n",res);
	}

}
#endif
