/* replay32 -- console replay using WIN32 API */

#include "SFSCONFG.h"
#ifdef DAC_WIN32

#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <windows.h>
#include <mmsystem.h>
#include <sys/types.h>
#include <sys/stat.h>

/* replay done flag */
volatile int replaydone;

/* replay callback function */
static void CALLBACK waveOutProc(HWAVEOUT hwo,UINT uMsg,DWORD dwInstance,DWORD dwParam1,DWORD dwParam2)
{
	replaydone=1;
}

/* replay function */
int win32replay(byte *buf,int len,int smr,int nbs,int nch)
{
	HGLOBAL  	hWaveOut;	/* waveform device handle */
	PCMWAVEFORMAT 	pcm;	/* waveform device configuration */
	HGLOBAL		whdr;
	LPWAVEHDR	lpwhdr;

	/* set up device */
	pcm.wf.wFormatTag	= WAVE_FORMAT_PCM;
	pcm.wf.nChannels	= nch;
	pcm.wf.nSamplesPerSec	= smr;
	pcm.wf.nAvgBytesPerSec	= smr * nbs * nch;
	pcm.wf.nBlockAlign	= nbs * nch;
	pcm.wBitsPerSample	= 8 * nbs;

	/* query the replay devices */
	if (waveOutOpen(NULL,(UINT)WAVE_MAPPER,(LPCWAVEFORMATEX)&pcm,0,0,WAVE_FORMAT_QUERY)) {
		/* not supported */
		fprintf(stderr,"Unsupported waveform format/rate");
		return(-1);
	}
		
	/* open device for real */
	if (waveOutOpen(&hWaveOut,(UINT)WAVE_MAPPER,(LPCWAVEFORMATEX)&pcm,(DWORD)waveOutProc,0,CALLBACK_FUNCTION)) {
		/* some kind of error on open - in use? */
		fprintf(stderr,"Unable to open waveform output device\n");
		return(-1);
	}

	/* reset everything */
	waveOutReset(hWaveOut);

	/* malloc up a wave header */
	whdr = GlobalAlloc(GHND|GMEM_SHARE,sizeof(WAVEHDR));
	lpwhdr = (LPWAVEHDR)GlobalLock(whdr);

	/* point it to the data to replay */
	lpwhdr->lpData			= buf;
	lpwhdr->dwBufferLength  = len;
	lpwhdr->dwBytesRecorded = 0L;
	lpwhdr->dwUser			= 0L;
	lpwhdr->dwFlags			= 0L;
	lpwhdr->dwLoops			= 1L;
	lpwhdr->lpNext			= NULL;
	lpwhdr->reserved		= 0L;

	/* prepare it */
	waveOutPrepareHeader(hWaveOut,lpwhdr,sizeof(WAVEHDR));
	replaydone=0;

	/* write it */
	if (waveOutWrite(hWaveOut,lpwhdr,sizeof(WAVEHDR))!=0) {
		/* argh! */
		waveOutUnprepareHeader(hWaveOut,lpwhdr,sizeof(WAVEHDR));
		fprintf(stderr,"Error on write to wave out device\n");
		waveOutClose(hWaveOut);
		GlobalUnlock(whdr);
		GlobalFree(whdr);
		return(-1);
	}

	/* wait for done */
	while (!replaydone) /* loop */;

	/* close down */
	waveOutClose(hWaveOut);
	GlobalUnlock(whdr);
	GlobalFree(whdr);
	return(0);

}

#ifdef EMO
/* global parameters */
int		srate;			/* sampling rate */
int		nbytesamp;		/* num bytes per samp */
int		nchan;			/* number of channels */
char	fname[256];		/* filename */
byte	*sbuf;			/* sample buffer */

/* main program */
void main(int argc,char **argv)
{
	struct stat st;
	FILE	*ip;

	if (argc!=5) {
		fprintf(stderr,"Usage: replay32 rate nbytesamp nchan file\n");
		exit(1);
	}
	srate = atoi(argv[1]);
	nbytesamp = atoi(argv[2]);
	nchan = atoi(argv[3]);
	strcpy(fname,argv[4]);

	/* get file size */
	if (stat(fname,&st)!=0) {
		fprintf(stderr,"Could not stat '%s'\n",fname);
		exit(1);
	}

	/* report on progress so far */
	printf("Replaying %d bytes at %d samp/sec, %d bytes/samp, %d channels\n",
			st.st_size,srate,nbytesamp,nchan);

	/* malloc up a buffer and load data into memory */
	if ((sbuf=(byte *)malloc(st.st_size))==NULL) {
		fprintf(stderr,"Could not get buffer\n");
		exit(1);
	}

	ip = fopen(fname,"rb");
	if (fread(sbuf,1,st.st_size,ip)!=st.st_size) {
		fprintf(stderr,"Read error on '%s'\n",fname);
		exit(1);
	}
	fclose(ip);

	/* call replay function */
	win32replay(sbuf,st.st_size,srate,nbytesamp,nchan);

	/* that's all folks */
	free(sbuf);
	exit(0);
}
#endif
#endif
