/* dicode -- convert coefficients items to display items (SFS format) */

/* M.A.Huckvale - November 1987 */

/* version 1.0
	- from codi
*/
/* version 1.1 - May 1988
	- don't interfere with item.params, just add "sfsformat"
*/
/* version 1.2 - December 1990
	- scale maximum dynamic range available
*/
/* version 1.3 - August 2012
	- change to 256 grey levels
*/

/*-------------------------------------------------------------------*/
/**MAN
.TH DICODE SFS1 UCL
.SH NAME
dicode - convert coefficient items to display items
.SH SYNOPSIS
.B dicode
(-i item) (-d dbrange) (-n numpixel) (-g greylevels) file
.SH DESCRIPTION
.I dicode
performs the rescaling of spectral coefficients so that
they are in a compact and easily displayed format.
The routine reads in the spectral coefficient data and scales it.
The 16-level grey scale is found by linearly scaling over
the dynamic range from the maximum to
.I dbrange
below the maximum.
Values below the threshold are set white.
.PP
.TP 14
.B Options
.TP 11
.B -I
Identify the program version.
.TP 11
.BI -d dbrange
Specify dbrange below maximum (default 50.0).  If dbrange is zero
or negative then maximum available range will be used.
.TP 11
.BI -n numpixel
Specify height of output display in pixels (default is number of
input coefficients). Must be less than or equal to number of input
coefficients.
.TP 11
.BI -g greylevels
Specify number of grey levels to use. Range 2-256. Default 256.
.TP 11
.BI -i item
Select input item number.
.SH INPUT ITEMS
.IP 11.xx 11
Any coefficient item
.SH OUTPUT ITEMS
.IP 9 11
Display item
.SH VERSION/AUTHOR
1.2 - Mark Huckvale
*/
/*---------------------------------------------------------------*/

/* program name and version */
#define	PROGNAME "dicode"
#define PROGVERS "1.2s"
char	*progname = PROGNAME;

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "sfs.h"		/* database filing system structures */

/* global data */
struct item_header 	coitem;			/* coefficient item header  */
struct co_rec		*co;			/* coefficient record */
struct item_header 	diitem;			/* display item header */
struct di_rec		*di;			/* display record */
float		        dbrange=(float)50.0;		/* energy range */
int			numpixel;		/* # output pixels */
char			filename[SFSMAXFILENAME];	/* dbase file name */
int			numgrey=256;
int			debug=0;

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;			/* option index */
	extern char	*optarg;		/* option argument ptr */
	int		errflg = 0;		/* option error flag */
	int		it;			/* item type */
	char		*ty;			/* sub-type  */
	int		c;			/* option switch */

	/* processing variables */
	char		*cotype="0";		/* coefficient sub-type */
	int		nocoeff;		/* # input coefficients */
	register float	*f;			/* float pointer */
	float		minval,maxval;		/* coeff values */
	register float	scale,energy;		/* scale factor + energy at pixel */
	register int	i,j;
	int		fid,ofid;		/* file descriptors */
	float		imult,ind1,ind2,sum;	/* energy averaging */
	int		in1,in2;
	char	msg[16];
	int		pixel;

	/* external routines */
	double		param(),atof();

	/* decode switches */
	while ( (c = getopt(argc,argv,"Id:i:n:g:D")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Code spectral coefficients for display V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* specific item */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == CO_TYPE)
					cotype = ty;
				else
					error("unsuitable item specifier %s",optarg);
			}
			else
				error("illegal item specifier %s",optarg);
			break;
		case 'd' :	/* specify dbrange */
			dbrange = (float)atof(optarg);
			break;
		case 'n' :	/* specify num output coeff */
			numpixel = atoi(optarg);
			break;
		case 'g' :	/* number of grey levels */
			numgrey = atoi(optarg);
			if (numgrey < 2) numgrey=2;
			if (numgrey > 256) numgrey=256;
			break;
		case 'D' :	/* debug - produce grey bar as output */
			debug=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	/* check command line */
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-d dbrange) (-n numpixel) (-g greylevels) file",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open input file */
	if ((fid=sfsopen(filename,"w",NULL)) < 0) {
		if (fid==-1)
			error("cannot find file %s",filename);
		else
			error("access error on %s",filename);
	}

	/* find coefficient item */
	if (!sfsitem(fid,CO_TYPE,cotype,&coitem))
		error("cannot find input coefficients in %s",filename);

	/* get input buffer */
	if ((co=(struct co_rec *)sfsbuffer(&coitem,1))==NULL)
		error("could not get memory buffer",NULL);
	nocoeff = (coitem.datasize*coitem.framesize-sfsstruct[CO_TYPE])/sizeof(float);

	/* find maximum value in coefficient record */
	maxval = (float)-10E6;
	minval = (float)10E6;
	for (i=0;sfsread(fid,i,1,co);i++) {
		f = co->data;
		for (j=0;j<nocoeff;j++,f++) {
			maxval = (maxval > *f) ? maxval : *f;
			minval = (minval < *f) ? minval : *f;
		}
	}
	if (maxval==minval)
		error("CO data has zero dynamic range!");
	if (numpixel==0) numpixel = nocoeff;

	/* create display item header */
	sfsheader(&diitem,DI_TYPE,0,1,numpixel+sfsstruct[DI_TYPE],
		coitem.frameduration,coitem.offset,coitem.windowsize,
		coitem.overlap,coitem.lxsync);
	sprintf(msg,"ngrey=%d",numgrey-1);
	strncpy(diitem.params,coitem.params,sizeof(diitem.params)-strlen(msg)-1);
	if (diitem.params[0] != '\0') strcat(diitem.params,",");
	strcat(diitem.params,msg);
	if (dbrange > 0)
		sprintf(diitem.history,"%s(%d.%02d;dbr=%.2f,nump=%d,ngrey=%d)",PROGNAME,CO_TYPE,coitem.subtype,dbrange,numpixel,numgrey);
	else
		sprintf(diitem.history,"%s(%d.%02d;dbr=%.2f,nump=%d,ngrey=%d)",PROGNAME,CO_TYPE,coitem.subtype,maxval-minval,numpixel,numgrey);

	/* create output channel */
	if ((ofid=sfschannel(filename,&diitem)) < 0)
		error("could not open temporary file",NULL);

	/* get output buffer */
	if ((di=(struct di_rec *)sfsbuffer(&diitem,1))==NULL)
		error("could not get memory buffer",NULL);

	/* copy coefficient data to output */
	if (dbrange > 0)
		minval = maxval - dbrange;
	else
		dbrange = maxval - minval;
	scale = (float)((numgrey-1)/dbrange);
	imult = (float)nocoeff/numpixel;
	for (j=0;sfsread(fid,j,1,co);j++) {
		di->posn=co->posn;
		di->size=co->size;
		for (i=0;i<numpixel;i++) {
			if (numpixel==nocoeff)
				energy=co->data[i];
			else {
				ind1 = i * imult;
				in1  = (int) ind1;
				ind2 = (i+1) * imult;
				in2  = (int) ind2;
				sum  = ((in1+1) - ind1) * co->data[in1];
				in1++;
				while (in1 < in2) sum += co->data[in1++];
				sum += (ind2 - in2) * co->data[in2];
				energy = sum / imult;
			}
			if (debug)
				di->pixel[i] = (256*j)/coitem.numframes;
			else if (energy < minval)
				di->pixel[i]=0;
			else {
				pixel=1 + (int)((energy-minval)*scale);
				if (pixel < 0)
					di->pixel[i] = 0;
				else if (pixel >= numgrey)
					di->pixel[i] = numgrey-1;
				else
					di->pixel[i] = pixel;
			}
		}
		if (!sfswrite(ofid,1,di))
			error("write error on temporary file",NULL);
	}

	/* update file */
	sfsclose(fid);
	if (!sfsupdate(filename))
		error("update error on %s",filename);

	/* that's all folks */
	exit(0);

}

