/* fxanal -- estimate FX by autocorrelation and DP */

/* M.A. Huckvale - University College London */

/* version 1.0 - April 2000 */

/* version 1.1 - June 2000
	- add -f female flag
*/
/* version 1.1a - February
	- change from '-f' to '-h' for 'higher pitch'
*/
/* version 1.2 - November 2003
	- add pitch halving check option
*/
/* version 1.3 - November 2003
	- add improvements to pitch halving check option
*/

/* version 2.0 - December 2003
	- major makeover, adjustments and testing
*/

#undef EBUG

#define PROGNAME "fxanal"
#define PROGVERS "2.0"
char *progname=PROGNAME;

/*--------------------------------------------------------------------------*/
/**MAN
.TH FXANAL SFS1 UCL
.SH NAME
fxanal - estimate fx from sp using autocorrelation and tracking
.SH SYNOPSIS
.B fxanal
(-i item) (-f) (-X) file
.SH DESCRIPTION
.I fxanal
estimates FX from a speech waveform by autocorrelation.
The algorithm
   implemented here is similar to that presented by B. Secrest and
   G. Doddington, "An integrated pitch tracking algorithm for speech
   systems", Proc. ICASSP-83, pp.1352-1355.
.PP
.I Options
and their meanings are:
.TP 11
.B -I
Identify program and exit.
.TP 11
.BI -i item
Select input item number and type.
.TP 11
.B -l
Use LX signal as source.  Some processing is modified for LX signals.
.TP 11
.B -h
Modify default settings to improve the decoding of high-pitched voices.
.B -X
Run extra check for "pitch halving", where the chosen Fx is exactly half
the correct answer.  Only use this if you are having problems with pitch
halving.
.SH INPUT ITEMS
.IP 1.xx 11
Speech waveform.
.IP 2.xx 11
Laryngograph waveform.
.TP 11
.SH OUTPUT ITEMS
.IP 4 11
FX estimate by autocorrelation.
.SH VERSION/AUTHOR
2.0 - Mark Huckvale.
.SH SEE ALSO
fxac(SFS1) fxcep(SFS1) fxrapt(SFS1)
*/
/*--------------------------------------------------------------------------*/

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "sfs.h"		/* header structures */
#include "filter.h"		/* filtering */

/* FX estimation control */
#define	STEP		0.005	/* 5ms step */
#define WINDOW		0.040	/* analysis window = 3 cycles @ MINFX */
#define WINDOW_FEM	0.030	/* analysis window = 3 cycles @ MINFX */
#define ENERGYDIV	300;	/* 50dB */
#define HIFX		500.0	/* highest Fx */
#define LOFX		33.0	/* lowest Fx * 2/3 */
#define HIFX_FEM	1000.0	/* highest Fx */
#define LOFX_FEM	66.0	/* lowest Fx * 2/3 */
#define LOFILT_DEF	20.0	/* high pass frequency */
double LOFILT=LOFILT_DEF;
#define HIFILT_DEF	1200.0	/* low-pass frequency */
double HIFILT=HIFILT_DEF;

#define THRESHOLD_DEF	0.2	/* normalised autocorrelation threshold */
double THRESHOLD=THRESHOLD_DEF;
#define NHYP_DEF	100	/* number of hypotheses per frame */
int NHYP=NHYP_DEF;
#define NORMFX_STDDEV_DEF 0.1	/* normalised Fx standard deviation */
double NORMFX_STDDEV=NORMFX_STDDEV_DEF;
#define VUV_PENALTY_DEF	-1.0	/* V/UV switch penalty = 1 standard deviation */
double VUV_PENALTY=VUV_PENALTY_DEF;

/* global data */
struct item_header 	spitem;		/* item header data */
short			*sp;		/* speech data buffer (1 window) */
float			*fsp,*msp;	/* float speech buffers (1 window) */
struct item_header 	fxitem;		/* item header data */
int			sampfreq;	/* current sampling frequency */
float			*acoeff;	/* autocorrelation function */
float			*fxpeak;	/* Fx peaks */
float			*fxsize;	/* Fx peak sizes */
int			female=0;	/* female settings */
int			checkhalf=0;	/* check for pitch halving */
double		maxenergy;

/* pitch tracking */
struct hyp_rec {
	int	posn;			/* time in frames */
	float	fx;			/* estimated Fx value */
	float	score;			/* current score */
	struct hyp_rec *back;		/* back pointer */
};
struct hyp_rec **hyptab;		/* hypothesis table */
int	hyplen;				/* length of hypothesis table */

/* autocorrelation */
float autoc(float *sp,int len,float *acoeff,int l1,int l2)
{
	register int	i,j;
	int		num;
	float		sum,sumsq1,sumsq2,norm;
	float		*s1,*s2;
	float		mean=0;

	/* zero autocorrelation vector */
	for (i=0;i<len;i++) acoeff[i]=(float)0.0;

	/* for zero delay */
	sum=(float)0.0;
	num=len;
	s1 = sp;
	for (j=0;j<num;j++,s1++) sum += *s1 * *s1;
	acoeff[0] = sum/len;

	/* for each delay in expected freq. range */
	for (i=l1;i<=l2;i++) {
		sumsq1=sumsq2=sum=(float)0.0;
#ifdef OLD
		num = 3*i;	/* up to three possible cycles */
#else
		num = 3*sampfreq;
#endif
		if (num < sampfreq) num=sampfreq;	/* but not less than 10ms */
		if (num > (len-i)) num = len-i;

		s1 = sp;
		s2 = s1 + i;
		for (j=0;j<num;j++) {
			sumsq1 += *s1 * *s1;
			sumsq2 += *s2 * *s2;
			sum += *s1++ * *s2++;
		}
		norm = (float)(sqrt(sumsq1)*sqrt(sumsq2)/num);
		acoeff[i] = (sum/num)/norm;
		mean += acoeff[i];
	}
	return(mean/(l2-l1+1));
}

/* find peaks */
int findpeaks(float *acoeff,int len,int l1,int l2,float *fxpeak,float *fxsize)
{
	register int	i;
	int		imax,npeak=0;

	/* find all peaks */
	imax = 2*l2/3;
	for (i=l1+1;i<imax;i++) {
		if ((acoeff[i-1] < acoeff[i]) && (acoeff[i] > acoeff[i+1]) && (acoeff[i] > THRESHOLD)) {
			fxpeak[npeak] = (float)(100.0*sampfreq/i);
			if (checkhalf && ((3*i/2) <= l2)) {
				fxsize[npeak] = acoeff[i]-0.1*acoeff[3*i/2];
#ifdef EBUG
				printf("fxsize[%.1f]=%g-%g=%g\n",1.0/(i*spitem.frameduration),acoeff[i],0.1*acoeff[3*i/2],fxsize[npeak]);
#endif
			}
			else {
				fxsize[npeak] = acoeff[i];
#ifdef EBUG
				printf("fxsize[%.1f]=%g\n",1.0/(i*spitem.frameduration),fxsize[npeak]);
#endif
			}
			npeak++;
		}
	}
	return(npeak);
}

/* sort peak list */
void sortpeaks(float *fxpeak,float *fxsize,int len)
{
	float	tpeak,tsize;
	int	i,j;

	for (i=1;i<len;i++) {
		j=i;
		tpeak = fxpeak[j];
		tsize = fxsize[j];
		while ((j>0) && (tsize > fxsize[j-1])) {
			fxpeak[j] = fxpeak[j-1];
			fxsize[j] = fxsize[j-1];
			j--;
		}
		fxpeak[j] = tpeak;
		fxsize[j] = tsize;
	}
}

/* zero crossing rate */
float zeroc(short *sp,int len)
{
	register int	i;
	short	last=sp[0];
	short	this;
	int		count=0;

	for (i=1;i<len;i++) {
		this = sp[i];
		if ((last<0)&&(this>=0)) count++;
		last=this;
	}

	return((float)(count/(len*spitem.frameduration)));
}

/* safe log */
double mylog(double val)
{
	if (val < 1.0E-10)
		return(log(1.0E-10));
	else
		return(log(val));
}

/* log likelihood of an Fx - not enabled */
float fxlikelihood(float fx)
{
	float	stddev=70;
	float	mean=120;
	if (female) {
		if (fx==0) fx=mean;
	}
	else {
		if (fx==0) fx=2*mean;
	}
/*	return ((float)(-0.5*((fx-mean)/stddev)*((fx-mean)/stddev)));	*/
	return(0.0);
}

/* log similarity between two Fx values */
float fxsimilarity(float fx1,float fx2)
{
	float stddev=NORMFX_STDDEV;
	float diff;
	float	e,f;
	float	prob;
	if (fx1==fx2)
		diff=0;
	else if ((fx1==0)||(fx2==0))
		diff=stddev;
	else
		diff = 2*(fx1-fx2)/(fx1+fx2);

	e = exp(-0.5*(diff/stddev)*(diff/stddev));
	f = sqrt(2*3.14159*stddev*stddev);
	prob = mylog(0.1*e/f);
/* printf("fxsimilarity(%g,%g)=>%g (%g,%g,%g,%g)\n",fx1,fx2,prob,diff,e,f,e/f); */
	return(prob);
}

/* log cost of skipping one time frame */
float skipcost()
{
	float stddev=NORMFX_STDDEV;
	float diff=stddev;
	float	prob;
	prob = mylog(0.1*exp(-0.5*(diff/stddev)*(diff/stddev))/sqrt(2*3.14159*stddev*stddev));
	return((float)prob);
}

/* add hypothesis to list */
void addhyp(struct hyp_rec hold[],int posn,float fx,float score,struct hyp_rec *back)
{
	int	i;

	/* check if too bad already */
	if (score < hold[NHYP-1].score) return;

	/* shift down those worse */
	i=NHYP-1;
	while ((i > 0) && (score > hold[i-1].score)) {
		memcpy(&hold[i],&hold[i-1],sizeof(struct hyp_rec));
		i--;
	}

	/* insert in right place */
	hold[i].posn = posn;
	hold[i].fx = fx;
	hold[i].score = score;
	hold[i].back = back;
}

/* dynamic programming */
void dptrack()
{
	int	i,j,k;
	struct hyp_rec *hold;
	float	score,basescore,maxscore;
	struct hyp_rec *maxrec;

	hold = (struct  hyp_rec *)calloc(NHYP,sizeof(struct hyp_rec));

	/* initialise first frame */
	for (j=0;j<NHYP;j++) {
		hyptab[0][j].score = 2*(float)mylog(hyptab[0][j].score) + fxlikelihood(hyptab[0][j].fx);
		hyptab[0][j].back = NULL;
	}

	/* do viterbi */
	for (i=1;i<hyplen;i++) {
		/* clear hold vector */
		for (j=0;j<NHYP;j++) hold[j].score = -1000;
#ifdef EBUG
		printf("N%.3f. ",i*STEP);
		for (j=0;j<NHYP;j++) {
			if (hyptab[i][j].score > 0)
				printf("%.1f(%g) ",hyptab[i][j].fx,hyptab[i][j].score);
		}
		printf("\n");
#endif
		/* do search */
		for (j=0;j<NHYP;j++) if (hyptab[i][j].score > 0) {
			/* convert score to log scale */
			basescore = 5*(float)mylog(hyptab[i][j].score) + fxlikelihood(hyptab[i][j].fx);
			/* find best hyp on previous frame */
			maxrec = &hyptab[i-1][0];
			maxscore = basescore + fxsimilarity(hyptab[i][j].fx,hyptab[i-1][0].fx) + hyptab[i-1][0].score;
#ifdef EBUG
printf("hyp(%.1f<-%.1f) %g+%g+%g=>%g\n",hyptab[i][j].fx,hyptab[i-1][0].fx,basescore,fxsimilarity(hyptab[i][j].fx,hyptab[i-1][0].fx),hyptab[i-1][0].score,maxscore);
#endif
			for (k=1;k<NHYP;k++) {
				score = basescore + fxsimilarity(hyptab[i][j].fx,hyptab[i-1][k].fx) + hyptab[i-1][k].score;
#ifdef EBUG
if (k < 5) printf("hyp(%.1f<-%.1f) %g+%g+%g=>%g\n",hyptab[i][j].fx,hyptab[i-1][k].fx,basescore,fxsimilarity(hyptab[i][j].fx,hyptab[i-1][k].fx),hyptab[i-1][k].score,score);
#endif
				if (score > maxscore) {
					maxrec = &hyptab[i-1][k];
					maxscore = score;
				}
			}
			/* find best hyp on frame before */
			if (i > 1) {
				for (k=0;k<NHYP;k++) {
					score = basescore + fxsimilarity(hyptab[i][j].fx,hyptab[i-2][k].fx) + hyptab[i-2][k].score + skipcost();
					if (score > maxscore) {
						maxrec = &hyptab[i-2][k];
						maxscore = score;
					}
				}
			}
			addhyp(hold,hyptab[i][j].posn,hyptab[i][j].fx,maxscore,maxrec);
		}
		/* save hypotheses */
		for (j=0;j<NHYP;j++)
			hyptab[i][j] = hold[j];
#ifdef EBUG
		printf("L%.3f. ",i*STEP);
		for (j=0;j<NHYP;j++) {
			if (hold[j].score > -100)
				printf("%.1f(%g) ",hold[j].fx,hold[j].score);
		}
		printf("\n");
#endif
		if (i > 5) {
			if ((hyptab[i-2][1].score <= -1000) &&
			    (hyptab[i-1][1].score <= -1000) &&
			    (hyptab[i  ][1].score <= -1000))
				hyptab[i][0].score=0;
		}
	}

	free(hold);
}

/* back tracking */
void backtrack(short *ofx)
{
	int	pos=hyplen-1;
	struct hyp_rec *hyp=&hyptab[hyplen-1][0];

	while (hyp!=NULL) {
		for (;pos >= hyp->posn;pos--)
			ofx[pos] = (short)(hyp->fx);
		hyp = hyp->back;
	}
}

/* smoothing */
void smoothfx(short *ofx)
{
	int	i,j,k;
	short	hold[5];
	short	proc[5];

	/* do 5 point median filter */
	for (i=0;i<4;i++) hold[i]=ofx[i];
	for (i=5;i<hyplen;i++) {
		hold[4] = ofx[i];
		proc[0] = hold[0];
		for (j=1;j<5;j++) {
			k = j;
			while ((k>0)&&(proc[k-1] > hold[j])) {
				proc[k] = proc[k-1];
				k--;
			}
			proc[k] = hold[j];
		}
		ofx[i-3] = proc[2];
		hold[0] = hold[1];
		hold[1] = hold[2];
		hold[2] = hold[3];
		hold[3] = hold[4];
	}
#ifdef OLD
	for (i=1;i<hyplen-1;i++)
		if (((ofx[i]/(ofx[i-1]+1.0)) > 1.05) &&
		    ((ofx[i]/(ofx[i+1]+1.0)) > 1.05))
			ofx[i]=0;
	for (i=1;i<hyplen-2;i++)
		if ((ofx[i-1]==0)&&(ofx[i]!=0)&&((ofx[i+1]==0)||(ofx[i+2]==0)))
			ofx[i]=0;
#endif
}

/* get a voicing decision */
int isvoiced(short *nsp,float *fsp,int len,float *acoeff,int tlo,int thi)
{
	short	*s1,*s2;
	float	sum;
	int		num;
	float	a,z,e,v,r;
	float	ap,zp,ep,rp;
	float	min;
	int		i;

	/* get energy in unfiltered signal */
	sum=(float)0.0;
	num=len;
	s1 = nsp;
	for (i=0;i<num;i++,s1++) sum += (float)*s1 * (float)*s1;
	e = sum/len;

	/* get first reflection coefficient for normal signal */
	sum=(float)0.0;
	num = len-1;
	s1 = nsp;
	s2 = nsp+1;
	for (i=0;i<num;i++,s1++,s2++) sum += (float)*s1 * (float)*s2;
	r = (sum/len)/e;
	rp = (float)(1.0/(1+exp(-(r-0.6)/0.2)));

	/* get autocorrelation peak from filtered signal */
	a=0;
	for (i=tlo;i<=thi;i++) if (acoeff[i] > a) a=acoeff[i];
	ap = (float)(1.0/(1+exp(-(a-0.75)/0.1)));

	/* get zero crossing rate */
	z=zeroc(nsp,len);
	zp = (float)(1.0/(1+exp((z-1000)/200)));

	/* get energy */
	ep = (float)(1.0/(1+exp(-((10.0*log10(e))-maxenergy+30)/5)));

	/* combine scores */
	min=ap;
	if (zp < min) min=zp;
	if (ep < min) min=ep;
	if (rp < min) min=rp;

	v = ap * zp * ep * rp / min;
#if 0
printf("%.3f\t%.3f\t%.3f\t%.3f\t%.3f\n",ap,zp,ep,rp,v);
#endif

	return(v > 0.5);
}

/* main program */
void main(int argc,char **argv)
{
	/* local variables */
	extern int	optind;		/* option index */
	extern char	*optarg;	/* option argument */
	int		errflg=0;	/* option error flag */
	int		c;		/* option char */
	int		it;
	char		*ty;
	int		iptype=SP_TYPE;
	char		*sptype = "0";	/* input sp sub-type = last */
	char		filename[SFSMAXFILENAME];
					/* database file name */
	int		fid,ofid;	/* file descriptors */

	/* processing variables */
	int		stsize;		/* # samples in step */
	int		wisize;		/* # samples in window */
	int		i,f=0;		/* counters */
	int		nframes;	/* # output frames */
	int		j,k,npeak;
	int		lofx,hifx;
	float		sum,val,rave;
	double		sumsq,maxsumsq,rms;
	int		rmscnt;
	short		*ofx;
	FILTER		*bpfilt;

	/* decode switches */
	while ( (c = getopt(argc,argv,"Ii:lfhN:V:S:T:L:H:X")) != EOF ) switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Fx by autocorrelation and tracking V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'i' :	/* item spec */
			if (itspec(optarg,&it,&ty) == 0) {
				if (it == SP_TYPE) {
					iptype=SP_TYPE;
					sptype=ty;
				}
				else if (it == LX_TYPE) {
					iptype=LX_TYPE;
					sptype=ty;
				}
				else
					error("unsuitable item specification %s",optarg);
			}
			else
				error("illegal item specification %s",optarg);
			break;
		case 'l' :	/* from LX */
			iptype=LX_TYPE;
			break;
		case 'f' :	/* female switch */
			female++;
			break;
		case 'h' :	/* high-pitch switch */
			female++;
			break;
		case 'N' :
			NHYP = atoi(optarg);
			break;
		case 'V' :
			VUV_PENALTY = atof(optarg);
			break;
		case 'S' :
			NORMFX_STDDEV = atof(optarg);
			break;
		case 'T' :
			THRESHOLD = atof(optarg);
			break;
		case 'L' :
			LOFILT = atof(optarg);
			break;
		case 'H' :
			HIFILT = atof(optarg);
			break;
		case 'X' :	/* check for pitch halving */
			checkhalf=1;
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	if (errflg || (argc<2))
		error("usage: %s (-I) (-i item) (-l) (-h) (-X) file\n",PROGNAME);

	/* get filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open file */
	if ((fid=sfsopen(filename,"w",NULL)) < 0) {
		if (fid==-1)
			error("'%s': file not found",filename);
		else
			error("access error on '%s'",filename);
	}

	/* locate input speech item */
	if (!sfsitem(fid,iptype,sptype,&spitem))
		error("cannot find input item in '%s'",filename);

	/* get window parameters */
	sampfreq = (int)(0.5 + (0.01/spitem.frameduration));
	stsize = (int)(0.5 + (STEP/spitem.frameduration));
	if (female) {
		wisize = (int)(0.5 + (WINDOW_FEM/spitem.frameduration));
		hifx = (int)(1.0/(HIFX_FEM*spitem.frameduration));
		lofx = (int)(0.5+1.0/(LOFX_FEM*spitem.frameduration));
	}
	else {
		wisize = (int)(0.5 + (WINDOW/spitem.frameduration));
		hifx = (int)(1.0/(HIFX*spitem.frameduration));
		lofx = (int)(0.5+1.0/(LOFX*spitem.frameduration));
	}

	/* estimate # output frames for user info */
	nframes = 1+(spitem.numframes/stsize);

	/* get buffers */
	if ((sp=(short *)sfsbuffer(&spitem,wisize)) == NULL)
		error("cannot get buffer for speech",NULL);
	if ((fsp=(float *)calloc(wisize,sizeof(float)))==NULL)
		error("cannot get buffer for speech",NULL);
	if ((msp=(float *)calloc(wisize,sizeof(float)))==NULL)
		error("cannot get buffer for speech",NULL);
	if ((acoeff=(float *)calloc(wisize,sizeof(float)))==NULL)
		error("cannot get buffer for speech",NULL);
	if ((fxpeak=(float *)calloc(wisize,sizeof(float)))==NULL)
		error("cannot get buffer for speech",NULL);
	if ((fxsize=(float *)calloc(wisize,sizeof(float)))==NULL)
		error("cannot get buffer for speech",NULL);
	if ((hyptab=(struct hyp_rec **)calloc(nframes+1,sizeof(struct hyp_rec *)))==NULL)
		error("cannot get buffer for tracking",NULL);
	for (i=0;i<=nframes;i++)
		if ((hyptab[i]=(struct hyp_rec *)calloc(NHYP,sizeof(struct hyp_rec)))==NULL)
			error("cannot get buffer for tracking",NULL);

	/* design band-pass filter */
	bpfilt = filter_design(FILTER_BAND_PASS,4,LOFILT,HIFILT,1.0/spitem.frameduration);

	/* get energy thresholds */
	maxsumsq=0;
	rms=0; rmscnt=0;
	for (i=0;sfsread(fid,i,wisize,sp)==wisize;i+=wisize) {

		/* calculate mean */
		for (j=0,sum=0;j<wisize;j++) sum += (float)(sp[j]);
		sum /= wisize;

		/* remove mean and calculate energy */
		for (j=0,sumsq=0;j<wisize;j++) {
			val = (float)sp[j] - sum;
			sumsq += val * val;
		}
		if (sumsq > maxenergy) maxenergy=sumsq;

		/* filter */
		filter_signal(bpfilt,sp,sp,wisize);

		/* calculate mean */
		for (j=0,sum=0;j<wisize;j++) sum += (float)(sp[j]);
		sum /= wisize;

		/* remove mean and calculate energy */
		for (j=0,sumsq=0;j<wisize;j++) {
			val = (float)sp[j] - sum;
			sumsq += val * val;
		}
		rms += sumsq;
		rmscnt += wisize;

		if (sumsq > maxsumsq) maxsumsq=sumsq;
	}
	rms /= rmscnt;
/* printf("rms=%g maxsumsq=%.1f\n",rms,20*log10(maxsumsq)); */
	maxsumsq /= ENERGYDIV;	/* 50dB down from max */
	maxenergy = 10.0 * log10(maxenergy/wisize);

	/* perform processing */
	filter_clear(bpfilt);
	hyplen=0;
	for (i=0;sfsread(fid,i,wisize,sp)==wisize;i+=stsize,f++) {

		/* do filtering */
		if (i==0) {
			/* first time - filter whole window */
			for (j=0,sum=0;j<wisize;j++) {
				fsp[j] = filter_sample(bpfilt,(double)sp[j]);
				sum += fsp[j];
			}
		}
		else {
			/* take copy from previous window */
			for (j=stsize,sum=0;j<wisize;j++) {
				fsp[j-stsize] = fsp[j];
				sum += fsp[j-stsize];
			}
			/* and append filtered version of new part */
			for (j=wisize-stsize;j<wisize;j++) {
				fsp[j] = filter_sample(bpfilt,(double)sp[j]);
				sum += fsp[j];
			}
		}

		/* remove mean and calculate energy */
		sum /= wisize;
		for (j=0,sumsq=0;j<wisize;j++) {
			msp[j] = fsp[j] - sum;
			sumsq += msp[j]*msp[j];
		}

#if 0
		/* check above threshold */
#ifdef EBUG
printf("E%d=%g, %g, %g\n",hyplen,maxsumsq,sum,sumsq);
#endif
		if (sumsq > maxsumsq) {
#endif

		/* process window */
		rave = autoc(msp,wisize,acoeff,hifx,lofx);

		/* get voicing decision */
		if (isvoiced(sp,msp,wisize,acoeff,hifx,lofx)) {

			/* process window */
#ifdef EBUG
printf("t=%.3f\n",i*spitem.frameduration);
#endif
			npeak = findpeaks(acoeff,wisize,hifx,lofx,fxpeak,fxsize);

			if (npeak==0) {
				hyptab[hyplen][0].posn = hyplen;
				hyptab[hyplen][0].fx = 0;
				hyptab[hyplen][0].score = 1;
				hyplen++;
			}
			else {
				/* sort peaks by size */
				sortpeaks(fxpeak,fxsize,npeak);

				/* add no peak option */
				hyptab[hyplen][0].posn = hyplen;
				hyptab[hyplen][0].fx = 0;
				hyptab[hyplen][0].score = (float)0.3;

				/* add peaks */
				for (j=1,k=0;(k<npeak)&&(j<NHYP);j++,k++) {
					hyptab[hyplen][j].posn = hyplen;
					hyptab[hyplen][j].fx = fxpeak[k];
					hyptab[hyplen][j].score = fxsize[k];
				}
				hyplen++;
			}

		}
		else {
			hyptab[hyplen][0].posn = hyplen;
			hyptab[hyplen][0].fx = 0;
			hyptab[hyplen][0].score = 1;
			hyplen++;
		}

		/* report progress */
		if (((f%100)==99) && ttytest()) {
			printf("\rFrame %d/%d",f+1,nframes);
			fflush(stdout);
		}

	}
	if (ttytest()) printf("\r                       \r");

	/* do dynamic programming */
	dptrack();

	/* generate output item header */
	sfsheader(&fxitem,FX_TYPE,0,2,1,stsize*spitem.frameduration,spitem.offset+WINDOW/2,1,0,0);
	if (female||checkhalf)
		sprintf(fxitem.history,"%s(%d.%02d;%s%s%s)",
			PROGNAME,spitem.datatype,spitem.subtype,
			(female)?"highpitch":"",
			(female&&checkhalf)?",":"",
			(checkhalf)?"checkhalving":"");
	else
		sprintf(fxitem.history,"%s(%d.%02d)",
			PROGNAME,spitem.datatype,spitem.subtype);

	/* open output channel */
	if ((ofid=sfschannel(filename,&fxitem)) < 0)
		error("cannot open temporary file",NULL);

	/* create output buffer */
	if ((ofx=(short *)calloc(hyplen,sizeof(short)))==NULL)
		error("cannot get buffer for speech",NULL);

	/* save estimates */
	backtrack(ofx);

	/* smooth contour */
	smoothfx(ofx);
	sfswrite(ofid,hyplen,ofx);

	/* and update file */
	if (!sfsupdate(filename))
		error("backup error on '%s'",filename);

	/* ... that's all folks */
	exit(0);
}
