/* txload -- load a Tx item from an ASCII list of numbers */

/* M.A.Huckvale - July 2006 */

/* version 1.0 */
/* version 2.0 - December 2007 */

/*-------------------------------------------------------------------*/
/**MAN
.TH TXLOAD SFS1 UCL
.SH NAME
txload -- load a TX item from a text file of numbers
.SH SYNOPSIS
.B txload
(-f freq) (-h history) (-O timeoffset) textfile file
.SH DESCRIPTION
.I txload
constructs a new TX item in the given file from text numbers
a text file.  The numbers can be separated by any white-space.
The numbers should represent pitch period intervals or pitch period epoch locations
in either samples or seconds. A simple heuristic is used to decide the file format.
In addition Praat point-process text files are recognised and the header removed.
.PP
.B Options
.TP 11
.B -I
Identify the program version.
.TP 11
.BI -f freq
Use a sampling interval based on 1/freq seconds for calculating the duration of each
Tx value. Default 10000.
.TP 11
.BI -h history
Place a "history=" indicator in the created item header, with the argument
.I history.
.TP 11
.BI -O timeoffset
Set "offset" field in output track item to given value.
.SH OUTPUT ITEMS
.IP TX 11
Loaded TX item
.SH VERSION/AUTHOR
1.0 - Mark Huckvale
*/
/*---------------------------------------------------------------*/

/* program name and version */
#define	PROGNAME "txload"
#define PROGVERS "2.0"
char	*progname = PROGNAME;

/* global declarations */
#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "sfs.h"		/* database filing system structures */

/* global data */
struct item_header 	txitem;			/* TX item header  */
char			textname[SFSMAXFILENAME];	/* text file name */
char			filename[SFSMAXFILENAME];	/* dbase file name */
double			fdur=1.0E-4;		/* default sample duration */
char			*txhist;		/* tx history */
double			offset=0;

/* main program */
void main(argc,argv)
int	argc;
char	*argv[];
{
	/* option decoding */
	extern int	optind;			/* option index */
	extern char	*optarg;		/* option argument ptr */
	int		errflg = 0;		/* option error flag */
	int		c;			/* option switch */

	/* processing variables */
	FILE		*ip;			/* input text file */
	int		ofid;			/* output file descriptors */
	double		val;			/* input value */
	int			itx;
	double		srate,atof();
	int			issampled=0;
	int			isperiods=0;
	int			isPraat=0;
	double		lval=0;
	double		mindiff=1E6;
	char		line[1024];

	/* decode switches */
	while ( (c = getopt(argc,argv,"If:h:O:")) != EOF )
		switch (c) {
		case 'I' :	/* Identify */
			fprintf(stderr,"%s: Load track data from text V%s\n",PROGNAME,PROGVERS);
			exit(0);
			break;
		case 'f' :	/* select sampling frequency */
			srate = atof(optarg);
			if (srate<=0)
				error("bad sampling frequency: %s",optarg);
			else
				fdur = 1.0/srate;
			break;
		case 'h' :	/* history */
			txhist=optarg;
			break;
		case 'O' :	/* offset */
			offset=atof(optarg);
			break;
		case '?' :	/* unknown */
			errflg++;
	}
	/* check command line */
	if (errflg || (argc<3))
		error("usage: %s (-I) (-f freq) (-h txhist) (-O timoffset) textfile file",PROGNAME);

	/* get text file name */
	if (optind < argc)
		strcpy(textname,argv[optind]);
	else
		error("no text file specified",NULL);
	optind++;

	/* get data filename */
	if (optind < argc)
		strcpy(filename,sfsfile(argv[optind]));
	else
		error("no data file specified",NULL);

	/* open text file */
	if ((ip=fopen(textname,"r"))==NULL)
		error("cannot open %s",textname);

	/* open data file */
	if ((ofid=sfsopen(filename,"w",NULL)) < 0) {
		if (ofid==-1)
			error("cannot find file %s",filename);
		else
			error("access error on %s",filename);
	}
	sfsclose(ofid);

	/* create output item header */
	sfsheader(&txitem,TX_TYPE,0,4,1,fdur,offset,1,0,0);
	if (txhist)
		sprintf(txitem.history,"%s(file=%s,rate=%g,offset=%g,history=%s)",PROGNAME,textname,1.0/fdur,offset,txhist);
	else
		sprintf(txitem.history,"%s(file=%s,rate=%g,offset=%g)",PROGNAME,textname,1.0/fdur,offset);

	/* open output channel */
	if ((ofid=sfschannel(filename,&txitem)) < 0)
		error("unable to open temporary file",NULL);

	/* check for Praat format file */
	if (fgets(line,sizeof(line),ip)) {
		if (strstr(line,"File type") && strstr(line,"ooTextFile")) {
			/* is Praat file - skip 5 lines */
			fgets(line,sizeof(line),ip);
			fgets(line,sizeof(line),ip);
			fgets(line,sizeof(line),ip);
			fgets(line,sizeof(line),ip);
			fgets(line,sizeof(line),ip);
			isPraat=1;
		}
		else
			rewind(ip);
	}
	/* process input file to determine format */
	while (fscanf(ip,"%lf",&val) == 1) {
		if (val < lval)
			isperiods=1;
		else if ((val-lval) < mindiff)
			mindiff = val-lval;
		lval=val;
	}
	if (mindiff >= 1) issampled=1;

	fprintf(stderr,"file=%s sampled=%d periods=%d Praat=%d\n",textname,issampled,isperiods,isPraat);

	/* process input file to determine format */
	rewind(ip);
	if (isPraat) {
		/* skip Praat header */
		fgets(line,sizeof(line),ip);
		fgets(line,sizeof(line),ip);
		fgets(line,sizeof(line),ip);
		fgets(line,sizeof(line),ip);
		fgets(line,sizeof(line),ip);
		fgets(line,sizeof(line),ip);
	}
	lval=0;
	while (fscanf(ip,"%lf",&val) == 1) {
		if (issampled) {
			if (isperiods)
				itx=(int)val;
			else
				itx=(int)(val-lval);
		}
		else {
			if (isperiods)
				itx=(int)(0.5+val/fdur);
			else
				itx=(int)(0.5+(val-lval)/fdur);
		}
		if (sfswrite(ofid,1,&itx) != 1)
			error("write error on output file",NULL);
		lval=val;
	}

	/* update data file */
	if (!sfsupdate(filename))
		error("backup error on %s",filename);

	/* that's all folks */
	exit(0);

}
