/* tree -- DIG graphic representation of file structure */

/* m.a. huckvale - feb 1988 */

/* version 1 */

/*--------------------------------------------------------------------------*/
/**MAN
.TH TREE SFS1 UCL SFS
.SH NAME
tree - display file structure tree as menu
.SH SYNOPSIS
.B tree
[tree options] (program [program options]) file
.SH DESCRIPTION
.I tree
is a program to display the tree structure of the items in a speech data 
file.  Items selected from the tree may then be passed as item selections
to a given program.
.PP
.I Options
and their meanings are:
.TP 11
.TP 11
.B -I
Identify program and exit.
.B -d
Tree display only.
.TP 11
.B -p
Tree display only.  Send output direct to laser printer
.TP 11
.B -r
Repetitive execution of program and tree display.
.TP 11
.B -f
Fast repetitive execution.  As "-r" but with single box selection (use "Go" button) and
no redrawing of tree.  Chosen program should not use graphics.
.SH MOUSE BUTTONS
.SS LEFT
Select. Select/Deselect items. In top region of display requests re-draw of screen.
.SS MIDDLE
Go.  Execute program with selections.
.SS RIGHT
Quit. Exit tree program.
.SH GRAPHICS
Each item is described using the item number and the first 8 characters of
the program name in the item history.  Deleted items have a cross-bar.  Selected
items are displayed in inverse video.
.SH VERSION/AUTHOR
1.0 - Mark Huckvale
*/
/*--------------------------------------------------------------------------*/

#define PROGNAME "tree"
#define PROGVERS "1.0s"
char *progname=PROGNAME;

/* global declarations */
#include <stdio.h>
#include <ctype.h>
#include <fcntl.h>
#include "stdlib.h"
#include "io.h"
#include "sfs.h"
#include "sfstree.h"
#include "dig.h"
#include "digdata.h"

/* mouse commands */
#define COM_NULL	0
#define COM_QUIT	1
#define	COM_GO		2
#define	COM_BOX		3
#define COM_FAST	4
#define COM_REDRAW	5

/* display structure */
#define ITXMAX	10
#define ITYMAX	10
struct {
	int	entry;
	float	x;
	float	y;
	int	select;
} dtab[ITXMAX][ITYMAX];
int	dcnt[ITXMAX];
int	xent[MAXTREELEN];
int	yent[MAXTREELEN];
int	dlevel=0;
float	bheight,bwidth,cheight,cwidth;	/* graphics constants */
int	plen;
float	xpos=0.5,ypos=0.5;			/* stored mouse x,y */

/* global data */
int	interact=1;			/* interactive */
int	execmode=0;			/* execution control */
int	laser=0;			/* laser output */
extern char	*sfslabel[];
struct main_header head;
char	**gargv;
int	gargc;
char	*com[100];

/* main program */
main(argc,argv)
int argc;
char *argv[];
{
        /* option decoding variables */
        extern int      optind;
        extern char     *optarg;
        int             errflg=0;
        char            c;
	/* file variables */
        int             fid;
        char            filename[SFSMAXFILENAME];
	int		com,row,col;

        /* decode switches */
        while ( (c = getopt(argc,argv,"Iprdf")) != EOF) switch (c) {
                case 'I' :      /* Identify */
                        fprintf(stderr,"%s: file structure tree V%s\n",argv[0],PROGVERS);
                        exit(0);
                        break;
		case 'd' :	/* display only */
			interact=0;
			break;
		case 'r' :	/* repetitive */
			execmode=1;
			break;
		case 'f' :	/* fast repetitive */
			execmode=2;
			break;
		case 'p' :	/* laser print */
			laser++;
			interact=0;
			break;
                case '?' :      /* unknown */
                        errflg++;
        }
        if (errflg || (argc<2)) 
                error("usage: %s (-I) (-d|-r|-f|-p) (program program_options) file",argv[0]);

	/* save program if specified */
	if (optind < (argc-1)) {
		gargv = argv+optind;
		gargc = argc-optind;
	}
	else
		execmode = -1;

        /* get filename */
        if (optind < argc)
                strcpy(filename,sfsfile(argv[argc-1]));
        else
                error("no database file specified",NULL);

        /* open datafile */
        if ((fid = sfsopen(filename,"r",&head)) < 0) {
                if (fid == -1)
                        error("cannot open %s",filename);
                else
                        error("access error on %s",filename);
        }
	sfsclose(fid);

        /* get tree */
        if (!sfstree(filename,0))
		error("cannot open %s",filename);

	/* init graphics */
	if (laser)
		digstart('k',NULL,1);
	else
		digstart('\0',NULL,1);

	/* set up scaling */
	digscale(1.0,1.0+3.0*digdata.chheight/digdata.yscale,0);
	digorigin(0.0,1.5*digdata.chheight/digdata.yscale);

	/* display tree */
	display(filename);

	/* interactive mode */
	if (interact && ttytest()) {
		drawprompt("Select box with mouse");
		diginitmouse(xpos,ypos);
		while ((com=getmousecom(&col,&row)) != COM_QUIT) switch (com) {
		case COM_BOX:	if ((execmode == 0) || (execmode==1))
					selectbox(col,row);
				drawprompt(tree[dtab[col][row].entry].history);
				break;
		case COM_GO:	buttonup();
				digkillmouse();
				if (!buildcom()) {
					diginitmouse(xpos,ypos);
					break;
				}
				digquit(0);
				if (execmode==0)
					execonce();
				else {
					execrepeat();
					digstart('\0',NULL,1);
					digscale(1.0,1.0+3.0*digdata.chheight/digdata.yscale,0);
					digorigin(0.0,1.5*digdata.chheight/digdata.yscale);
					display(filename);
					drawprompt("Select box with mouse");
					diginitmouse(xpos,ypos);
				}
				break;
		case COM_FAST:	selectbox(col,row);
				if (dtab[col][row].select) {
					buttonup();
					digkillmouse();
					if (!buildcom()) {
						diginitmouse(xpos,ypos);
						break;
					}
					execrepeat();
					digprompt("");
					selectbox(col,row);
					diginitmouse(xpos,ypos);
				}
				break;
		case COM_REDRAW:
				buttonup();
				digkillmouse();
				display(filename);
				diginitmouse(xpos,ypos);
				break;
		case COM_NULL:
				putchar(7);
				fflush(stdout);
				break;
		}
		digprompt("");
		buttonup();
		digkillmouse();
	}

	/* that's all folks */
	digquit(14);
        exit(0);
}

/* display item relation tree */
display(filename)
char	*filename;
{
	int	i,j,k,nassign=0;
	int	entry,lentry;
	float	len,digtextlen(),gap;
	char	title[256];

	/* assign items to level in tree */
	dlevel=0;
	for (i=0;(dlevel<ITXMAX) && (nassign<treelen);i++) {
		dcnt[dlevel]=0;
		for (j=0;j<treelen;j++) {
			if (tree[j].level==i) {
				dtab[dlevel][dcnt[dlevel]].entry=j;
				dtab[dlevel][dcnt[dlevel]].select=0;
				xent[j]=dlevel;
				yent[j]=dcnt[dlevel];
				dcnt[dlevel]++;
				nassign++;
				if (dcnt[dlevel] == ITYMAX) {
					dlevel++;
					dcnt[dlevel]=0;
				}
			}
		}
		dlevel++;
	}

	/* allow space for title */
	cheight = (float)digdata.chheight/digdata.yscale;
	cwidth  = (float)digdata.chwidth/digdata.xscale;
	bheight = 1.25 * cheight;
	bwidth  = 4.5 * cwidth;
	gap = (1.0-2*dlevel*bwidth)/(dlevel+1);
	if (gap < 0) gap=0;
	plen = (1.0/cwidth);	/* max char on line */

	/* plot title */
	digclearscreen();
	sprintf(title,"file=%s speaker=%s token=%s",filename,head.speaker,head.token);
	title[plen]='\0';
	digtext(24,0.0,-1.5*cheight,title);
	digbox(20,0.0,0.0,1.0,1.0);
	digclip(0.0,0.0,1.0,1.0);

	/* calculate x,y co-ordinates */
	for (i=0;i<dlevel;i++) {
		for (j=0;j<dcnt[i];j++) {
			dtab[i][j].x = (gap+bwidth) + i*(gap + 2*bwidth);
			dtab[i][j].y = 1.0 - ((float)(j+1))/(dcnt[i]+1);
		}
	}

	/* plot lines */
	for (i=0;i<dlevel;i++) {
		for (j=0;j<dcnt[i];j++) {
			entry = dtab[i][j].entry;
			for (k=0;k<tree[entry].numlink;k++) {
				lentry = tree[entry].link[k];
				digline(27,dtab[xent[lentry]][yent[lentry]].x+bwidth,
					   dtab[xent[lentry]][yent[lentry]].y,
					   dtab[i][j].x-bwidth,dtab[i][j].y);
			}
		}
	}

	/* plot itemnos */
	for (i=0;i<dlevel;i++) {
		for (j=0;j<dcnt[i];j++) {
			entry = dtab[i][j].entry;
			drawbox(dtab[i][j].x,dtab[i][j].y,
				tree[entry].datatype,
				tree[entry].subtype,
				tree[entry].history);
		}
	}

	digflush();
}

/* draw item box */
drawbox(x,y,it,ty,hist)
float	x,y;
int	it,ty;
char	*hist;
{
	char	iname[8];
	char	ilabel[32];
	int	llen;
	float	tlen;
	int	i;

	/* get text for box */
	sprintf(iname,"%s.%02d",sfslabel[abs(it)],ty);
	llen = strcspn(hist," (/");
	for (i=0;(i<llen) && (i<8);i++) ilabel[i]=hist[i];
	ilabel[i]='\0';
	tlen = digtextlen(ilabel);

	/* clear box */
	digflush();
	digfillbox(0,x-bwidth,y-bheight,x+bwidth,y+bheight);

	/* write outline if undeleted item */
	if (it > 0) {
		digbox(31,x-bwidth,y-bheight,x+bwidth,y+bheight);
	}
	else {
		digbox(29,x-bwidth,y-bheight,x+bwidth,y+bheight);
		digline(29,x-bwidth,y+bheight,x+bwidth,y-bheight);
	}

	/* write item number */
	digtext(24,x-2.0*cwidth,y,iname);

	/* write item label */
	digtext(24,x-tlen/2,y-1.125*cheight,ilabel);

}

/* select/deselect box */
selectbox(col,row)
int	col,row;
{
	if (tree[dtab[col][row].entry].datatype > 0) {
		dtab[col][row].select = !dtab[col][row].select;
		diglinemode(1);
		digfillbox(24,dtab[col][row].x-bwidth,
				dtab[col][row].y-bheight,
				dtab[col][row].x+bwidth,
				dtab[col][row].y+bheight);
		diglinemode(0);
		digflush();
	}
}

/* display prompt */
drawprompt(prompt)
char	*prompt;
{
	int	i;

	digprompt("");
	for (i=0;(prompt[i]) && (i<plen);i++) putchar(prompt[i]);
	fflush(stdout);
}

getmousecom(col,row)
int	*row,*col;
{
	char	button,string[80];
	int	i,j;

	/* get mouse command */
	do {
		diggetmouse(string,&button,&xpos,&ypos);
	} while (!button);

	/* see if mouse in box */
	*col = -1;
	*row = -1;
	for (i=0;i<dlevel;i++) {
		for (j=0;j<dcnt[i];j++) {
			if ((xpos > dtab[i][j].x-bwidth) &&
			    (xpos < dtab[i][j].x+bwidth) &&
			    (ypos > dtab[i][j].y-bheight) &&
			    (ypos < dtab[i][j].y+bheight)) {
				*row=j;
				*col=i;
			}
		}
	}

	/* decode according to button */
	switch (button) {
	case 4:		/* select */
		if (ypos > 1.0) 
			return(COM_REDRAW);
		else if (ypos > 0.0) {
			if (*col >= 0) 
				return(COM_BOX);
			else
				return(COM_NULL);
		}
		else
			return(COM_NULL);
		break;
	case 2:		/* go */
		if (execmode==2) {
			if (*col >= 0)
				return(COM_FAST);
			else
				return(COM_NULL);
		}
		else if (execmode >= 0)
			return(COM_GO);
		else
			return(COM_QUIT);
		break;
	case 1:		/* quit */
		return(COM_QUIT);
		break;
	}
	return(COM_NULL);
}

/* build command line */
buildcom()
{
	int	i,j,ccnt=0,entry;
	int	flag=0,len=0;
	char	*sel;

	/* copy in program switches */
	for (i=0;i<(gargc-1);i++) com[ccnt++]=gargv[i];

	/* put in selections */
	for (i=0;i<dlevel;i++) {
		for (j=0;j<dcnt[i];j++) if (dtab[i][j].select) {
			entry = dtab[i][j].entry;
			sel = (char *)malloc(8);
			sprintf(sel,"-i%d.%02d",tree[entry].datatype,tree[entry].subtype);
			com[ccnt++]=sel;
			flag++;
		}
	}

	/* check something selected */
	if (!flag) {
		putchar(7);
		digprompt("No selections");
		return(0);
	}

	/* put in filename */
	com[ccnt++]=gargv[gargc-1];

	/* terminate arguments */
	com[ccnt]=NULL;

	/* display result */
	digprompt("");
	for (i=0;(i<ccnt) && (len < plen);i++) {
		len += strlen(com[i])+1;
		if (len < plen) printf("%s ",com[i]);
	}
	printf("\n");

	return(1);
}

/* execute once */
execonce()
{
	int	i;

	/* close everything */
	for (i=3;i<39;i++) close(i);	/* only three channels open */

	/* execute program */
	if (execvp(com[0],com))
		error("unable to execute '%s'",com[0]);
}

/* execute repetitively */
execrepeat()
{
	int	i,status;
	short	pgrp;

	/* close everything */
/*	for (i=3;i<39;i++) close(i);	/* only three channels open */

	/* fork to process */
	i = fork();
	if (i == 0) {

#ifdef JHGJHGJHGJH
		/* "magic" code supplied by Masscomp for correct chaining into windows */
		syscall(39,1);
		close(open("/dev/tty",0));	/* ensures keyboard signals */
		pgrp = getpgrp();
		ioctl(0,TCSPGRP,&pgrp);		/* set program group, i.e. operate set UID bit */
#endif

		/* execute program */
		if (execvp(com[0],com)) 
			error("unable to execute '%s'",com[0]);

	}
	else if ( i < 0) error("fork error",NULL);
	else {
		wait(&status);
		if (status) 
			error("'%s' terminated with error",com[0]);
	}

}

/* wait for mouse button up */
buttonup()
{
	char	ch,button;
	float	mx,my;

	do {
		ch=diggetmousech(&button,&mx,&my);
	} while (ch || button);
}
