/* DSP routines from Mike Brookes, Imperial College London */

#include "SFSCONFG.h"
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include "libic.h"

#define NAME "dftci"

void dftci(data,spect,len)
COMPLEX *data,*spect;
int len;
/*
	Complex inverse fourier transform with arbitrary length

	data[len] <- spect[len]

        data and spect can be the same array if desired

This routine uses the Chirp-Z algorithm. If W = exp(j2PI/2N), we need
to calculate X(k) = 1/N * SUM{x(n) * W**(2kn)} =
      1/N * W**kk * SUM{x(n)*W**nn * W**-(k-n)(k-n)} since
2kn = -(k-n)(k-n) + kk + nn.

Thus the fourier transform consists of multiplying by W**nn, convolving
with W**-nn and multiplying by W**kk. An ordinary fft is used to perform
the convolution; the data must be zero extended to avoid wraparound.
The array W**nn and its transform are saved in case the next transform
is of the same length.
*/
{
    COMPLEX *tdata,*chirp,*Tchirp,*tend;
    COMPLEX *pchirp,*pdata,*ptdata,*qtdata,*pTchirp;
    static COMPLEX czero = {0.0,0.0};
    REAL datar,datai,chirpr,chirpi;
    register REAL recipn;
    int tlen,halftlen;

#ifdef IAG
    int i;		/* for diagnostics only */
#endif

/* Create the chirp function  */

    makechirp(len,&halftlen,&chirp);
    tlen = halftlen + halftlen;
    Tchirp = chirp + len;
    tdata = Tchirp + halftlen + 1;
    tend = tdata + tlen;

#ifdef IAG
printf("Chirp Function\n\n");
for (i=0; i<len; i++) printf("%8d %10g %10g\n",i,chirp[i].r,chirp[i].i);

printf("Transform\n\n");
for (i=0; i<(halftlen+1); i++) 
    printf("%8d %10g %10g\n",i,Tchirp[i].r,Tchirp[i].i);
#endif


/* Multiply Original data by Chirp function */

#ifdef IAG
printf("Original spectrum\n\n");
for (i=0; i<len; i++) 
    printf("%8d %10g %10g\n",i,spect[i].r,spect[i].i);
#endif

    pchirp = chirp;
    pdata = spect;
    ptdata = tdata;
    while (pchirp < Tchirp) {
	datar = pdata->r;
	datai = (pdata++)->i;
	chirpr = pchirp->r;
	chirpi = (pchirp++)->i;
	ptdata->r = chirpr*datar + chirpi*datai;
	(ptdata++)->i = chirpr*datai - chirpi*datar;
	}
    while (ptdata < tend) *ptdata++ = czero;

#ifdef IAG
printf("Multiplied data\n\n");
for (i=0; i<tlen; i++) 
    printf("%8d %10g %10g\n",i,tdata[i].r,tdata[i].i);
#endif

    fftc(tdata,tdata,tlen);

#ifdef IAG
printf("Transformed data\n\n");
for (i=0; i<tlen; i++) 
    printf("%8d %10g %10g\n",i,tdata[i].r,tdata[i].i);
#endif

/* Now do the convolution (multiply by conjugate of Tchirp array) */

    ptdata = tdata;
    qtdata = tend;
    pTchirp = Tchirp;
    chirpr = pTchirp->r; chirpi = (pTchirp++)->i;	/* DC Term */
    datar = ptdata->r; datai = ptdata->i;
    ptdata->r = chirpr*datar - chirpi*datai;
    (ptdata++)->i = chirpr*datai + chirpi*datar;
    while (ptdata < --qtdata) {
	chirpr = pTchirp->r; chirpi = (pTchirp++)->i;
	datar = ptdata->r; datai = ptdata->i;
	ptdata->r = chirpr*datar - chirpi*datai;
	(ptdata++)->i = chirpr*datai + chirpi*datar;

	datar = qtdata->r; datai = qtdata->i;
	qtdata->r = chirpr*datar - chirpi*datai;
	qtdata->i = chirpr*datai + chirpi*datar;
	}
    chirpr = pTchirp->r; chirpi = pTchirp->i;	/* Nyquist Term */
    datar = ptdata->r; datai = ptdata->i;
    ptdata->r = chirpr*datar - chirpi*datai;
    ptdata->i = chirpr*datai + chirpi*datar;

#ifdef IAG
printf("Times Tchirp\n\n");
for (i=0; i<tlen; i++) 
    printf("%8d %10g %10g\n",i,tdata[i].r,tdata[i].i);
#endif

    fftci(tdata,tdata,tlen);

#ifdef IAG
printf("Inverse Transform\n\n");
for (i=0; i<tlen; i++) 
    printf("%8d %10g %10g\n",i,tdata[i].r,tdata[i].i);
#endif

/* Now copy back into the destination array */

    ptdata = tdata;
    pdata = data;
    pchirp = chirp;
    recipn = 1.0/len;
    while (pchirp < Tchirp) {
	datar = ptdata->r;
	datai = (ptdata++)->i;
	chirpr = pchirp->r;
	chirpi = (pchirp++)->i;
	pdata->r = (chirpr*datar + chirpi*datai)*recipn;
	(pdata++)->i = (chirpr*datai - chirpi*datar)*recipn;
	}

#ifdef IAG
printf("Output data\n\n");
for (i=0; i<len; i++) 
    printf("%8d %10g %10g\n",i,spect[i].r,spect[i].i);
#endif

    return;
    }
#ifdef EMO

#include <stdio.h>
#include <math.h>
#define LEN 32

main ()
{
    COMPLEX error,data[LEN],data2[LEN],spect[LEN];
    REAL sum,theta;
    int i,len;

    for(;;) {
	do {
	    printf("Enter length : ");
	    scanf("%d",&len);
	    if (len <= 0) exit(0);
	    } while ((len <2) || (len > LEN));
	printf("Test for dftc and dftci: (1.0,1.1) + 2 [f] + 3j [3f]\n");
	for (i=0; i<len; i++) {
	    theta = (TWOPI * i) / len;
	    data[i].r = 1.0 + 2.0*cos(theta) - 3.0*sin(3.0*theta);
	    data[i].i = 1.1 + 2.0*sin(theta) + 3.0*cos(3.0*theta);
	    }

	dftc(data,spect,len);

	for (i=0; i<len; i++) {
	    if ((spect[i].r*spect[i].r + spect[i].i*spect[i].i) > 0.01*len*len) 
		printf("%3d %8.2f %8.2f\n",i,spect[i].r/len,spect[i].i/len);
	    }

	dftci(data2,spect,len);

#ifdef IAG
printf(" Double Transformed      Original       Error\n\n");
#endif
	sum = 0.0;
	for (i=0; i<len; i++) {

	    error.r = data2[i].r - data[i].r;
	    error.i = data2[i].i - data[i].i;
	    sum = sum + error.r * error.r + error.i * error.i;
#ifdef IAG
printf("%4d (%9g %9g)-(%9g %9g)=(%9g %9g)\n",
    i,data2[i].r,data2[i].i,data[i].r,data[i].i,error.r,error.i);
#endif
	    }
	printf("\nTotal square error after forward + inverse = %8g\n",sum);
	fflush(stdout);
	}
    }
#endif
